//! Encryption and siging using PKCS#1.

use nettle_sys::{
    __gmpz_clear, __gmpz_init, nettle_rsa_decrypt_tr,
    nettle_rsa_sec_decrypt, nettle_rsa_encrypt,
    nettle_rsa_pkcs1_sign_tr, nettle_rsa_pkcs1_verify,
};
use std::mem::zeroed;

use crate::hash::insecure_do_not_use::{Md2, Md5, Sha1, Ripemd160};
use crate::hash::{Sha224, Sha256, Sha384, Sha512};
use crate::rsa::{PrivateKey, PublicKey};
use crate::{helper, Error, hash::Hash, random::Random, Result};

/// Marker trait for hash algorithms usable for PKCS#1 signatures.
pub trait Pkcs1Hash: Hash {
    /// Internal to `sign_pkcs1` and `verify_pkcs1`.
    fn oid(&self) -> &'static [u8];
}

/// ASN.1 OID for MD2
pub const ASN1_OID_MD2: &[u8] = &[
    0x30, 0x20, 0x30, 0x0c, 0x06, 0x08, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d,
    0x02, 0x02, 0x05, 0x00, 0x04, 0x10,
];
/// ASN.1 OID for MD5
pub const ASN1_OID_MD5: &[u8] = &[
    0x30, 0x20, 0x30, 0x0c, 0x06, 0x08, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d,
    0x02, 0x05, 0x05, 0x00, 0x04, 0x10,
];
/// ASN.1 OID for RipeMD160
pub const ASN1_OID_RIPEMD160: &[u8] = &[
    0x30, 0x21, 0x30, 0x09, 0x06, 0x05, 0x2B, 0x24, 0x03, 0x02, 0x01, 0x05,
    0x00, 0x04, 0x14,
];
/// ASN.1 OID for SHA1
pub const ASN1_OID_SHA1: &[u8] = &[
    0x30, 33, 0x30, 9, 0x06, 5, 0x2b, 0x0e, 0x03, 0x02, 0x1a, 0x05, 0, 0x04,
    20,
];
/// ASN.1 OID for SHA224
pub const ASN1_OID_SHA224: &[u8] = &[
    0x30, 0x2D, 0x30, 0x0D, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03,
    0x04, 0x02, 0x04, 0x05, 0x00, 0x04, 0x1C,
];
/// ASN.1 OID for SHA256
pub const ASN1_OID_SHA256: &[u8] = &[
    0x30, 49, 0x30, 13, 0x06, 9, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04,
    0x02, 0x01, 0x05, 0, 0x04, 32,
];
/// ASN.1 OID for SHA384
pub const ASN1_OID_SHA384: &[u8] = &[
    0x30, 0x41, 0x30, 0x0D, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03,
    0x04, 0x02, 0x02, 0x05, 0x00, 0x04, 0x30,
];
/// ASN.1 OID for SHA512
pub const ASN1_OID_SHA512: &[u8] = &[
    0x30, 81, 0x30, 13, 0x06, 9, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04,
    0x02, 0x03, 0x05, 0, 0x04, 64,
];

impl Pkcs1Hash for Md2 {
    fn oid(&self) -> &'static [u8] {
        ASN1_OID_MD2
    }
}

impl Pkcs1Hash for Md5 {
    fn oid(&self) -> &'static [u8] {
        ASN1_OID_MD5
    }
}

impl Pkcs1Hash for Sha1 {
    fn oid(&self) -> &'static [u8] {
        ASN1_OID_SHA1
    }
}

impl Pkcs1Hash for Ripemd160 {
    fn oid(&self) -> &'static [u8] {
        ASN1_OID_RIPEMD160
    }
}

impl Pkcs1Hash for Sha224 {
    fn oid(&self) -> &'static [u8] {
        ASN1_OID_SHA224
    }
}

impl Pkcs1Hash for Sha256 {
    fn oid(&self) -> &'static [u8] {
        ASN1_OID_SHA256
    }
}

impl Pkcs1Hash for Sha384 {
    fn oid(&self) -> &'static [u8] {
        ASN1_OID_SHA384
    }
}

impl Pkcs1Hash for Sha512 {
    fn oid(&self) -> &'static [u8] {
        ASN1_OID_SHA512
    }
}

/// Creates a PKCS#1.5 padded RSA signature for `digest_info || digest`.
pub fn sign_digest_pkcs1<R: Random>(
    public: &PublicKey,
    private: &PrivateKey,
    digest: &[u8],
    digest_info: &[u8],
    random: &mut R,
    signature: &mut [u8],
) -> Result<()> {
    unsafe {
        let mut sig = zeroed();
        let mut msg = vec![0u8; digest.len() + digest_info.len()];

        msg[0..digest_info.len()].clone_from_slice(digest_info);
        msg[digest_info.len()..].clone_from_slice(digest);

        __gmpz_init(&mut sig);

        if nettle_rsa_pkcs1_sign_tr(
            &public.context,
            &private.context,
            random.context(),
            Some(R::random_impl),
            msg.len(),
            msg.as_ptr(),
            &mut sig,
        ) == 1
        {
            helper::write_gmpz_into_slice(sig, signature, "signature")
        } else {
            __gmpz_clear(&mut sig);

            Err(Error::SigningFailed)
        }
    }
}

/// Verifies a PKCS#1.5 padded RSA signature for `digest_info || digest`.
pub fn verify_digest_pkcs1(
    public: &PublicKey,
    digest: &[u8],
    digest_info: &[u8],
    signature: &[u8],
) -> Result<bool> {
    unsafe {
        let mut sig = helper::convert_buffer_to_gmpz(signature);
        let mut msg = vec![0u8; digest.len() + digest_info.len()];

        msg[0..digest_info.len()].clone_from_slice(digest_info);
        msg[digest_info.len()..].clone_from_slice(digest);

        let res = nettle_rsa_pkcs1_verify(
            &public.context,
            msg.len(),
            msg.as_ptr(),
            &mut sig,
        ) == 1;

        __gmpz_clear(&mut sig);
        Ok(res)
    }
}

/// Signs the message hashed with `hash` using `public`/`private`,
/// producing `signature`.
///
/// The `signature` buffer is expected to be of the size of the modulo
/// of `public`.
///
/// The function signs the digest produced by `hash` after padding the
/// digest using `EMSA-PKCS1-v1_5`.
pub fn sign_pkcs1<H: Hash + Pkcs1Hash, R: Random>(
    public: &PublicKey,
    private: &PrivateKey,
    hash: &mut H,
    random: &mut R,
    signature: &mut [u8],
) -> Result<()> {
    let mut dst = vec![0u8; hash.digest_size()];

    hash.digest(&mut dst);
    sign_digest_pkcs1(public, private, &dst, hash.oid(), random, signature)
}

/// Verifies `signature` of the data hashed by `hash` using `public`.
///
/// Returns `true` if the signature is valid.
///
/// The signature is expected to be padded using `EMSA-PKCS1-v1_5`.
pub fn verify_pkcs1<H: Hash + Pkcs1Hash>(
    public: &PublicKey,
    hash: &mut H,
    signature: &[u8],
) -> Result<bool> {
    let mut dst = vec![0u8; hash.digest_size()];

    hash.digest(&mut dst);
    verify_digest_pkcs1(public, &dst, hash.oid(), signature)
}

/// Encrypts `plaintext` using `public`, producing `ciphertext`.
///
/// `ciphertext` must have the same size as the modulo of `public`.
///
/// The plaintext is padded using `RSAES-PKCS1-v1_5`.
pub fn encrypt_pkcs1<R: Random>(
    public: &PublicKey,
    random: &mut R,
    plaintext: &[u8],
    ciphertext: &mut [u8],
) -> Result<()> {
    unsafe {
        let mut out = zeroed();

        __gmpz_init(&mut out);

        if nettle_rsa_encrypt(
            &public.context,
            random.context(),
            Some(R::random_impl),
            plaintext.len(),
            plaintext.as_ptr(),
            &mut out,
        ) == 1
        {
            helper::write_gmpz_into_slice(out, ciphertext, "ciphertext")
        } else {
            __gmpz_clear(&mut out);

            Err(Error::EncryptionFailed)
        }
    }
}

/// Decrypts `ciphertext` using `public`/`private`.
///
/// Returns the resulting plaintext.
///
/// The ciphertext expected to be padded using
/// `RSAES-PKCS1-v1_5`.
///
/// # Errors
///
/// Returns `InvalidArgument` if `plaintext`'s size is greater or
/// equal than `public`'s modulo.
pub fn decrypt_pkcs1<R: Random>(
    public: &PublicKey,
    private: &PrivateKey,
    random: &mut R,
    ciphertext: &[u8],
    plaintext: &mut [u8],
) -> Result<()> {
    if plaintext.len() >= public.modulo_bytes {
        return Err(Error::invalid_argument("plaintext"));
    }

    // Fill with random bytes prior to decryption as recommended by
    // Nettle's documentation.  Should the decryption fail, the
    // plaintext will be unusable.
    random.random(plaintext);

    let mut inp = helper::convert_buffer_to_gmpz(ciphertext);
    unsafe {
        // Try to decrypt it and ignore the return value as
        // recommended by Nettle's documentation.
        nettle_rsa_sec_decrypt(
            &public.context,
            &private.context,
            random.context(),
            Some(R::random_impl),
            plaintext.len(),
            plaintext.as_mut_ptr(),
            &mut inp,
        );
        __gmpz_clear(&mut inp);
    }
    Ok(())
}

/// Decrypts `ciphertext` using `public`/`private`.
///
/// Returns the resulting plaintext.
///
/// The ciphertext expected to be padded using `RSAES-PKCS1-v1_5`.
///
/// # <span style="color: red">Important note:</span>
///
/// Side-channel leakage from the caller's use of length and return
/// value may still provide an oracle useable for a
/// Bleichenbacher-style chosen ciphertext attack.
///
/// If you know the size of the plaintext in advance, it is better to
/// use [`decrypt_pkcs1`](fn.decrypt_pkcs1.html).
///
/// # Errors
///
/// Returns `DecryptionFailed` if the decryption failed.
pub fn decrypt_pkcs1_insecure<R: Random>(
    public: &PublicKey,
    private: &PrivateKey,
    random: &mut R,
    ciphertext: &[u8],
) -> Result<Box<[u8]>> {
    unsafe {
        let mut inp = helper::convert_buffer_to_gmpz(ciphertext);
        let mut buf = vec![0u8; public.modulo_bytes];
        let mut buf_len = public.modulo_bytes;
        let res = nettle_rsa_decrypt_tr(
            &public.context,
            &private.context,
            random.context(),
            Some(R::random_impl),
            &mut buf_len,
            buf.as_mut_ptr(),
            &mut inp,
        ) == 1;

        __gmpz_clear(&mut inp);

        if res {
            buf.truncate(buf_len);
            Ok(buf.into())
        } else {
            Err(Error::DecryptionFailed)
        }
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::random::Yarrow;

    const N: &[u8] = b"\xc8\xa2\x06\x91\x82\x39\x4a\x2a\xb7\xc3\xf4\x19\x0c\x15\x58\x9c\x56\xa2\xd4\xbc\x42\xdc\xa6\x75\xb3\x4c\xc9\x50\xe2\x46\x63\x04\x84\x41\xe8\xaa\x59\x3b\x2b\xc5\x9e\x19\x8b\x8c\x25\x7e\x88\x21\x20\xc6\x23\x36\xe5\xcc\x74\x50\x12\xc7\xff\xb0\x63\xee\xbe\x53\xf3\xc6\x50\x4c\xba\x6c\xfe\x51\xba\xa3\xb6\xd1\x07\x4b\x2f\x39\x81\x71\xf4\xb1\x98\x2f\x4d\x65\xca\xf8\x82\xea\x4d\x56\xf3\x2a\xb5\x7d\x0c\x44\xe6\xad\x4e\x9c\xf5\x7a\x43\x39\xeb\x69\x62\x40\x6e\x35\x0c\x1b\x15\x39\x71\x83\xfb\xf1\xf0\x35\x3c\x9f\xc9\x91";
    const E: &[u8] = b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x01";
    const D: &[u8] = b"\x5d\xfc\xb1\x11\x07\x2d\x29\x56\x5b\xa1\xdb\x3e\xc4\x8f\x57\x64\x5d\x9d\x88\x04\xed\x59\x8a\x4d\x47\x02\x68\xa8\x90\x67\xa2\xc9\x21\xdf\xf2\x4b\xa2\xe3\x7a\x3c\xe8\x34\x55\x50\x00\xdc\x86\x8e\xe6\x58\x8b\x74\x93\x30\x35\x28\xb1\xb3\xa9\x4f\x0b\x71\x73\x0c\xf1\xe8\x6f\xca\x5a\xee\xdc\x3a\xfa\x16\xf6\x5c\x01\x89\xd8\x10\xdd\xcd\x81\x04\x9e\xbb\xd0\x39\x18\x68\xc5\x0e\xde\xc9\x58\xb3\xa2\xaa\xef\xf6\xa5\x75\x89\x7e\x2f\x20\xa3\xab\x54\x55\xc1\xbf\xa5\x50\x10\xac\x51\xa7\x79\x9b\x1f\xf8\x48\x36\x44\xa3\xd4\x25";
    const P: &[u8] = b"\xd0\x6f\xfa\x56\x3b\xd0\xb8\xb9\x7f\x00\x72\xea\x07\x51\x03\xec\x1f\xf0\xb1\x7f\xd6\xbd\xaf\x94\xa7\x5a\xe6\x03\x59\x82\xf6\x80\x24\x9d\x77\xcc\x15\x96\xa6\x54\xf1\x4c\x51\x7a\x13\xb9\x5f\x1a\x8c\x5d\xe8\xcc\x84\xad\x38\xd2\xe6\xaa\x4f\xcd\x54\x1a\x32\xeb";
    const Q: &[u8] = b"\xf6\x6a\x24\x68\xb1\x48\x1a\x0d\x6b\x60\x49\x67\xf0\x58\x03\x89\xd6\x06\x1f\x03\x3a\x9b\x57\xb7\x4d\x1b\xef\xe8\x10\x11\x98\x94\x5b\x80\x93\x04\x24\x1d\x02\xfd\x5f\x92\x1f\xe9\x59\xfb\xec\xdb\x18\x00\x54\x86\x9a\xdb\x17\xd3\x21\xf9\xd5\xd5\x98\x12\x3e\x73";

    #[test]
    fn pkcs1_ciphertext_too_short() {
        let mut rnd = Yarrow::default();
        let n = &b"\xc8\xa2\x06\x91\x82\x39\x4a\x2a\xb7\xc3\xf4\x19\x0c\x15\x58\x9c\x56\xa2\xd4\xbc\x42\xdc\xa6\x75\xb3\x4c\xc9\x50\xe2\x46\x63\x04\x84\x41\xe8\xaa\x59\x3b\x2b\xc5\x9e\x19\x8b\x8c\x25\x7e\x88\x21\x20\xc6\x23\x36\xe5\xcc\x74\x50\x12\xc7\xff\xb0\x63\xee\xbe\x53\xf3\xc6\x50\x4c\xba\x6c\xfe\x51\xba\xa3\xb6\xd1\x07\x4b\x2f\x39\x81\x71\xf4\xb1\x98\x2f\x4d\x65\xca\xf8\x82\xea\x4d\x56\xf3\x2a\xb5\x7d\x0c\x44\xe6\xad\x4e\x9c\xf5\x7a\x43\x39\xeb\x69\x62\x40\x6e\x35\x0c\x1b\x15\x39\x71\x83\xfb\xf1\xf0\x35\x3c\x9f\xc9\x91"[..];
        let e = &b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x01"[..];
        let public = PublicKey::new(n, e).unwrap();
        let mut ciphertext = [0u8; 1];

        assert!(encrypt_pkcs1(&public, &mut rnd, &b"123"[..], &mut ciphertext)
            .is_err());
    }

    /// Insufficient validation of RSA ciphertexts crash Nettle.
    ///
    /// See CVE-2021-3580.
    #[test]
    fn cve_2021_3580_ciphertext_too_long() {
        let mut rnd = Yarrow::default();
        let public = PublicKey::new(N, E).unwrap();
        let private = PrivateKey::new(D, P, Q, None).unwrap();

        let ciphertext = vec![1u8; public.n().len() / 8 + 1000];
        assert!(decrypt_pkcs1_insecure(&public, &private,
                                       &mut rnd, &ciphertext).is_err());
        let mut plaintext = vec![0; 256];
        assert!(decrypt_pkcs1(&public, &private, &mut rnd,
                              &ciphertext, &mut plaintext).is_err());
    }

    /// Insufficient validation of RSA ciphertexts crash Nettle.
    ///
    /// See CVE-2021-3580.
    #[test]
    fn cve_2021_3580_zero_ciphertext() {
        let mut rnd = Yarrow::default();
        let public = PublicKey::new(N, E).unwrap();
        let private = PrivateKey::new(D, P, Q, None).unwrap();

        let ciphertext = vec![0u8; 0];
        assert!(decrypt_pkcs1_insecure(&public, &private,
                                       &mut rnd, &ciphertext).is_err());

        let mut plaintext = vec![0; 256];
        assert!(decrypt_pkcs1(&public, &private, &mut rnd,
                              &ciphertext, &mut plaintext).is_err());
    }

    #[test]
    fn rsa_pkcs15_sign() {
        let mut rnd = Yarrow::default();
        let n = &b"\xc8\xa2\x06\x91\x82\x39\x4a\x2a\xb7\xc3\xf4\x19\x0c\x15\x58\x9c\x56\xa2\xd4\xbc\x42\xdc\xa6\x75\xb3\x4c\xc9\x50\xe2\x46\x63\x04\x84\x41\xe8\xaa\x59\x3b\x2b\xc5\x9e\x19\x8b\x8c\x25\x7e\x88\x21\x20\xc6\x23\x36\xe5\xcc\x74\x50\x12\xc7\xff\xb0\x63\xee\xbe\x53\xf3\xc6\x50\x4c\xba\x6c\xfe\x51\xba\xa3\xb6\xd1\x07\x4b\x2f\x39\x81\x71\xf4\xb1\x98\x2f\x4d\x65\xca\xf8\x82\xea\x4d\x56\xf3\x2a\xb5\x7d\x0c\x44\xe6\xad\x4e\x9c\xf5\x7a\x43\x39\xeb\x69\x62\x40\x6e\x35\x0c\x1b\x15\x39\x71\x83\xfb\xf1\xf0\x35\x3c\x9f\xc9\x91"[..];
        let e = &b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x01"[..];
        let d = &b"\x5d\xfc\xb1\x11\x07\x2d\x29\x56\x5b\xa1\xdb\x3e\xc4\x8f\x57\x64\x5d\x9d\x88\x04\xed\x59\x8a\x4d\x47\x02\x68\xa8\x90\x67\xa2\xc9\x21\xdf\xf2\x4b\xa2\xe3\x7a\x3c\xe8\x34\x55\x50\x00\xdc\x86\x8e\xe6\x58\x8b\x74\x93\x30\x35\x28\xb1\xb3\xa9\x4f\x0b\x71\x73\x0c\xf1\xe8\x6f\xca\x5a\xee\xdc\x3a\xfa\x16\xf6\x5c\x01\x89\xd8\x10\xdd\xcd\x81\x04\x9e\xbb\xd0\x39\x18\x68\xc5\x0e\xde\xc9\x58\xb3\xa2\xaa\xef\xf6\xa5\x75\x89\x7e\x2f\x20\xa3\xab\x54\x55\xc1\xbf\xa5\x50\x10\xac\x51\xa7\x79\x9b\x1f\xf8\x48\x36\x44\xa3\xd4\x25"[..];
        let p = &b"\xd0\x6f\xfa\x56\x3b\xd0\xb8\xb9\x7f\x00\x72\xea\x07\x51\x03\xec\x1f\xf0\xb1\x7f\xd6\xbd\xaf\x94\xa7\x5a\xe6\x03\x59\x82\xf6\x80\x24\x9d\x77\xcc\x15\x96\xa6\x54\xf1\x4c\x51\x7a\x13\xb9\x5f\x1a\x8c\x5d\xe8\xcc\x84\xad\x38\xd2\xe6\xaa\x4f\xcd\x54\x1a\x32\xeb"[..];
        let q = &b"\xf6\x6a\x24\x68\xb1\x48\x1a\x0d\x6b\x60\x49\x67\xf0\x58\x03\x89\xd6\x06\x1f\x03\x3a\x9b\x57\xb7\x4d\x1b\xef\xe8\x10\x11\x98\x94\x5b\x80\x93\x04\x24\x1d\x02\xfd\x5f\x92\x1f\xe9\x59\xfb\xec\xdb\x18\x00\x54\x86\x9a\xdb\x17\xd3\x21\xf9\xd5\xd5\x98\x12\x3e\x73"[..];
        let public = PublicKey::new(n, e).unwrap();
        let private = PrivateKey::new(d, p, q, None).unwrap();

        {
            let m = &b"\xe8\x31\x27\x42\xae\x23\xc4\x56\xef\x28\xa2\x31\x42\xc4\x49\x08\x95\x83\x27\x65\xda\xdc\xe0\x2a\xfe\x5b\xe5\xd3\x1b\x00\x48\xfb\xee\xe2\xcf\x21\x8b\x17\x47\xad\x4f\xd8\x1a\x2e\x17\xe1\x24\xe6\xaf\x17\xc3\x88\x8e\x6d\x2d\x40\xc0\x08\x07\xf4\x23\xa2\x33\xca\xd6\x2c\xe9\xea\xef\xb7\x09\x85\x6c\x94\xaf\x16\x6d\xba\x08\xe7\xa0\x69\x65\xd7\xfc\x0d\x8e\x5c\xb2\x65\x59\xc4\x60\xe4\x7b\xc0\x88\x58\x9d\x22\x42\xc9\xb3\xe6\x2d\xa4\x89\x6f\xab\x19\x9e\x14\x4e\xc1\x36\xdb\x8d\x84\xab\x84\xbc\xba\x04\xca\x3b\x90\xc8\xe5"[..];
            let expected = &b"\x28\x92\x8e\x19\xeb\x86\xf9\xc0\x00\x70\xa5\x9e\xdf\x6b\xf8\x43\x3a\x45\xdf\x49\x5c\xd1\xc7\x36\x13\xc2\x12\x98\x40\xf4\x8c\x4a\x2c\x24\xf1\x1d\xf7\x9b\xc5\xc0\x78\x2b\xce\xdd\xe9\x7d\xbb\xb2\xac\xc6\xe5\x12\xd1\x9f\x08\x50\x27\xcd\x57\x50\x38\x45\x3d\x04\x90\x54\x13\xe9\x47\xe6\xe1\xdd\xdb\xeb\x35\x35\xcd\xb3\xd8\x97\x1f\xe0\x20\x05\x06\x94\x10\x56\xf2\x12\x43\x50\x3c\x83\xea\xdd\xe0\x53\xed\x86\x6c\x0e\x02\x50\xbe\xdd\xd9\x27\xa0\x82\x12\xaa\x8a\xc0\xef\xd6\x16\x31\xef\x89\xd8\xd0\x49\xef\xb3\x6b\xb3\x5f"[..];
            let mut hsh = Sha1::default();
            let mut sig = vec![0u8; 1024 / 8];

            hsh.update(m);
            assert!(sign_pkcs1(
                &public, &private, &mut hsh, &mut rnd, &mut sig
            )
            .is_ok());
            assert_eq!(sig, expected);
            hsh.update(m);
            assert_eq!(
                verify_pkcs1(&public, &mut hsh, expected).ok(),
                Some(true)
            );
        }

        {
            let m = &b"\x4c\x95\x07\x3d\xac\x19\xd0\x25\x6e\xaa\xdf\xf3\x50\x59\x10\xe4\x31\xdd\x50\x01\x81\x36\xaf\xea\xf6\x90\xb7\xd1\x80\x69\xfc\xc9\x80\xf6\xf5\x41\x35\xc3\x0a\xcb\x76\x9b\xee\x23\xa7\xa7\x2f\x6c\xe6\xd9\x0c\xbc\x85\x8c\x86\xdb\xbd\x64\xba\x48\xa0\x7c\x6d\x7d\x50\xc0\xe9\x74\x6f\x97\x08\x6a\xd6\xc6\x8e\xe3\x8a\x91\xbb\xee\xeb\x22\x21\xaa\x2f\x2f\xb4\x09\x0f\xd8\x20\xd4\xc0\xce\x5f\xf0\x25\xba\x8a\xdf\x43\xdd\xef\x89\xf5\xf3\x65\x3d\xe1\x5e\xdc\xf3\xaa\x80\x38\xd4\x68\x69\x60\xfc\x55\xb2\x91\x7e\xc8\xa8\xf9\xa8"[..];
            let expected = &b"\x53\xab\x60\x0a\x41\xc7\x13\x93\xa2\x71\xb0\xf3\x2f\x52\x19\x63\x08\x7e\x56\xeb\xd7\xad\x04\x0e\x4e\xe8\xaa\x7c\x45\x0a\xd1\x8a\xc3\xc6\xa0\x5d\x4a\xe8\x91\x3e\x76\x3c\xfe\x96\x23\xbd\x9c\xb1\xeb\x4b\xed\x1a\x38\x20\x05\x00\xfa\x7d\xf3\xd9\x5d\xea\x48\x5f\x03\x2a\x0a\xb0\xc6\x58\x96\x78\xf9\xe8\x39\x1b\x5c\x2b\x13\x92\x99\x7a\xc9\xf8\x2f\x1d\x16\x88\x78\x91\x6a\xac\xe9\xac\x74\x55\x80\x80\x56\xaf\x81\x55\x23\x1a\x29\xf4\x29\x04\xb7\xab\x87\xa5\xd7\x1e\xd6\x39\x5e\xe0\xa9\xd0\x24\xb0\xca\x3d\x01\xfd\x71\x50"[..];
            let mut hsh = Sha1::default();
            let mut sig = vec![0u8; 1024 / 8];

            hsh.update(m);
            assert!(sign_pkcs1(
                &public, &private, &mut hsh, &mut rnd, &mut sig
            )
            .is_ok());
            assert_eq!(sig, expected);
            hsh.update(m);
            assert_eq!(
                verify_pkcs1(&public, &mut hsh, expected).ok(),
                Some(true)
            );
        }

        {
            let m = &b"\xe0\x75\xad\x4b\x0f\x9b\x5b\x20\x37\x6e\x46\x7a\x1a\x35\xe3\x08\x79\x3b\xa3\x8e\xd9\x83\xd0\x38\x87\xb8\xb8\x2e\xda\x63\x0e\x68\xb8\x61\x8d\xc4\x5b\x93\xde\x55\x55\xd7\xbc\xfe\xd2\x37\x56\x40\x1e\x61\xf5\x51\x67\x57\xde\x6e\xc3\x68\x7a\x71\x75\x5f\xb4\xa6\x6c\xfa\xa3\xdb\x0c\x9e\x69\xb6\x31\x48\x5b\x4c\x71\xc7\x62\xee\xa2\x29\xa0\x46\x9c\x73\x57\xa4\x40\x95\x07\x92\xba\x9c\xd7\xae\x02\x2a\x36\xb9\xa9\x23\xc2\xeb\xd2\xaa\x69\x89\x7f\x4c\xce\xba\x0e\x7a\xee\x97\x03\x3d\x03\x81\x07\x25\xa9\xb7\x31\x83\x3f\x27"[..];
            let expected = &b"\x64\x26\x09\xce\x08\x4f\x47\x92\x71\xdf\x59\x64\x80\x25\x2e\x2f\x89\x2b\x3e\x79\x82\xdf\xf9\x59\x94\xc3\xee\xda\x78\x7f\x80\xf3\xf6\x19\x8b\xbc\xe3\x3e\xc5\x51\x53\x78\xd4\xb5\x71\xd7\x18\x60\x78\xb7\x5b\x43\xae\xd1\x1d\x34\x25\x47\x38\x6c\x56\x96\xeb\x37\x99\xa0\xb2\x84\x75\xe5\x4c\xd4\xca\x7d\x03\x6d\xcd\x8a\x11\xf5\xe1\x08\x06\xf7\xd3\xb8\xcc\x4f\xcb\x3e\x93\xe8\x57\xbe\x95\x83\x44\xa3\x4e\x12\x68\x09\xc1\x5b\x3d\x33\x66\x1c\xf5\x7b\xf5\xc3\x38\xf0\x7a\xcc\xed\x60\xf1\x40\x19\x33\x5c\x15\x2d\x86\xb3\xb2"[..];
            let mut hsh = Sha1::default();
            let mut sig = vec![0u8; 1024 / 8];

            hsh.update(m);
            assert!(sign_pkcs1(
                &public, &private, &mut hsh, &mut rnd, &mut sig
            )
            .is_ok());
            assert_eq!(sig, expected);
            hsh.update(m);
            assert_eq!(
                verify_pkcs1(&public, &mut hsh, expected).ok(),
                Some(true)
            );
        }

        {
            let m = &b"\x18\x50\x01\x55\xd2\xe0\x58\x7d\x15\x26\x98\xc0\x7b\xa4\x4d\x4f\x04\xaa\x9a\x90\x0b\x77\xce\x66\x78\xa1\x37\xb2\x38\xb7\x3b\x1a\xea\x24\xa4\x09\xdb\x56\x3c\xf6\x35\x20\x9a\xea\x73\x5d\x3b\x3c\x18\xd7\xd5\x9f\xa1\x67\xe7\x60\xb8\x5d\x95\xe8\xaa\x21\xb3\x88\x1b\x1b\x20\x76\xf9\xd1\x55\x12\xae\x4f\x3d\x6e\x9a\xcc\x48\x0e\xc0\x8a\xbb\xec\xbf\xfe\x4a\xbf\x05\x31\xe8\x7d\x3f\x66\xde\x1f\x13\xfd\x1a\xa4\x12\x97\xca\x58\xe8\x7b\x2a\x56\xd6\x39\x9a\x4c\x63\x8d\xf4\x7e\x4e\x85\x1c\x0e\xc6\xe6\xd9\x7a\xdd\xcd\xe3\x66"[..];
            let expected = &b"\x42\xf3\xc3\xc7\x5f\x65\xad\x42\x05\x7b\xfa\xc1\x31\x03\x83\x7b\xf9\xf8\x42\x7c\x6e\xbc\x22\xa3\xad\xf7\xb8\xe4\x7a\x68\x57\xf1\xcb\x17\xd2\xa5\x33\xc0\xa9\x13\xdd\x9a\x8b\xdc\x17\x86\x22\x23\x60\xcb\xd7\xe6\x4b\x45\xfc\xf5\x4f\x5d\xa2\xf3\x42\x30\xab\x48\x06\xa0\x87\xf8\xbe\x47\xf3\x5c\x4e\x8f\xee\x2e\x6a\xa2\x91\x9a\x56\x67\x9c\xe2\xa5\x28\xa4\x4b\xf8\x18\x62\x0d\x5b\x00\xb9\xab\x0e\x1c\x8d\x2d\x72\x2b\x53\xd3\xa8\xcc\xa3\x5a\x99\x0e\xd2\x55\x36\xea\x65\x33\x5e\x82\x53\xa5\x4a\x68\xa6\x4a\x37\x3e\x0e\xd7"[..];
            let mut hsh = Sha1::default();
            let mut sig = vec![0u8; 1024 / 8];

            hsh.update(m);
            assert!(sign_pkcs1(
                &public, &private, &mut hsh, &mut rnd, &mut sig
            )
            .is_ok());
            assert_eq!(sig, expected);
            hsh.update(m);
            assert_eq!(
                verify_pkcs1(&public, &mut hsh, expected).ok(),
                Some(true)
            );
        }

        {
            let m = &b"\xf7\xf7\x9f\x9d\xf2\x76\x0f\xc8\x3c\x73\xc7\xcc\xea\x7e\xae\x48\x2d\xcf\xa5\xe0\x2a\xcf\x05\xe1\x05\xdb\x48\x28\x3f\x44\x06\x40\x43\x9a\x24\xca\x3b\x2a\x48\x22\x28\xc5\x8f\x3f\x32\xc3\x83\xdb\x3c\x48\x47\xd4\xbc\xc6\x15\xd3\xca\xc3\xeb\x2b\x77\xdd\x80\x04\x5f\x0b\x7d\xb8\x82\x25\xea\x7d\x4f\xa7\xe6\x45\x02\xb2\x9c\xe2\x30\x53\x72\x6e\xa0\x08\x83\xea\x5d\x80\x50\x25\x09\xa3\xb2\xdf\x74\xd2\x14\x2f\x6e\x70\xde\x22\xd9\xa1\x34\xa5\x02\x51\xe1\xa5\x31\x79\x8e\x74\x7e\x9d\x38\x6f\xe7\x9a\xe1\xde\xa0\x9e\x85\x1b"[..];
            let expected = &b"\xac\x2a\xe6\x6b\xca\x1e\xc1\x2a\x66\xa2\x90\x9f\xe2\x14\x8a\x1d\x49\x2d\x1e\xdd\x00\x06\x3b\x8b\x33\xaf\x74\x76\x0d\xc4\x05\x67\x18\xfd\x50\x41\xd4\xdf\xee\x12\xbe\xc7\x08\x1a\xb1\xba\xb2\xd0\xeb\x27\x12\xf3\x34\x50\x9f\x68\x89\xb1\x9d\x75\xd1\xfd\x0f\xc6\x1b\xf1\x29\x76\x10\x9c\x36\x14\xc4\x60\x51\xe2\xa4\x01\xb2\x08\x80\xd6\xe6\x4a\xd6\xa4\x7f\x23\x93\x98\x03\xd1\x38\xaa\x0a\x44\xbc\x41\xba\x63\x03\x07\x46\x62\x22\x48\x77\x14\x31\xdf\xf9\x7e\x8a\x85\x6f\x0b\x61\xd1\x14\xf8\x13\x91\x1e\xe2\x29\x65\x51\x55"[..];
            let mut hsh = Sha1::default();
            let mut sig = vec![0u8; 1024 / 8];

            hsh.update(m);
            assert!(sign_pkcs1(
                &public, &private, &mut hsh, &mut rnd, &mut sig
            )
            .is_ok());
            assert_eq!(sig, expected);
            hsh.update(m);
            assert_eq!(
                verify_pkcs1(&public, &mut hsh, expected).ok(),
                Some(true)
            );
        }

        {
            let m = &b"\x09\x9b\xf1\x7f\x16\xbc\xfd\x4c\x19\xb3\x4f\xec\xb4\xb3\x23\x3c\x9f\x9f\x98\x71\x8f\x67\xb3\x06\x5d\x95\xa5\xf8\x64\x23\x51\x36\x2b\x90\x09\x53\x44\x33\x98\x7f\x73\xce\x86\xb5\x13\x66\x97\x36\xb6\x52\x95\x35\x0c\x93\x4f\xd4\x06\x63\xe2\x4f\x3a\x10\x37\x77\x8a\x0b\xcd\x63\x00\x3c\xb9\x62\xfd\x99\xeb\x33\x93\xf7\xb2\x79\x2f\x20\x83\x69\x7b\x25\xf6\xc6\x82\xf6\x11\x0f\x16\x2f\xc9\xf7\x6e\x35\xc6\x15\x14\x82\x67\xdd\xff\x3d\x06\xcf\xfb\x0e\x7d\xee\x52\x30\xe8\x74\xa5\xc8\xad\xc4\x1b\x75\xba\xa0\xbe\x28\x0e\x9c"[..];
            let expected = &b"\x3a\x2b\x75\x71\x61\x92\x72\xb8\x1d\x35\x62\xa1\x1c\x64\x45\x02\x89\x44\x21\x58\x3e\x02\x87\x9f\x5a\x77\x59\xfb\x64\xec\x2a\xb8\x10\x5f\x7d\x11\x94\x7c\x8e\x4b\xfc\xa8\x72\x19\xe5\x29\x87\xaa\xd3\xb8\x1c\xbd\x48\x31\x66\xed\x78\x15\x2a\xf2\x44\x60\xc9\x08\x87\x9f\x34\xc8\x70\x57\x31\x27\xe3\x44\x8c\x8f\xbf\x43\x02\x83\x50\xc9\x75\xbb\xc3\xa9\x99\x19\x6a\x3e\x99\x61\x22\x8a\x2b\xb1\x5b\x49\x85\xe9\x5b\xba\x97\x0a\xc4\xad\x2a\xc5\xb4\x2a\xc5\x1d\xbc\x65\x09\xef\xfc\x13\x39\x66\x93\x98\x0f\xc8\x9b\xa4\x4c\x7b"[..];
            let mut hsh = Sha1::default();
            let mut sig = vec![0u8; 1024 / 8];

            hsh.update(m);
            assert!(sign_pkcs1(
                &public, &private, &mut hsh, &mut rnd, &mut sig
            )
            .is_ok());
            assert_eq!(sig, expected);
            hsh.update(m);
            assert_eq!(
                verify_pkcs1(&public, &mut hsh, expected).ok(),
                Some(true)
            );
        }

        {
            let m = &b"\xfb\x40\xa7\x3d\xc8\x2f\x16\x7f\x9c\x2b\xf9\x8a\x99\x1e\xa8\x2f\xdb\x01\x41\xdb\xad\x44\x87\x1a\xfd\x70\xf0\x5a\x0e\x0b\xf9\xf2\x6d\xbc\xbd\x62\x26\xaf\xc6\xdc\x37\x3b\x23\x04\x45\xc2\xba\xf5\x8e\xd9\xe0\x84\x1f\xa9\x27\xc8\x47\x95\x77\xda\x4b\x1e\x61\xd9\x5b\x03\xaf\x31\xc5\xac\x40\x1d\x69\xc8\x13\x6b\x6d\x36\xa1\x80\x32\x21\x70\x9b\x86\x70\xe5\x5e\x1b\x5d\x5a\x8a\x37\x63\x70\x0a\xae\x5e\xa6\x33\x0e\xee\x2b\x4a\x19\x1c\xf1\x46\x78\x40\x03\xd8\xad\x22\x18\xa9\x4a\x5f\x68\xe3\x60\x0e\xbe\xf2\x3b\xa4\xcf\x8c"[..];
            let expected = &b"\xb1\x03\x22\x60\x2c\x28\x4f\x40\x79\xe5\x09\xfa\xf3\xf4\x0a\x3d\x2a\xf3\xab\xef\x9f\x09\x17\x1f\xdd\x16\x46\x9d\x67\x9b\xb9\xad\xc7\xe2\xac\xb1\xad\xdb\x0b\xd5\xb3\x8b\x5c\x4d\x98\x6b\x43\xc7\x9b\x97\x24\xf6\x1e\x99\xb5\xb3\x03\x63\x0b\x62\xd0\xd8\xd5\xf7\x65\x77\xfe\x7e\xa3\x87\x71\x0b\x43\x78\x9e\xe1\xb3\x5b\x61\x4b\x69\x1f\x0a\x27\xb7\x3b\xaf\x6b\xc3\xf2\x8e\xc2\x10\xb9\xd3\xe4\xc5\xa2\x72\x9c\xc1\x20\x3b\x74\xef\x70\xe3\x15\xcf\xe5\xd0\x6e\x04\x0a\xee\x6b\x3d\x22\xd9\x1d\x6e\x22\x9f\x69\x0a\x96\x6d\xd9"[..];
            let mut hsh = Sha1::default();
            let mut sig = vec![0u8; 1024 / 8];

            hsh.update(m);
            assert!(sign_pkcs1(
                &public, &private, &mut hsh, &mut rnd, &mut sig
            )
            .is_ok());
            assert_eq!(sig, expected);
            hsh.update(m);
            assert_eq!(
                verify_pkcs1(&public, &mut hsh, expected).ok(),
                Some(true)
            );
        }

        {
            let m = &b"\x97\xe7\x49\x60\xdb\xd9\x81\xd4\x6a\xad\xc0\x21\xa6\xcf\x18\x1d\xdd\xe6\xe4\xcf\xcb\x4b\x63\x82\x60\xc0\xa5\x19\xc4\x5f\xab\xa2\x99\xd0\xca\x2e\x80\xbf\x50\xdf\xde\x8d\x6a\x42\xe0\x46\x45\xdf\xbc\xd4\x74\x0f\x3a\x72\x92\x0e\x74\x63\x28\x51\xd9\xe3\xd0\x1a\x78\x5e\x9b\x49\x7c\xe0\xb1\x75\xf2\xcd\x37\x3b\xd3\xd2\x76\xd6\x3e\x1b\x39\xf0\x05\xc6\x76\xb8\x6b\x98\x31\x35\x2c\xef\x9e\xda\xbe\xf8\x86\x5a\xd7\x22\xeb\xbe\x2f\xd3\xef\xb4\x87\x59\xf2\x2a\xea\x23\xfb\x1b\x33\x31\x59\xa9\xcf\xc9\x8a\x6d\xc4\x6c\x5b\x0b"[..];
            let expected = &b"\x60\xeb\xc9\xe4\xe2\xe2\xb4\xfa\x6d\x31\xc5\x7d\x0b\x86\x83\x5e\x8d\x20\x1d\x21\xc2\x74\xcf\x54\x52\xcd\xd7\xef\x28\x57\xdc\x78\x0d\xde\x35\x26\xf3\x65\x8c\x4f\x2c\x87\x10\xea\xae\x48\x70\xd2\x75\x99\x7e\x5c\xbb\x26\x8e\x3b\xd2\x51\xf5\x43\xb8\x82\x8f\xeb\x85\xc2\x11\xc8\x58\xe4\x7a\x74\xcb\x12\x2d\xc1\x7f\x26\xfe\x92\xb4\xaf\xee\xcb\xf1\xe2\x0b\xea\x75\xc7\x94\xc0\x48\x2a\xa6\x53\x2e\x87\x95\x5d\xba\x24\x9f\x0f\xa6\x56\x2b\xdf\x8f\x4c\xcd\x8a\x63\xda\x69\xd1\xf3\x37\x52\x3f\x65\x20\x6f\xb8\xeb\x16\x31\x73"[..];
            let mut hsh = Sha1::default();
            let mut sig = vec![0u8; 1024 / 8];

            hsh.update(m);
            assert!(sign_pkcs1(
                &public, &private, &mut hsh, &mut rnd, &mut sig
            )
            .is_ok());
            assert_eq!(sig, expected);
            hsh.update(m);
            assert_eq!(
                verify_pkcs1(&public, &mut hsh, expected).ok(),
                Some(true)
            );
        }

        {
            let m = &b"\x95\xd0\x46\x24\xb9\x98\x93\x8d\xd0\xa5\xba\x6d\x70\x42\xaa\x88\xa2\x67\x4d\xad\x43\x8a\x0d\x31\xab\xb7\x97\x9d\x8d\xe3\xde\xa4\x1e\x7e\x63\x58\x7a\x47\xb5\x9d\x43\x64\x33\xdd\x8b\xb2\x19\xfd\xf4\x5a\xbb\x90\x15\xa5\x0b\x4b\x20\x11\x61\xb9\xc2\xa4\x7c\x30\x4b\x80\xc4\x04\x0f\xb8\xd1\xfa\x0c\x62\x31\x00\xcd\xed\x66\x1b\x8e\xb5\x2f\xa0\xa0\xd5\x09\xa7\x0f\x3c\xf4\xbd\x83\x04\x7a\xd9\x64\xff\xee\x92\x41\x92\xf2\x8e\x73\xc6\x3b\x3e\xfd\x9c\x99\xc8\xb7\xa1\x31\x45\xac\xc3\x0d\x2d\xc0\x63\xd8\x0f\x96\xab\xe2\x86"[..];
            let expected = &b"\x85\x9c\xc4\xfc\xd1\xb8\x8c\xcd\xa6\x95\xb1\x23\x11\xcf\x8b\xdc\xa3\xb4\xc1\x35\xfa\xa1\x1f\x90\x53\xdc\x10\xf4\xbf\x12\xe5\xf2\x17\x9b\xe6\xab\x5a\xd9\x0f\x8d\x11\x5f\x5d\xf7\x95\xa7\x73\x40\xe2\x06\x62\x80\x9f\xa7\x32\xb9\x25\x60\xad\xcf\xfd\xb0\xdd\xb7\x2d\x33\x81\x1e\x94\xf8\x54\x33\x06\x80\xf2\xb2\x38\x30\x09\x95\xa9\x11\x3a\x46\x9a\xfd\x9e\x75\x6f\x64\x92\x08\xd2\x94\x2f\xeb\xff\xb2\x2e\x83\x22\x79\x06\x3e\xc5\xb5\x7a\xb5\x42\xd9\xbb\xc5\x6e\x82\xcd\xc6\xa0\x3b\x00\xd1\x0d\x45\x80\x15\x75\xe9\x49\xe1"[..];
            let mut hsh = Sha1::default();
            let mut sig = vec![0u8; 1024 / 8];

            hsh.update(m);
            assert!(sign_pkcs1(
                &public, &private, &mut hsh, &mut rnd, &mut sig
            )
            .is_ok());
            assert_eq!(sig, expected);
            hsh.update(m);
            assert_eq!(
                verify_pkcs1(&public, &mut hsh, expected).ok(),
                Some(true)
            );
        }

        {
            let m = &b"\x20\x71\x02\xf5\x98\xec\x28\x00\x45\xbe\x67\x59\x2f\x5b\xba\x25\xba\x2e\x2b\x56\xe0\xd2\x39\x7c\xbe\x85\x7c\xde\x52\xda\x8c\xca\x83\xae\x1e\x29\x61\x5c\x70\x56\xaf\x35\xe8\x31\x9f\x2a\xf8\x6f\xdc\xcc\x44\x34\xcd\x77\x07\xe3\x19\xc9\xb2\x35\x66\x59\xd7\x88\x67\xa6\x46\x7a\x15\x4e\x76\xb7\x3c\x81\x26\x0f\x3a\xb4\x43\xcc\x03\x9a\x0d\x42\x69\x50\x76\xa7\x9b\xd8\xca\x25\xeb\xc8\x95\x2e\xd4\x43\xc2\x10\x3b\x29\x00\xc9\xf5\x8b\x6a\x1c\x8a\x62\x66\xe4\x38\x80\xcd\xa9\x3b\xc6\x4d\x71\x4c\x98\x0c\xd8\x68\x8e\x8e\x63"[..];
            let expected = &b"\x77\xf0\xf2\xa0\x48\x48\xfe\x90\xa8\xeb\x35\xab\x5d\x94\xca\xe8\x43\xdb\x61\x02\x4d\x01\x67\x28\x9e\xea\x92\xe5\xd1\xe1\x0a\x52\x6e\x42\x0f\x2d\x33\x4f\x1b\xf2\xaa\x7e\xa4\xe1\x4a\x93\xa6\x8d\xba\x60\xfd\x2e\xde\x58\xb7\x94\xdc\xbd\x37\xdc\xb1\x96\x78\x77\xd6\xb6\x7d\xa3\xfd\xf2\xc0\xc7\x43\x3e\x47\x13\x4d\xde\x00\xc9\xc4\xd4\x07\x2e\x43\x36\x1a\x76\x7a\x52\x76\x75\xd8\xbd\xa7\xd5\x92\x1b\xd4\x83\xc9\x55\x19\x50\x73\x9e\x9b\x2b\xe0\x27\xdf\x30\x15\xb6\x1f\x75\x1a\xc1\xd9\xf3\x7b\xea\x32\x14\xd3\xc8\xdc\x96"[..];
            let mut hsh = Sha1::default();
            let mut sig = vec![0u8; 1024 / 8];

            hsh.update(m);
            assert!(sign_pkcs1(
                &public, &private, &mut hsh, &mut rnd, &mut sig
            )
            .is_ok());
            assert_eq!(sig, expected);
            hsh.update(m);
            assert_eq!(
                verify_pkcs1(&public, &mut hsh, expected).ok(),
                Some(true)
            );
        }

        {
            let m = &b"\x64\x4a\x3d\xe6\x10\x0d\xd8\xa3\x6c\x84\x14\x46\xb9\x37\x6b\x34\x5d\x47\x8a\x32\x94\x50\xa6\x6f\x62\x93\x13\xc5\xcc\x55\x13\x3c\x47\x82\xec\xd0\x71\x96\x3d\x74\xff\xbd\x91\x56\xf6\x39\x35\x65\x1f\x05\x47\x11\xb4\xb8\x10\x51\x60\xab\x94\xeb\x67\x5e\x66\xf0\x20\x18\xc1\x85\xac\xce\xd5\xe9\xe2\x46\x5f\xb4\x89\x7d\x7c\x5d\xca\x45\xbf\xff\xe4\x32\x81\x5d\x22\xb7\x78\x23\x0b\x8d\x8c\x6f\xd8\x54\xe5\x8f\x2b\x7d\x49\x0d\xfd\x67\xa5\xe8\xd8\xc8\xe9\x25\x47\x4f\x19\x86\x8c\x88\xa1\x9a\x90\xff\xea\xf6\x20\xa8\x2a\xc1"[..];
            let expected = &b"\x48\x11\x57\xd6\x5c\x91\xa0\x9b\x87\x1b\xc9\xf7\x04\xac\x2c\x31\xb1\xe6\xb0\x97\xe0\xd4\x9e\x29\x1e\x03\x7c\x11\xe7\x07\xf4\x8a\xc1\x66\x55\x8e\x9e\x8c\x11\x4e\xe4\x44\xc4\xd6\x49\xe7\xe2\xcf\x73\x74\x26\x9b\x03\x6f\xea\x99\xbb\x35\x66\xd1\x34\xd7\xae\x5d\xd4\x14\x1f\x73\x26\x88\x47\x46\xd3\x96\xbe\x32\x07\xda\x2d\x9f\x56\x86\x25\x85\x99\x3d\xb4\x2c\xe2\x82\xaa\x2f\x04\x8d\x12\xbc\xcf\xf1\x2f\x14\xb9\xd1\xac\x4e\x33\xd3\x71\xfd\x4c\x0d\xe1\x56\x3c\x97\x88\x26\x46\x8e\x6a\x37\x99\xf6\x4d\x39\x85\xd4\x68\x9e"[..];
            let mut hsh = Sha224::default();
            let mut sig = vec![0u8; 1024 / 8];

            hsh.update(m);
            assert!(sign_pkcs1(
                &public, &private, &mut hsh, &mut rnd, &mut sig
            )
            .is_ok());
            assert_eq!(sig, expected);
            hsh.update(m);
            assert_eq!(
                verify_pkcs1(&public, &mut hsh, expected).ok(),
                Some(true)
            );
        }

        {
            let m = &b"\x46\x7e\x8e\xa6\x34\xf7\x99\x5d\xc4\x6c\x11\xb8\xab\x0b\x75\x08\x89\x46\x81\xe8\x1c\x35\x02\xc3\xb3\x35\xe8\x97\xe6\xd6\x9d\xf8\x85\xf4\x95\x57\xce\x23\x27\x84\xe3\x51\x9b\x72\x7b\xa6\x84\x3b\xd7\xaf\x50\x63\xf8\xbc\x1d\x61\x0f\x86\xce\x5b\x35\x15\x5e\x32\x5c\xe1\x75\xbe\x85\x38\x39\x5b\x34\xdf\x67\xa4\x21\xfc\xa2\x7e\x31\x33\x0b\x59\xa4\x10\x11\xb2\x90\xa5\x8b\xdc\x8e\x74\x04\x01\xb3\x8f\x55\x64\xc2\xfd\x7a\xe8\x9f\x60\x9e\xd6\x07\xd5\x78\xdb\x7f\x1c\xda\x50\x8a\xf9\x87\xbe\x1f\xd9\x46\xa2\x5a\xb9\x34\x6d"[..];
            let expected = &b"\x2b\x1f\xfb\x37\x0d\x51\x8a\x82\x64\x6d\x86\x82\x8d\xb1\xfc\x7e\x8b\xfe\x73\xee\x87\x8d\xa1\x20\xfa\x92\x73\x7c\x91\x74\x68\x89\x95\xf2\x25\x5b\x29\xe8\x3b\x28\xc2\x44\xcc\x56\x3c\x9b\x33\xef\xd3\xf9\xf9\xe1\x63\x8e\x2c\x16\xe2\x4f\x2e\xae\x19\x43\x56\x96\xb2\xf4\xd1\xcf\x73\x06\x4f\xc1\xcf\xcc\xb2\x27\x8c\x01\xf0\x97\x9e\x7d\xe7\x46\x3b\xf8\x41\x7b\xd6\x98\x6f\xbf\x1d\x34\xd3\x82\xa9\x78\xce\x79\x95\x82\x44\x2a\xfc\xc9\x2b\x4f\xe7\x43\x21\x6b\x6f\x15\x1f\x6a\x56\x1d\x97\x9c\xf6\x83\xca\xb6\xaf\x2f\xf4\xc5"[..];
            let mut hsh = Sha256::default();
            let mut sig = vec![0u8; 1024 / 8];

            hsh.update(m);
            assert!(sign_pkcs1(
                &public, &private, &mut hsh, &mut rnd, &mut sig
            )
            .is_ok());
            assert_eq!(sig, expected);
            hsh.update(m);
            assert_eq!(
                verify_pkcs1(&public, &mut hsh, expected).ok(),
                Some(true)
            );
        }

        {
            let m = &b"\x46\x7e\x8e\xa6\x34\xf7\x99\x5d\xc4\x6c\x11\xb8\xab\x0b\x75\x08\x89\x46\x81\xe8\x1c\x35\x02\xc3\xb3\x35\xe8\x97\xe6\xd6\x9d\xf8\x85\xf4\x95\x57\xce\x23\x27\x84\xe3\x51\x9b\x72\x7b\xa6\x84\x3b\xd7\xaf\x50\x63\xf8\xbc\x1d\x61\x0f\x86\xce\x5b\x35\x15\x5e\x32\x5c\xe1\x75\xbe\x85\x38\x39\x5b\x34\xdf\x67\xa4\x21\xfc\xa2\x7e\x31\x33\x0b\x59\xa4\x10\x11\xb2\x90\xa5\x8b\xdc\x8e\x74\x04\x01\xb3\x8f\x55\x64\xc2\xfd\x7a\xe8\x9f\x60\x9e\xd6\x07\xd5\x78\xdb\x7f\x1c\xda\x50\x8a\xf9\x87\xbe\x1f\xd9\x46\xa2\x5a\xb9\x34\x6d"[..];
            let expected = &b"\xc7\x00\x28\x35\x57\xda\xfd\x1f\x88\x34\xec\x0b\x7a\x7e\xc7\x33\x71\x9c\xb0\xbe\x6e\xdb\x19\xf1\x06\x4e\xe9\x4c\x75\xb8\xd9\xd4\x94\x58\xfa\x18\x4f\x7d\x6d\xfa\x5a\x6d\xa2\xd2\xeb\xee\xf0\x65\x0d\x8a\xf3\x54\x47\x82\x3c\x83\xa7\x73\x78\x24\x50\x9a\x74\x25\x33\x9c\xaf\x99\x10\x1c\xd1\xfe\x3d\xb8\x83\xfb\x98\xd1\x72\x12\x7e\x30\xe3\x8d\x1d\x6f\x9e\x36\x54\x93\x7c\xd6\x8c\xbb\x4e\xa2\x28\xc8\x16\x06\x4f\xa8\xca\x09\x50\xc7\xe7\xb6\xad\x25\x04\x55\x74\xa6\xa4\x06\x3b\x63\xf0\x74\x66\xb2\xcb\x5d\x73\x11\xb7\xcf"[..];
            let mut hsh = Sha384::default();
            let mut sig = vec![0u8; 1024 / 8];

            hsh.update(m);
            assert!(sign_pkcs1(
                &public, &private, &mut hsh, &mut rnd, &mut sig
            )
            .is_ok());
            assert_eq!(sig, expected);
            hsh.update(m);
            assert_eq!(
                verify_pkcs1(&public, &mut hsh, expected).ok(),
                Some(true)
            );
        }

        {
            let m = &b"\x46\x7e\x8e\xa6\x34\xf7\x99\x5d\xc4\x6c\x11\xb8\xab\x0b\x75\x08\x89\x46\x81\xe8\x1c\x35\x02\xc3\xb3\x35\xe8\x97\xe6\xd6\x9d\xf8\x85\xf4\x95\x57\xce\x23\x27\x84\xe3\x51\x9b\x72\x7b\xa6\x84\x3b\xd7\xaf\x50\x63\xf8\xbc\x1d\x61\x0f\x86\xce\x5b\x35\x15\x5e\x32\x5c\xe1\x75\xbe\x85\x38\x39\x5b\x34\xdf\x67\xa4\x21\xfc\xa2\x7e\x31\x33\x0b\x59\xa4\x10\x11\xb2\x90\xa5\x8b\xdc\x8e\x74\x04\x01\xb3\x8f\x55\x64\xc2\xfd\x7a\xe8\x9f\x60\x9e\xd6\x07\xd5\x78\xdb\x7f\x1c\xda\x50\x8a\xf9\x87\xbe\x1f\xd9\x46\xa2\x5a\xb9\x34\x6d"[..];
            let expected = &b"\x5d\x5c\xf1\x80\xbd\xac\x8a\xe1\x4a\x50\xc4\xe8\xb5\x7b\x88\xba\x68\xb1\xc3\x52\x7d\x34\xa6\x3f\xd3\x90\xf3\x3b\x8b\x02\x5a\x75\xf7\xce\x82\x4f\x1e\x20\xde\x81\x49\x91\x36\x4a\x97\xc2\xab\x9e\x6c\x88\xcd\x19\x6a\xbf\x6c\x4a\x0a\xc6\x60\x13\xa7\x3d\xb9\xe9\x4a\x98\x8a\x46\x65\xa3\x1c\xd4\xe0\x73\x18\x08\xc8\x8e\xc1\x45\x7c\x48\x10\x47\xf3\xe3\x2b\x08\xbf\xf6\x56\x0c\xbe\x85\xce\xc2\x12\x23\x9d\xc8\xb2\x54\x67\xde\xbe\x53\x6a\x8a\xb1\xa9\x51\xaf\xcd\xcb\x2c\xfc\x41\x1a\xfd\x5c\x18\x9f\x4b\x0e\xab\xee\xc3\xf7"[..];
            let mut hsh = Sha512::default();
            let mut sig = vec![0u8; 1024 / 8];

            hsh.update(m);
            assert!(sign_pkcs1(
                &public, &private, &mut hsh, &mut rnd, &mut sig
            )
            .is_ok());
            assert_eq!(sig, expected);
            hsh.update(m);
            assert_eq!(
                verify_pkcs1(&public, &mut hsh, expected).ok(),
                Some(true)
            );
        }
    }

    #[test]
    fn rsa_pkcs15_encrypt() {
        let mut rnd = Yarrow::default();
        let n = &b"\xc8\xa2\x06\x91\x82\x39\x4a\x2a\xb7\xc3\xf4\x19\x0c\x15\x58\x9c\x56\xa2\xd4\xbc\x42\xdc\xa6\x75\xb3\x4c\xc9\x50\xe2\x46\x63\x04\x84\x41\xe8\xaa\x59\x3b\x2b\xc5\x9e\x19\x8b\x8c\x25\x7e\x88\x21\x20\xc6\x23\x36\xe5\xcc\x74\x50\x12\xc7\xff\xb0\x63\xee\xbe\x53\xf3\xc6\x50\x4c\xba\x6c\xfe\x51\xba\xa3\xb6\xd1\x07\x4b\x2f\x39\x81\x71\xf4\xb1\x98\x2f\x4d\x65\xca\xf8\x82\xea\x4d\x56\xf3\x2a\xb5\x7d\x0c\x44\xe6\xad\x4e\x9c\xf5\x7a\x43\x39\xeb\x69\x62\x40\x6e\x35\x0c\x1b\x15\x39\x71\x83\xfb\xf1\xf0\x35\x3c\x9f\xc9\x91"[..];
        let e = &b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x01"[..];
        let d = &b"\x5d\xfc\xb1\x11\x07\x2d\x29\x56\x5b\xa1\xdb\x3e\xc4\x8f\x57\x64\x5d\x9d\x88\x04\xed\x59\x8a\x4d\x47\x02\x68\xa8\x90\x67\xa2\xc9\x21\xdf\xf2\x4b\xa2\xe3\x7a\x3c\xe8\x34\x55\x50\x00\xdc\x86\x8e\xe6\x58\x8b\x74\x93\x30\x35\x28\xb1\xb3\xa9\x4f\x0b\x71\x73\x0c\xf1\xe8\x6f\xca\x5a\xee\xdc\x3a\xfa\x16\xf6\x5c\x01\x89\xd8\x10\xdd\xcd\x81\x04\x9e\xbb\xd0\x39\x18\x68\xc5\x0e\xde\xc9\x58\xb3\xa2\xaa\xef\xf6\xa5\x75\x89\x7e\x2f\x20\xa3\xab\x54\x55\xc1\xbf\xa5\x50\x10\xac\x51\xa7\x79\x9b\x1f\xf8\x48\x36\x44\xa3\xd4\x25"[..];
        let p = &b"\xd0\x6f\xfa\x56\x3b\xd0\xb8\xb9\x7f\x00\x72\xea\x07\x51\x03\xec\x1f\xf0\xb1\x7f\xd6\xbd\xaf\x94\xa7\x5a\xe6\x03\x59\x82\xf6\x80\x24\x9d\x77\xcc\x15\x96\xa6\x54\xf1\x4c\x51\x7a\x13\xb9\x5f\x1a\x8c\x5d\xe8\xcc\x84\xad\x38\xd2\xe6\xaa\x4f\xcd\x54\x1a\x32\xeb"[..];
        let q = &b"\xf6\x6a\x24\x68\xb1\x48\x1a\x0d\x6b\x60\x49\x67\xf0\x58\x03\x89\xd6\x06\x1f\x03\x3a\x9b\x57\xb7\x4d\x1b\xef\xe8\x10\x11\x98\x94\x5b\x80\x93\x04\x24\x1d\x02\xfd\x5f\x92\x1f\xe9\x59\xfb\xec\xdb\x18\x00\x54\x86\x9a\xdb\x17\xd3\x21\xf9\xd5\xd5\x98\x12\x3e\x73"[..];
        let public = PublicKey::new(n, e).unwrap();
        let private = PrivateKey::new(d, p, q, None).unwrap();

        {
            let m = &b"Hello, World"[..];
            let mut c = vec![0u8; 1024 / 8];

            assert!(encrypt_pkcs1(&public, &mut rnd, m, &mut c).is_ok());

            let m2 = decrypt_pkcs1_insecure(&public, &private, &mut rnd, &c)
                .unwrap();
            assert_eq!(m, &m2[..]);

            let mut m3 = vec![0; m.len()];
            decrypt_pkcs1(&public, &private, &mut rnd, &c, &mut m3)
                .unwrap();
            assert_eq!(m, &m3[..]);
        }
    }

    #[test]
    fn fips_186_3() {
        use crate::hash::*;

        fn test<H>(msg: &[u8],
                   n: &[u8], e: &[u8],
                   s: &[u8])
        where
            H: Default + Hash + Pkcs1Hash,
        {
            let mut h = H::default();
            h.update(msg);

            let key = PublicKey::new(n, e).unwrap();
            assert_eq!(verify_pkcs1(&key, &mut h, s).ok(), Some(true));

            // Sanity check: Retry with the hash being reset.
            assert_eq!(verify_pkcs1(&key, &mut h, s).ok(), Some(false));
        }

        // [mod = 2048]
        let n = b"\xce\xa8\x04\x75\x32\x4c\x1d\xc8\x34\x78\x27\x81\x8d\xa5\x8b\xac\x06\x9d\x34\x19\xc6\x14\xa6\xea\x1a\xc6\xa3\xb5\x10\xdc\xd7\x2c\xc5\x16\x95\x49\x05\xe9\xfe\xf9\x08\xd4\x5e\x13\x00\x6a\xdf\x27\xd4\x67\xa7\xd8\x3c\x11\x1d\x1a\x5d\xf1\x5e\xf2\x93\x77\x1a\xef\xb9\x20\x03\x2a\x5b\xb9\x89\xf8\xe4\xf5\xe1\xb0\x50\x93\xd3\xf1\x30\xf9\x84\xc0\x7a\x77\x2a\x36\x83\xf4\xdc\x6f\xb2\x8a\x96\x81\x5b\x32\x12\x3c\xcd\xd1\x39\x54\xf1\x9d\x5b\x8b\x24\xa1\x03\xe7\x71\xa3\x4c\x32\x87\x55\xc6\x5e\xd6\x4e\x19\x24\xff\xd0\x4d\x30\xb2\x14\x2c\xc2\x62\xf6\xe0\x04\x8f\xef\x6d\xbc\x65\x2f\x21\x47\x9e\xa1\xc4\xb1\xd6\x6d\x28\xf4\xd4\x6e\xf7\x18\x5e\x39\x0c\xbf\xa2\xe0\x23\x80\x58\x2f\x31\x88\xbb\x94\xeb\xbf\x05\xd3\x14\x87\xa0\x9a\xff\x01\xfc\xbb\x4c\xd4\xbf\xd1\xf0\xa8\x33\xb3\x8c\x11\x81\x3c\x84\x36\x0b\xb5\x3c\x7d\x44\x81\x03\x1c\x40\xba\xd8\x71\x3b\xb6\xb8\x35\xcb\x08\x09\x8e\xd1\x5b\xa3\x1e\xe4\xba\x72\x8a\x8c\x8e\x10\xf7\x29\x4e\x1b\x41\x63\xb7\xae\xe5\x72\x77\xbf\xd8\x81\xa6\xf9\xd4\x3e\x02\xc6\x92\x5a\xa3\xa0\x43\xfb\x7f\xb7\x8d";
        let e = b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x26\x04\x45";
        test::<Sha224>(
            b"\x74\x23\x04\x47\xbc\xd4\x92\xf2\xf8\xa8\xc5\x94\xa0\x43\x79\x27\x16\x90\xbf\x0c\x8a\x13\xdd\xfc\x1b\x7b\x96\x41\x3e\x77\xab\x26\x64\xcb\xa1\xac\xd7\xa3\xc5\x7e\xe5\x27\x6e\x27\x41\x4f\x82\x83\xa6\xf9\x3b\x73\xbd\x39\x2b\xd5\x41\xf0\x7e\xb4\x61\xa0\x80\xbb\x66\x7e\x5f\xf0\x95\xc9\x31\x9f\x57\x5b\x38\x93\x97\x7e\x65\x8c\x6c\x00\x1c\xee\xf8\x8a\x37\xb7\x90\x2d\x4d\xb3\x1c\x3e\x34\xf3\xc1\x64\xc4\x7b\xbe\xef\xde\x3b\x94\x6b\xad\x41\x6a\x75\x2c\x2c\xaf\xce\xe9\xe4\x01\xae\x08\x88\x4e\x5b\x8a\xa8\x39\xf9\xd0\xb5",
            n,
            e,
            b"\x27\xda\x41\x04\xea\xce\x19\x91\xe0\x8b\xd8\xe7\xcf\xcc\xd9\x7e\xc4\x8b\x89\x6a\x0e\x15\x6c\xe7\xbd\xc2\x3f\xd5\x70\xaa\xa9\xa0\x0e\xd0\x15\x10\x1f\x0c\x62\x61\xc7\x37\x1c\xec\xa3\x27\xa7\x3c\x3c\xec\xfc\xf6\xb2\xd9\xed\x92\x0c\x96\x98\x04\x6e\x25\xc8\x9a\xdb\x23\x60\x88\x7d\x99\x98\x3b\xf6\x32\xf9\xe6\xeb\x0e\x5d\xf6\x07\x15\x90\x2b\x9a\xea\xa7\x4b\xf5\x02\x7a\xa2\x46\x51\x08\x91\xc7\x4a\xe3\x66\xa1\x6f\x39\x7e\x2c\x8c\xcd\xc8\xbd\x56\xaa\x10\xe0\xd0\x15\x85\xe6\x9f\x8c\x48\x56\xe7\x6b\x53\xac\xfd\x3d\x78\x2b\x81\x71\x52\x90\x08\xfa\x5e\xff\x03\x0f\x46\x95\x67\x04\xa3\xf5\xd9\x16\x73\x48\xf3\x70\x21\xfc\x27\x7c\x6c\x0a\x8f\x93\xb8\xa2\x3c\xfb\xf9\x18\x99\x0f\x98\x2a\x56\xd0\xed\x2a\xa0\x81\x61\x56\x07\x55\xad\xc0\xce\x2c\x3e\x2a\xb2\x92\x9f\x79\xbf\xc0\xb2\x4f\xf3\xe0\xff\x35\x2e\x64\x45\xd8\xa6\x17\xf1\x78\x5d\x66\xc3\x22\x95\xbb\x36\x5d\x61\xcf\xb1\x07\xe9\x99\x3b\xbd\x93\x42\x1f\x2d\x34\x4a\x86\xe4\x12\x78\x27\xfa\x0d\x0b\x25\x35\xf9\xb1\xd5\x47\xde\x12\xba\x28\x68\xac\xde\xcf\x2c\xb5\xf9\x2a\x6a\x15\x9a",
        );
        test::<Sha224>(
            b"\x9a\xf2\xc5\xa9\x19\xe5\xda\xdc\x66\x87\x99\xf3\x65\xfc\x23\xda\x62\x31\x43\x7e\xa5\x1c\xa5\x31\x46\x45\x42\x50\x43\x85\x1f\x23\xd0\x0d\x37\x04\xee\xab\xb5\xc4\x3f\x49\x67\x4a\x19\xb7\x70\x7d\xd9\xaa\x3d\x65\x7a\x04\xba\x8c\x66\x55\xc5\xab\x8b\xa2\xe3\x82\xb2\x66\x31\x08\x0c\xd7\x9e\xc4\x0e\x6a\x58\x7b\x7f\x99\x84\x0b\xd0\xe4\x32\x97\xab\x16\x90\xe4\xce\xc9\x5d\x03\x1a\x2c\xa1\x31\xe7\x04\x9c\xfb\x9b\xf1\xfc\xa6\x7b\xf3\x53\xcd\xc1\x2c\xc7\x4c\xee\xe8\x0c\x5d\x61\xda\x8f\x01\x29\xa8\xf4\xa2\x18\xab\xc3\xf6",
            n,
            e,
            b"\xc5\xdf\xbe\xfd\x35\xce\xc8\x46\xe2\xc7\xb2\x43\x4d\xc9\xc4\x6a\x5a\x9b\x1b\x6c\xe6\x5b\x2b\x18\x66\x5a\xed\xb1\x40\x4d\xe1\xf4\x66\xe0\x24\xf8\x49\xee\xc3\x08\xc2\xd2\xf2\xf0\x19\x3d\xf1\x89\x8a\x58\x1c\x9e\xa3\x25\x81\x18\x55\x53\xb1\x71\xb6\x50\x70\x82\x61\x7c\x5c\x01\x8a\xfe\x0c\x3a\xf6\x4d\x2e\xc5\xa5\x63\x79\x5a\xa5\x85\xe7\x77\x53\xcd\x18\x83\x6f\x6f\x0c\x29\x53\x5f\x62\x00\xca\x89\x99\x28\xfe\x78\xe9\x49\xb0\xa2\x16\xec\x47\xa6\xad\xf2\x22\x3e\x17\x23\x6c\xfc\x16\x7c\xf0\x0e\xd6\x13\x6f\x03\xcf\x6f\xfd\x4f\x3f\x77\x87\xae\xb0\x05\x84\x09\x78\xd8\xd6\xba\x59\x3d\x4f\x4c\xfe\x69\x20\xbe\x10\x2b\x98\x47\xd1\x01\x40\xdf\xf8\x6b\x0d\xb1\x4f\xfc\xcc\x9a\x96\xe6\x73\xc6\x72\xc1\x12\x8a\xe4\x54\x89\xd2\xcb\xfe\x6e\x19\x5c\xa5\x20\x6e\xda\x51\x9c\xad\x3d\x6e\x0a\xbf\x46\x53\xe3\x6b\x5a\x26\x4e\x87\x49\x4a\x4d\x63\xee\x91\xff\x7c\x35\xa6\xab\x12\xad\xfa\x3b\xb5\x37\xf6\x19\x8b\x06\xf5\xde\x07\x17\x07\x6b\x0e\xc8\x3a\xe0\xda\x9e\xa4\x19\xcc\x0c\x96\x66\x9d\x1d\x7c\x9e\x52\x92\x71\x42\x84\x01\xe0\x9e\x04\x88\x8a",
        );
        test::<Sha224>(
            b"\x59\xb5\xb8\x5b\x9d\xc2\x46\xd3\x0a\x3f\xc8\xa2\xde\x3c\x9d\xfa\x97\x16\x43\xb0\xc1\xf7\xc9\xe4\x0c\x9c\x87\xe4\xa1\x5b\x0c\x4e\xb6\x64\x58\x75\x60\x47\x4c\x06\xa9\xb6\x5e\xec\xe3\x8c\x91\x70\x3c\x0f\xa5\xa5\x92\x72\x8a\x03\x88\x9f\x1b\x52\xd9\x33\x09\xca\xec\xc9\x15\x78\xa9\x7b\x83\xe3\x8c\xa6\xcb\xf0\xf7\xee\x91\x03\xcd\x82\xd7\x67\x3c\xa1\x72\xf0\xda\x5e\xba\xde\xf4\xa0\x86\x05\x22\x6c\x58\x2b\x1f\x67\xd4\xb2\xd8\x96\x77\x77\xc3\x69\x85\xf9\x72\xf8\x43\xbe\x68\x8c\x67\xf2\x2b\x61\xcd\x52\x9b\xaa\x6b\x48",
            n,
            e,
            b"\x29\xb5\xac\x41\x72\x26\x44\x4b\xc8\x57\x0a\x27\x9e\x0e\x56\x1a\x4c\x39\x70\x7b\xdb\xea\x93\x60\x64\xed\x60\x3b\xa9\x68\x89\xeb\x3d\x78\x6b\x19\x99\xb5\x18\x0c\xd5\xd0\x61\x17\x88\x83\x7a\x9d\xf1\x49\x6b\xac\xea\x31\xcb\xf8\xf2\x4a\x1a\x22\x32\xd4\x15\x89\x13\xc9\x63\xf5\x06\x6a\xad\x4b\x65\xe6\x17\xd0\x90\x33\x59\x69\x6d\x75\x9d\x84\xc1\x39\x2e\x22\xc2\x46\xd5\xf5\xbe\xd4\xb8\x06\xf4\x09\x1d\x5e\x8f\x71\xa5\x13\xf1\x31\x9b\xb4\xe5\x69\x71\xcd\x3e\x16\x8c\x9a\x7e\x27\x89\x83\x22\x93\x99\x1a\x73\xd3\x02\x70\x72\xec\xee\x68\x63\x51\x45\x49\x02\x9f\xb3\x55\x34\x78\xc8\xf4\x10\x3b\xf6\x2d\x7d\xe1\xfb\x53\xfe\x76\xce\x97\x78\xad\xa3\xbb\x9e\xfa\x62\xda\x44\xcd\x00\xd0\x2b\xb0\xeb\x74\x88\xac\x24\xda\x38\x14\xc6\x53\xcb\xa6\x12\x30\x13\x73\x83\x7a\x0c\x3f\x11\x88\x54\x93\xcb\xf3\x02\x4c\x35\x72\xea\xed\x39\x6d\x0e\xbb\x80\x39\xdd\xf8\x43\xc2\x18\xd8\xbc\x77\x83\x54\x90\x46\xc3\x35\x86\xfb\x34\x28\x56\x2c\xb8\x04\x60\x90\x04\x0c\x0e\x4e\xea\x50\xa1\x9a\x42\x8b\xde\x34\x62\x62\x77\xff\x48\xa8\x4f\xaa\x18\x9b\x54\x40",
        );
        test::<Sha224>(
            b"\x49\xa5\xf3\x93\x0a\xd4\x5a\xca\x5e\x22\xca\xac\x66\x46\xf0\xbe\xde\x12\x28\x83\x8d\x49\xf8\xf2\xe0\xb2\xdd\x27\xd2\x6a\x4b\x59\x0e\x7e\xef\x0c\x58\xb9\x37\x88\x29\xbb\x14\x89\x99\x4b\xff\x38\x82\xef\x3a\x5a\xe3\xb9\x58\xc8\x82\x63\xff\x1f\xd6\x9f\xed\xb8\x23\xa8\x39\xdb\xe7\x1d\xdb\x2f\x75\x0f\x6f\x75\xe0\x59\x36\x76\x1a\x2f\x5e\x3a\x5d\xfa\x83\x7b\xca\x63\x75\x59\x51\xae\x3c\x50\xd0\x4a\x59\x66\x7f\xa6\x4f\xa9\x8b\x46\x62\xd8\x01\x15\x9f\x61\xee\xfd\x1c\x8b\xc5\xb5\x81\xf5\x00\xda\xc7\x3f\x0a\x42\x40\x07",
            n,
            e,
            b"\x60\x4e\xb6\x37\xca\x54\xbe\xa5\xad\x1f\xd3\x16\x59\x11\xf3\xba\xa2\xe0\x6c\x85\x9d\xc7\x39\x45\xa3\x8b\xca\x7f\xf9\xbf\xa9\xed\x39\x43\x53\x48\x62\x3d\x3e\x60\xf1\xce\x48\x74\x43\x84\x0c\x6b\x2c\x00\x0f\x15\x82\xe8\x52\x60\x67\xa5\xe8\x92\x3f\x1a\x1b\xda\xab\xb1\xa4\x0c\x0f\x49\xee\x69\x06\xa4\xc8\xfc\x9b\x8c\xfa\x6d\x07\xc2\xcc\x5b\xdf\x2a\xda\x65\xc5\x3d\x79\x54\x80\x89\xc5\x24\xfa\x36\x43\x19\xa9\x0d\x46\x21\x3f\xeb\xdc\xe6\xdb\x79\x59\x14\xcb\xda\x04\xd7\xbb\xbf\x26\xbb\xb2\x99\xfc\x7d\x14\x49\xdc\xc8\x1d\x13\x9e\x3c\x33\xd4\xc1\xde\x96\x47\x39\x94\x73\x0a\x4b\x63\x96\x33\xd6\x77\xdb\x25\x69\x5f\xfd\x15\x7e\x59\x1b\xdd\xea\xd0\x3d\xd2\xf1\xc1\xb8\xf5\xc8\xa2\x13\xb7\x85\x87\x9b\xf7\xc9\xa9\x92\xbb\x11\xdd\x5e\x91\xdf\x3a\xff\x09\x31\xca\x76\xc4\x06\x23\x0a\x19\xe3\x07\xf3\x34\x19\xc9\xd9\xd3\xf6\xf6\x4b\xf8\x88\x1c\x0d\xdf\x74\xa5\x71\x6c\xbc\x43\x33\x29\x36\x8d\x6e\x55\xf1\xf7\x51\xd7\xb9\xf9\xb0\xa2\x6e\xb5\x81\x17\x72\xf5\xf6\x98\x53\x0e\xfc\x1e\xac\xee\xe6\xe1\xdc\x68\x39\xb2\x13\x3c\x2f\xcc\xfa\x8c",
        );
        test::<Sha224>(
            b"\x9b\xfc\x4d\xac\x8c\x22\x32\x38\x72\x16\xa5\x32\xce\x62\xd9\x8c\x1a\xaf\xa3\x5c\x65\xdc\x38\x8e\x3d\x4d\x37\xd6\xd1\x86\xea\xe9\x57\xf8\xc9\xed\xac\x1a\x3f\x2e\x3a\xbc\xb1\x12\x1f\x99\xbd\x4f\x8c\x2b\xbf\x5b\x6a\xc3\x9a\x25\x44\xd8\xb5\x02\x61\x9f\x43\xea\x30\xdd\xc8\xe4\xea\xfa\xd8\xbf\x72\x56\x22\x03\x80\xe0\xae\x27\xfe\xe4\x63\x04\xb2\x24\xcc\x8a\x1e\x2b\x1c\xb2\xa4\xde\x6f\xb3\xee\x54\x52\x79\x8d\xe7\x86\x53\xe0\x8b\x01\xec\x38\x5f\x36\x7c\x39\x82\x96\x3f\x84\x28\x57\x27\x93\xed\x74\xce\xe3\x69\xf5\xae",
            n,
            e,
            b"\x44\x4f\x7e\xfb\xfe\xf5\x86\xfa\xd4\x31\xe1\x7f\xea\x1a\x2d\x59\xf1\x9b\x3d\x61\x9b\xb6\xfa\x36\x64\x30\x18\x33\xa4\xdb\x12\x43\x45\x9e\x31\xaa\x6a\x70\x3b\x22\x57\x2f\x09\x12\x75\x4e\x56\xf7\x23\x1a\x55\xac\x7a\xbc\xa5\x14\xc7\x9d\x9f\xb3\x56\x42\x14\xb4\xaf\x83\x5d\x7d\x1e\xaf\x2b\x58\xce\xb6\xa3\x44\xf1\xc3\x68\x90\xf5\xe8\x3b\x50\x18\x8c\x01\x47\xd6\xd1\x15\x6d\xa2\x89\xcc\xf4\xbd\xb0\xb9\xa6\x6f\x1e\x4a\x1f\x26\x43\x59\x1d\x5f\xfb\x53\x70\x2c\xf7\x0d\xdf\x35\x15\x92\x57\x54\x88\xf1\x92\x90\x10\xac\xa3\x77\x14\xb2\x34\xee\xb5\xb9\x52\xb9\x32\x3a\xe2\x65\x33\xe9\xec\xd5\x16\xdf\x26\x39\x2d\x12\x54\x22\x8b\xd9\xca\x21\xa3\x69\xbb\x6a\xb0\xa3\x3d\x5e\xb4\x4c\xee\x92\xb0\xea\x74\x71\xff\xe5\xfa\x43\xc2\x1d\xe2\xa8\x97\x5d\x4c\x5c\x8e\x18\x5f\xcb\x7a\xab\x33\xd8\x8a\x83\x65\xdd\xf0\x11\x9c\x10\x88\x03\xc5\x62\x88\x64\x3a\x05\x6e\x78\x1a\xbd\x4a\x02\x42\xa9\x2e\x25\x29\xd4\x05\xef\xcf\xd4\x24\x86\x62\xcf\xbb\x33\x2d\x6e\x6f\xad\x6a\xce\xb9\x0b\x5b\x58\xa5\x54\x1a\xbe\x07\xbe\xf2\x5d\x9d\x89\x21\x5e\x39\x84\x26",
        );
        test::<Sha224>(
            b"\xbf\x5f\xf1\x96\x8a\x39\xf8\x09\xde\x73\xe6\xa8\x01\x4f\xc6\xe8\xdf\x15\x93\x67\xf4\x63\x40\xda\x6c\xc5\xfb\x46\x89\x85\xb3\x74\x46\xc5\xd8\x9f\x3a\xca\x62\x6f\xbe\x9b\x14\x2b\x52\xcb\x02\x2a\x3d\x93\x51\x8a\x74\x24\x3e\x25\xbd\x3a\x61\xc1\x14\xf5\x33\x87\x4e\xe5\xcf\xb7\xfc\x63\xf5\x99\x92\x28\x54\xb7\xc9\x18\x09\x49\x41\x5f\x63\xf1\x6b\xbf\xe9\xa8\xa6\x28\x9e\xf8\xa8\x8a\x83\x6d\x20\xe7\x5e\x46\x99\xac\xba\x6f\xa2\x41\x2f\xb4\x2c\xdf\xe3\x2f\x33\xa2\x51\x02\xa1\xdf\x49\x4c\x6f\xb7\x38\x55\x0d\xec\xaa\x0c",
            n,
            e,
            b"\x01\x7e\x05\x3d\x1e\xf8\x5c\x43\x19\x3a\x00\x09\xa9\x03\x95\x2a\xaf\x40\x0f\xbc\xfe\xe9\xc0\x28\x97\x57\x77\xab\x54\x0d\x2d\x22\xab\x5c\x25\xf4\xcf\x1d\x37\x94\xaf\xac\x66\x97\xe1\xf2\x43\x82\x90\x52\xa8\x4e\x28\x43\xcc\x0e\x25\x4d\xba\xc1\x02\x15\x72\x99\x9f\x2d\xca\xfa\xb5\x8b\x9d\xfe\xf2\xfc\xaf\x70\x1e\x43\x1b\xdc\xd1\x6d\xbe\xf1\x10\x09\x5b\xcf\xba\x50\x10\x59\xd7\x99\x4d\xad\x5b\x0b\x54\xd0\x81\x2a\x43\x80\xa1\xf0\xba\x8e\xc2\xbc\xba\x76\x8b\xf5\xb5\x44\x69\x56\x26\xa5\xf3\x95\xe7\x84\xd4\xb2\x96\x2f\xb7\x53\x38\x18\xde\x1d\x6e\xc6\x86\xed\xc9\xf6\x68\x68\xad\x03\xee\x64\x36\x1a\x6c\xb9\x1f\xd8\xef\x53\x6c\xa6\x45\x4d\x16\xc5\x37\xc0\x7a\xa4\x29\x23\xe6\x20\x57\xdf\x9d\xd9\xe7\xfa\x4a\xd0\x38\x4f\x35\x72\x1f\x6e\xb3\xb8\x16\xd3\x52\xa0\x95\xc6\x05\xd5\xc1\x0e\x0a\x7a\x2e\x86\x40\xe2\x73\x07\xcd\x44\xb9\xd7\x1a\xc5\x0c\x00\x43\xca\xca\x28\xae\x8d\x6f\x8f\xa5\xbb\x48\x31\x58\xa4\xe4\x15\xef\x6c\xfa\xd4\x7f\x34\xc0\x04\x2a\x2d\x58\x8a\xce\x0f\x13\x71\xd9\x38\x65\x39\x7b\xd2\x15\x16\xda\x2c\xc1\x5e\x90\x9c",
        );
        test::<Sha224>(
            b"\x2f\xf4\xfc\xd0\xbe\x26\x0b\xf4\xa0\xd7\x31\x12\xd0\xe5\x64\x9c\x0b\xef\x5b\xbc\xdf\x15\x42\x3a\x05\xff\xb2\xa1\xf0\x21\xe0\x9d\xa6\x3d\x15\xa8\xcf\x29\x5e\xe5\x0b\xd2\x84\x4c\x89\x81\x3e\x08\xd6\x5d\xa6\x1d\xf2\x32\xea\x4e\xa9\x70\x44\x3e\x20\x77\x2c\xd5\xaf\x11\xcc\xe5\xee\x40\xb4\x0e\x13\x3b\xcf\xdf\x7b\xb3\x95\x3d\x86\x5a\x83\x09\xa8\xa6\xc8\xfd\xbd\xd2\x42\xd7\x9d\x27\xa8\xba\xf1\x79\x09\xd1\x45\xf4\x75\x35\x5e\x19\xfa\x11\xcd\x03\xd2\x04\xc4\xef\xda\xc6\x29\xfb\x46\x0f\xe9\x2e\x93\xb4\x8f\xb9\xbe\x13",
            n,
            e,
            b"\xab\xee\x5c\x86\x8f\x85\x0c\x17\x79\x4f\x02\x1e\xe9\x70\x9c\xc2\x30\x13\x20\xdd\x24\x6f\xb3\xea\xdb\x78\x02\xa3\x00\xa9\x8a\x67\x08\x3a\x2e\x4e\x25\x0d\xf1\x33\x14\xc2\x54\x53\xb8\x98\x11\x08\x01\xf7\xe7\xac\xb9\xb6\x94\x64\x4e\x5c\x4a\x26\x23\xdf\xf1\x88\x49\x13\xc0\x5e\x63\x6f\xe7\x7e\xd5\x15\x5d\x95\x4e\xe3\x8f\x12\x62\xc6\xc2\xe3\x8d\x11\x14\xcf\x6c\xc5\x14\x3c\x72\x77\xc8\x64\x9f\x5a\x42\x3f\x83\xdf\xd5\xf8\x29\xd9\xdc\x74\xaa\x4b\x2f\xcd\xc8\x96\x0c\xde\x5c\xe1\x46\xb2\x89\x13\x60\x64\xb1\x3b\xd0\xd3\x6a\x1e\x64\xa2\x61\xd6\x80\xfb\x7e\x23\xd2\xae\x92\xef\xb7\x43\xc3\xdb\x54\x60\x9e\xca\x7a\x1b\xe0\xe4\x7e\x6f\x72\x4d\xc5\xcf\x61\xcb\x2a\x36\x9c\x2b\xb1\x73\xf2\xc6\xcf\xec\xb9\xa8\x87\xd5\x83\xd2\x77\xb8\xe3\x0b\x24\xec\x85\x49\xc4\xd5\x3b\xa3\x98\x86\x42\xa6\x1f\x1f\x93\x9f\x0f\x38\x98\x00\x5c\x5d\x13\xaa\xaa\x54\xbc\xb8\xae\x83\xb7\x2b\x3c\xb6\x44\xb9\x43\x9d\x1d\x2a\xcc\xc8\x00\x27\x1d\x23\xe5\x2f\x98\x48\x0d\x27\x0f\xad\x6a\xce\xd5\x12\x25\x2e\xe9\x83\x32\xaf\x90\x35\x63\xd9\x82\xd8\xcb\xde\xfb\x7d",
        );
        test::<Sha224>(
            b"\xb5\xdc\xa1\x53\x2d\xff\xda\x08\x31\xcb\x2d\x21\xeb\xd1\xbd\xca\x23\xc9\x31\x9c\x64\x27\xfd\xcc\x5a\xef\xe3\xa2\x7f\xc9\xb9\x2d\xf7\x58\x6c\x36\xb7\xc8\x45\x72\xed\xa6\x6b\xfb\x9c\xf5\xaa\x01\x87\x7e\x72\xbd\x51\x67\x23\xa7\xe2\x07\x87\xe9\x0d\xf9\xa0\x13\x6f\x6f\xa5\x10\x9a\xc9\x47\x59\x73\x67\x38\x68\xd8\xbb\xee\x70\x86\xa2\xa5\x4b\x3a\xf4\xa3\xb4\x17\x59\xbf\xb6\x48\x5f\x24\x64\xe6\xca\x53\xcb\x1c\x2c\x67\x25\x89\xb5\x9d\x50\xe5\x4b\x13\x7e\xe8\xdd\xd0\x2d\x67\xf5\x05\x5a\xc1\x8d\x92\xf1\x79\x24\xcc\x89",
            n,
            e,
            b"\x9a\xe5\xb9\x63\x3f\x9a\xdc\x7f\xf9\x23\xd8\x87\x57\x48\xbc\x62\x20\xdd\x8f\x67\x81\xb3\xd4\x6d\x60\x08\xae\x69\xfd\xa0\x72\xd2\x05\xf8\x7a\x12\xd5\x4c\x3c\x7e\xcc\x85\xb8\x8b\x6e\xf4\x77\x0e\xeb\x4b\x71\xde\xbe\xff\x84\x01\xe3\x29\xf6\xb3\xe8\xdc\x8a\x9a\xf1\x3a\x53\x3b\x60\xb9\x62\x93\x0b\xc0\xce\x3d\x65\xd0\xb5\xa2\x76\xe8\x5a\x0c\x74\xf4\x59\xfb\x07\x29\x92\x99\x1b\xa9\x68\x49\x02\x34\x78\xab\x28\xd3\x81\xaa\x67\xd2\x2c\x9c\x3b\x09\x2a\x02\x3f\x06\xc9\x6e\x11\xfd\x2f\x1b\x4d\x9d\xaf\x0f\x34\x49\xde\x17\x97\x61\x2a\x81\x13\xd6\xe6\x26\xcc\x3f\x99\x5e\x1c\x11\x0e\x65\xd1\x7c\x63\x6c\x92\x92\x9f\x91\x36\x39\xa9\x7c\xd0\x49\x15\x58\x30\xdc\x0f\x76\x04\x91\x23\xbe\x3d\x3d\x79\x15\x9f\xc2\xb4\x25\x8e\x94\xb8\xbf\x80\x8d\x7c\x46\xbe\xef\xe6\xdf\x0a\x83\x03\x7d\x15\xa7\x2a\x58\x1d\x8a\xde\xdd\x8f\x01\x3b\x38\xf5\x50\x2d\x73\x6d\x1d\x2f\x04\xb0\xe5\xdc\x22\xeb\x1a\x41\x4e\x52\xb1\xa9\xe8\x73\x5e\x05\x92\x28\x8c\x9e\x5a\x0a\x78\x53\x1e\x95\x97\x4a\x5d\x48\x86\x0f\x8e\x5b\x04\xeb\xd3\xeb\x56\xad\x12\xad\xc4\x6e\xc7",
        );
        test::<Sha224>(
            b"\x1e\x56\x3f\xc3\xad\x02\x7a\x9c\xc6\x06\xbe\x19\xb2\x58\xbf\x70\xdd\x8b\x52\x73\xe2\x96\x23\x6e\xe8\xd7\xa6\x53\x31\x58\x50\x14\xf0\x50\x06\x51\x5b\xed\xd6\x33\x02\x50\xe5\x98\x5f\xda\xa8\x70\xae\xa6\x57\x66\xff\x56\x9f\xc4\x89\x13\x98\x90\x41\xcf\xf6\xfb\xab\xcd\x83\xfd\xf0\x64\xcd\x39\x32\x00\x1b\x26\x1c\x69\xa6\x70\xbd\x48\x06\x9c\x96\xe7\xeb\xec\xf1\x38\x0d\x82\x75\x19\x66\xc7\xf8\xd6\x9e\x0e\x94\xef\xc7\x75\xfd\x1c\x4a\x0c\x11\x8f\x21\x3a\xb1\x79\x47\x5c\xd0\xcf\x6d\xae\xc9\x4e\xef\x6f\xf6\xbd\x06\x40",
            n,
            e,
            b"\x80\xd3\xff\x1f\x74\xa8\x10\x95\xd0\xba\xa2\xe9\xde\x24\x8c\x03\x12\xca\x5a\x81\x7b\xc9\xf5\x15\x6a\x29\x3d\x80\x89\x6a\xde\xc5\x50\x7e\xe8\xf2\xdf\x41\x7a\xfe\x87\x79\x66\x8e\x25\xb4\x6f\x49\xe4\x35\x7a\x71\x70\x53\x1e\xd3\x07\x61\x10\x3d\xbb\x99\x41\x35\xb5\x10\xd9\x1d\xb9\xfe\x1f\x12\x68\xf4\x37\xe0\xf3\xa7\xa4\xba\x6a\x4d\x0b\x9e\xb7\x0d\xfc\x09\xfe\xd4\xb4\x4b\x35\x60\x85\x01\xc2\xdf\xd7\xa2\x30\xa2\x8d\xad\x14\x92\x6d\xa4\x60\x0b\xa7\x85\xe4\x96\x21\x2e\x57\x73\x8d\xd5\x75\xb4\x0c\x23\x34\x7b\x16\x35\xec\xdf\x2b\x91\x94\xd9\x6b\x14\x50\xa6\x87\x6a\xa7\x6d\x04\xaa\x59\x47\xcc\xe7\x1d\x85\x12\x1e\x0b\xf5\x78\xe8\x1c\xf7\x8c\x6a\x04\x7e\x30\xfc\x1d\x87\xcf\xd3\x01\x9d\xe4\xbb\x48\x29\x4c\x25\x86\x0b\x45\x03\x55\xbc\x26\x62\xaa\x36\xd6\xe3\x3f\x00\xad\x79\x25\x7d\x2d\x8b\x91\xf7\x3f\x27\xc3\x2a\x9a\xfc\xb1\xe1\xf0\x15\xf7\x7c\xb6\xb0\xdf\x51\xfb\x39\xee\x1b\xd7\x6a\xc4\x2c\x20\x79\x1d\x79\xcf\x3f\x36\x3f\xb3\x24\xdb\x30\xee\x82\xbc\xc1\xdf\x1a\x95\x64\x33\x0c\x12\xa5\x49\x65\x9b\xd3\x01\x00\x01\x57\x31\x33",
        );
        test::<Sha224>(
            b"\x90\x0a\xe7\xe2\xe7\xe5\xf6\x15\x75\x0c\x4e\xe4\xc1\x3c\xca\x8f\x9f\x45\x07\x14\xa6\xb2\x73\xf2\xe4\xac\xa6\x32\xd1\x1c\xf6\xa8\x82\x10\x45\x77\x1f\x60\x1e\xd3\x97\x91\x01\x0b\x92\xf9\xfa\xc6\xa8\x24\x78\x8c\xd0\x77\x5d\x89\x1b\x13\x52\x8e\xa2\xfd\x5d\x59\xbc\x7b\xb5\x16\x75\xc1\xd5\x26\x3c\xcc\xcf\x1e\xdc\x8f\xe3\x13\xae\x4d\x50\x15\x0c\x46\x6a\xf9\x08\x95\xed\x5c\x5e\x59\x91\xe4\xa8\x13\xde\xc9\xd1\x4f\x42\x94\xcc\x87\x61\x27\x86\x44\xac\xfe\x19\x86\x35\xb4\x42\x66\xc1\xc9\x15\xfa\x1f\xa2\xef\x79\xb9\xd1",
            n,
            e,
            b"\x39\xc6\x48\x91\xd9\xac\x47\x41\xa5\x7d\xd8\xae\xc7\xf7\x24\x36\x13\xd1\x55\xdf\x44\x92\x81\x4b\x40\xce\xab\xee\x79\xea\xdb\x8d\x8b\xc5\xfa\x61\x1b\xde\xbe\x0e\x0d\x97\x14\xc4\x3d\x6d\x29\xef\x30\x9f\x78\x2b\xc8\xe6\x8a\x4d\x31\x7c\xe1\xec\xe4\x68\x55\x23\x05\xa7\x3d\xb9\xd0\xd2\x89\x1e\x28\x04\xf4\x20\x1b\x1b\xf8\xa3\x24\x6f\xa0\x82\xad\xde\x1f\xc9\xb3\xd2\x99\xf8\x8c\xb9\x3b\x7b\x47\xfe\x9f\x73\x13\x70\x96\xc2\xb8\xc5\x9e\xc0\x61\x2a\x08\x53\x63\xc0\x4c\xc3\x74\x76\x9a\x96\x4f\xea\xf1\xf8\xe4\x91\x38\x1e\x16\xd7\xae\x2a\x0c\x67\x2e\x69\xa3\x66\x73\x10\xfe\xed\x01\x21\x56\xdc\xa6\x30\xa6\x8d\x33\x9e\xc8\x04\x96\xc6\xb5\x94\xfe\xd1\x70\x91\xd3\xa1\xc6\xac\x3e\x4d\xa1\x41\x9b\x05\xd5\x89\xcb\x32\x46\x82\x88\xf7\xdf\x4d\xaa\xce\xff\x5a\x39\xbc\xf2\x97\xdc\x50\x8c\xe9\x54\x9f\x60\x2e\x97\x3e\xdb\xc2\xaa\x44\x33\x2e\xc3\x66\x1b\x19\xc8\xc5\x8c\x56\x16\x92\x4b\xeb\x89\x2f\x77\xb5\xe2\x00\xd6\xfb\x3f\xc7\x59\x26\x3a\x74\x9d\x15\x7e\xff\x9f\x73\x67\x98\xd2\x81\xb2\x5b\x71\xfb\x47\x0b\xdb\x70\x0f\x21\x1f\x84\x1d\xb7",
        );
        test::<Sha256>(
            b"\x5a\xf2\x83\xb1\xb7\x6a\xb2\xa6\x95\xd7\x94\xc2\x3b\x35\xca\x73\x71\xfc\x77\x9e\x92\xeb\xf5\x89\xe3\x04\xc7\xf9\x23\xd8\xcf\x97\x63\x04\xc1\x98\x18\xfc\xd8\x9d\x6f\x07\xc8\xd8\xe0\x8b\xf3\x71\x06\x8b\xdf\x28\xae\x6e\xe8\x3b\x2e\x02\x32\x8a\xf8\xc0\xe2\xf9\x6e\x52\x8e\x16\xf8\x52\xf1\xfc\x54\x55\xe4\x77\x2e\x28\x8a\x68\xf1\x59\xca\x6b\xdc\xf9\x02\xb8\x58\xa1\xf9\x47\x89\xb3\x16\x38\x23\xe2\xd0\x71\x7f\xf5\x66\x89\xee\xc7\xd0\xe5\x4d\x93\xf5\x20\xd9\x6e\x1e\xb0\x45\x15\xab\xc7\x0a\xe9\x05\x78\xff\x38\xd3\x1b",
            n,
            e,
            b"\x6b\x8b\xe9\x7d\x9e\x51\x8a\x2e\xde\x74\x6f\xf4\xa7\xd9\x1a\x84\xa1\xfc\x66\x5b\x52\xf1\x54\xa9\x27\x65\x0d\xb6\xe7\x34\x8c\x69\xf8\xc8\x88\x1f\x7b\xcf\x9b\x1a\x6d\x33\x66\xee\xd3\x0c\x3a\xed\x4e\x93\xc2\x03\xc4\x3f\x55\x28\xa4\x5d\xe7\x91\x89\x57\x47\xad\xe9\xc5\xfa\x5e\xee\x81\x42\x7e\xde\xe0\x20\x82\x14\x7a\xa3\x11\x71\x2a\x6a\xd5\xfb\x17\x32\xe9\x3b\x3d\x6c\xd2\x3f\xfd\x46\xa0\xb3\xca\xf6\x2a\x8b\x69\x95\x7c\xc6\x8a\xe3\x9f\x99\x93\xc1\xa7\x79\x59\x9c\xdd\xa9\x49\xbd\xaa\xba\xbb\x77\xf2\x48\xfc\xfe\xaa\x44\x05\x9b\xe5\x45\x9f\xb9\xb8\x99\x27\x8e\x92\x95\x28\xee\x13\x0f\xac\xd5\x33\x72\xec\xbc\x42\xf3\xe8\xde\x29\x98\x42\x58\x60\x40\x64\x40\xf2\x48\xd8\x17\x43\x2d\xe6\x87\x11\x2e\x50\x4d\x73\x40\x28\xe6\xc5\x62\x0f\xa2\x82\xca\x07\x64\x70\x06\xcf\x0a\x2f\xf8\x3e\x19\xa9\x16\x55\x4c\xc6\x18\x10\xc2\xe8\x55\x30\x5d\xb4\xe5\xcf\x89\x3a\x6a\x96\x76\x73\x65\x79\x45\x56\xff\x03\x33\x59\x08\x4d\x7e\x38\xa8\x45\x6e\x68\xe2\x11\x55\xb7\x61\x51\x31\x4a\x29\x87\x5f\xee\xe0\x95\x57\x16\x1c\xbc\x65\x45\x41\xe8\x9e\x42",
        );
        test::<Sha256>(
            b"\xc4\x30\x11\xf3\xee\x88\xc9\xc9\xad\xca\xc8\xbf\x37\x22\x1a\xfa\x31\x76\x9d\x34\x7d\xec\x70\x5e\x53\xac\xa9\x89\x93\xe7\x46\x06\x59\x18\x67\xcc\xd2\x89\xba\x1b\x4f\x19\x36\x5f\x98\x3e\x0c\x57\x83\x46\xda\x76\xc5\xe2\x22\x8a\x07\xe4\xfc\x9b\x3d\x48\x07\x16\x33\x71\xa5\x2b\x68\xb6\x68\x73\x20\x1d\xc7\xd6\xb5\x66\x16\xac\x2e\x4c\xb5\x22\x12\x07\x87\xdf\x7f\x15\xa5\xe8\x76\x3a\x54\xc1\x79\xc6\x35\xd6\x58\x16\xbc\x19\x48\x5d\xe3\xeb\x35\xa5\x20\x40\x59\x10\x94\xfe\x0e\x64\x85\xa7\xe0\xc6\x0e\x38\xe7\xc6\x15\x51",
            n,
            e,
            b"\xaa\x3a\x4e\x12\xeb\x87\x59\x6c\x71\x1c\x9a\x22\xbc\xab\xcb\x9d\xad\xff\xca\xbc\xec\xbd\x16\x22\x88\x89\xe9\xbb\x45\x7d\x5d\x22\x57\x1a\x72\xf0\x34\xbe\x47\x83\x38\x4f\x43\xce\x6f\xff\xc6\x05\x34\xb8\x33\x1c\xdd\x5d\x7c\x77\xf4\x91\x80\xbf\xd1\x94\xb5\xfd\x43\xa5\x08\xc6\x6d\x78\x6c\x55\x88\x76\x73\x58\x94\xe6\xa9\x30\x09\x52\xde\x79\x2f\x74\x70\x45\xe7\x4d\x87\xfd\x50\x98\x02\x30\x70\x7a\x34\xa4\xdf\x01\x3c\xe0\x50\xbb\xff\x0d\x6f\x57\x08\x85\xc9\xc7\xbf\x8d\xc4\x99\x13\x2c\xae\xe0\x71\xb4\x1d\x81\xff\x91\xb8\xce\x21\xaa\x2f\x28\x2c\xbf\x52\x38\x9f\x23\x9a\xfe\x14\x90\x89\x0b\xe2\x1f\x9d\x80\x8b\x3d\x70\xb9\x7e\xfd\x59\xc0\xb6\x0e\x46\x60\x88\xbb\x42\x71\x4f\x21\x2b\xc9\x0d\xb7\xe9\x42\xeb\xce\xe6\x0e\x7b\x10\x7f\xff\x44\xfb\x35\x64\xff\x07\xd6\xd0\x28\x50\x21\x5f\xd3\x57\xd8\x97\xc4\xd3\x2b\xef\x86\x61\x68\x9f\x2d\x84\xff\x89\x76\x37\xfb\x6d\x55\x68\xa7\x27\x0e\x78\x34\x26\xb7\x4b\x70\x37\x49\x3e\x51\x55\xfd\x7c\xb3\xdd\xdd\xfd\x36\xbd\x8a\x9c\x87\x7d\x71\xd2\xa9\x66\x05\x7c\x08\x26\x3d\x29\x39\xc8\x49\x87",
        );
        test::<Sha256>(
            b"\x61\xd7\xb3\x15\x01\x31\x35\x1e\x7b\x4c\x8e\x56\x45\xd3\x8b\xe9\x33\x5b\x40\x28\x9a\xf3\x4c\xc6\xb6\xfc\x5e\x48\x49\x3b\xf8\xb7\x85\x2c\x73\x98\x2c\x99\x44\x1e\xf6\x6c\x7d\x9d\x33\xc2\x97\x42\xb1\x40\x6e\x02\xe0\xaa\x8d\xd0\x34\xb1\xac\x13\xcb\x0d\x77\x57\x50\xcc\x91\x42\x1f\xea\xd9\xca\xa9\x21\xec\xa6\x1a\x02\xeb\x02\x3a\x45\x7e\x77\x91\x5e\x18\x3a\xcf\x51\x7d\x94\x6b\xc6\x82\x92\x89\x60\x14\xfd\x21\x4b\x7c\x8c\x5e\x14\xe1\x59\x44\xbe\x0f\x92\x96\x12\x77\x71\xf7\x36\x76\x6e\x4f\x81\xda\xb3\x70\x8e\xa2\xd0",
            n,
            e,
            b"\x84\xe9\x2a\x14\x5a\xe6\xbe\x1f\xf9\x24\x2d\x9e\xd2\xd6\x8d\xe6\x68\xe8\x02\x52\x4e\x8a\xc0\xa7\x9d\xe6\x2f\xe7\x40\x48\xc3\x54\x91\xfd\x2f\xfd\xb1\x85\x05\x7e\x66\x6d\xbf\xaa\xc8\x4c\x34\xfd\xe7\x89\x12\x63\xf8\xb2\xbc\x74\x74\x62\x30\x32\x0f\x67\xa7\xbd\x73\x19\xc9\xb9\xde\x41\x90\x54\x70\x14\xe2\xd7\xa2\xa5\x06\x0d\x62\x00\xaa\xdc\x3a\x44\xba\xc0\x29\xff\x39\x92\xed\xd3\x0e\xc5\x3a\xb0\xd9\x12\x3e\xaa\x6b\x14\x73\x52\xa0\x73\xa9\x81\x61\xe6\x4f\x39\x4b\xb9\x94\x92\xc6\x97\x7e\x24\xf4\x45\xc7\x12\x5b\xfb\x90\xf8\x7f\xaf\x26\x22\x72\x13\x4a\xcb\x18\x82\x3a\x99\xa5\x22\x8d\x14\x95\x46\x32\x97\xfd\x77\x48\x77\xfb\x63\xd4\x91\x81\x06\x34\x7e\x6f\x29\x31\x5e\x48\x36\x3f\x39\xb3\x32\x99\xea\xa3\x2d\x8d\xa7\x1b\x22\x9d\x8f\xfe\xe5\xf6\x6f\x72\x2a\xd3\xaa\x41\x75\xd3\xf8\x4e\xce\x9c\xc8\xec\xa8\xd6\xf2\xf3\x56\xa8\x5c\x15\x24\x89\x6c\x18\xf7\xb5\xc8\xf9\xbc\xde\xf4\x5c\x49\x6d\x53\x91\x79\x89\x1d\xdc\x76\xe5\x20\x8a\xd8\x35\x3d\x48\xc6\x24\x05\x4f\x34\x40\xee\xba\x44\x32\xa1\x06\x54\xa1\x1e\xf5\x37\x83\xbd\x11\x6f",
        );
        test::<Sha256>(
            b"\xb6\x77\x1a\xb0\xe1\x28\xb4\x1b\x32\xb8\xb0\x5e\x05\xad\xd2\x3c\xe0\xfb\x87\x7b\x40\xbf\xcc\x3b\x99\x2f\x4c\x86\x98\xd1\xc8\x28\xab\xec\xbc\xc1\xc3\x3d\x40\x18\x59\xea\x2c\xb2\xaf\xbc\x7f\xa4\x58\x88\x02\xa5\xfa\xee\x28\x67\x53\x46\x39\x28\x7a\xd8\xaf\x84\x67\x4b\xe1\x8d\xb6\x61\xde\x1d\xa8\xe1\x9c\x6b\x6b\xd4\x52\xdd\x9b\xf3\x22\x1d\x08\x61\xfb\x6f\xba\x96\xbe\x42\x32\x9b\x9f\x04\xf3\x7d\xcf\x3b\x41\xfc\x58\xd2\x29\x83\x48\xb0\xc1\x5d\x11\x90\xb1\x25\x30\x0c\xf2\x7e\x0d\xfa\xd6\x05\x22\xfc\x49\x84\x60\x53",
            n,
            e,
            b"\x62\x76\x92\x55\x68\x62\x6f\x0c\xbe\x6f\x51\x50\xb0\x50\xe1\x70\x25\x82\xf8\xda\xf9\x9a\x6f\x88\x0e\xf7\x5c\xd9\x6c\x2d\x42\x08\xfb\x6e\x91\xb0\x1b\xa6\xab\xa2\xa8\x16\xb2\xd3\xcb\x97\x5d\xf8\x50\xb1\xd2\x68\xc4\x66\x2d\xd1\xea\x3a\x30\x0c\x1d\x71\x71\xc6\x33\xdd\x2e\xfb\xac\x30\x00\xc5\x6a\xb8\x0f\x98\x9d\xbc\x18\x24\x3e\x63\x6b\xa5\xd4\xd2\x6a\x7d\x3f\x19\x65\xad\x3c\xb0\xf1\xa8\x51\x3f\x99\x80\x03\xf7\xb6\x7e\x2a\xc5\xc7\x18\xcb\x68\x8b\x32\x01\xd5\x6e\x68\xf0\xb9\xf8\x62\x57\xb8\x47\x94\xcd\xff\xbc\x1f\xe3\xea\x24\xb7\xbb\x6e\x9e\xf0\x53\x9b\xd4\xfb\xc1\xaf\xb5\x5b\xc1\xdc\xa3\x99\x96\xea\x8a\x63\x76\x9f\x6e\x22\x57\x07\xf6\x90\x47\x55\x5e\x1a\x4e\xf3\xc6\x39\xc5\xf2\xa4\x97\xb8\x89\x42\x4a\x90\x14\x86\x39\xbb\x64\xdf\x0a\x06\xe0\xb7\xf0\xe8\xed\x46\x6a\x97\x7b\xac\xa3\x2f\x48\x23\x37\xb2\xab\xe3\x98\x3e\xae\xc3\xfe\x10\x75\x01\x6e\x58\x67\x52\x17\x60\xfd\x06\x07\xd7\x99\xf1\x76\x6b\x3f\xf6\xe2\xae\x15\x5d\x69\x25\x0f\x8b\xf0\x8c\x8e\xdc\xa0\xb4\xf3\x1d\x0f\x83\x8c\xfd\x29\x8c\xb7\x31\x2d\xf9\x3f\x09\x97",
        );
        test::<Sha256>(
            b"\x6a\x81\xcb\x6c\x7b\x26\x8f\x4b\x9f\xb9\x17\x2a\xdb\xbb\x36\xa2\x37\xa0\xdc\xf1\xc3\xc8\x3a\x95\xdc\xb0\x27\x1a\xac\x6a\xc3\x30\xf0\x4a\x5a\x00\xfe\xe3\x8b\xc0\x06\x31\xa9\x85\x98\x18\x61\x59\x66\x0d\x9d\x8e\x4c\x14\xa9\x52\x8d\xea\x94\x83\x60\x83\xda\xc4\xab\xb7\x3f\xd0\x0e\x38\xfe\x0e\x23\xc7\x23\x66\x04\xa7\x36\x54\x0e\x52\x19\x3a\xe5\x6c\x33\xfb\xb8\xf5\xcf\xc5\xc7\xc2\xbe\x2e\x22\x2e\x44\x83\xb3\x0d\x32\x5c\x7e\xe1\x4f\x74\x28\x51\xfc\xb8\xb6\xd6\x18\x9e\x98\xb8\x22\xb8\xe6\x39\x9d\x89\xe9\x0f\xb9\x97",
            n,
            e,
            b"\xb6\x79\x91\x05\x0c\x08\x3e\x64\x50\x97\xdb\x03\xff\xf3\x47\x58\x86\x8b\xeb\x19\xe9\xc0\xc4\x84\x75\xf0\xf9\x13\x36\x1e\x71\xd3\xd6\xf2\x7a\x8c\x4f\x0b\x26\x9b\x49\xe8\x53\x40\x39\xe5\x3a\xd3\xba\xb9\xa3\xe6\x2a\xbe\x07\x8e\xe7\x5e\x7f\xb5\x95\x90\x06\xfb\xfb\x01\x4c\xa7\xb8\x1b\x3d\x5a\xfe\x0e\xe5\xf6\xfc\x2d\xfb\xc4\x50\xf2\x83\x95\x43\x00\x2f\x33\xf4\xf3\x54\xf8\x27\x27\x8c\x76\xc0\x41\x68\x6e\xea\x78\x86\xeb\xb2\xa7\xaf\xa5\x99\x5c\x6c\xdd\xb1\xc0\xb5\x80\x66\xdd\xb8\xdc\x54\xa6\x92\x7c\x14\x6c\x3b\x2a\x0f\xa7\xce\xf2\x89\x03\xc6\xc6\x72\xbc\x20\xef\x68\xff\xbf\xab\x24\x7e\xb6\x88\xab\x4b\xde\x71\x06\xd9\xc5\x9d\x21\x53\x09\x6d\xc9\xe5\x20\x72\x67\x03\x8d\x88\xe2\x17\x4e\x76\xad\xc1\x50\x8a\xe2\x4e\xb6\x02\x33\x2e\x53\xc0\xc2\xe3\x31\x54\xa6\x6a\x97\xa0\xf1\x2f\x66\xc6\x12\x58\xc7\xbf\x6b\xbf\x3f\x1d\xcb\xe9\xca\xf2\xfd\x30\xec\x68\xc0\xa9\xd0\x9f\x4f\xd7\x76\x30\x4b\x54\x0e\x62\xfc\x85\x12\xbe\xaa\xbc\x4b\xe2\x10\x7a\x1e\xc1\x8e\x87\xf6\x1f\x9d\xb2\x5e\x87\x1d\xc0\x69\x3c\xef\x17\xc2\xa6\x87\xfc\x85\x4f",
        );
        test::<Sha256>(
            b"\x05\x6c\x1e\x46\x44\x59\x9e\x31\x83\xdd\x8d\x2f\x64\xe4\xbb\x23\x52\xff\x00\xd0\x12\xab\x76\x3f\x9a\xd6\xe5\x60\x27\x9f\x7f\xf3\x8a\x5e\xce\xa9\xc2\xe4\xea\x87\xd0\x04\xef\x8c\xc7\x52\xae\x93\x23\x2a\xa3\x7b\x5b\xf4\x28\x84\xba\xa7\xe7\xfc\x6a\x8c\x95\x1c\xd2\x45\xde\x2d\x22\x0d\x9b\xee\x2b\x41\x4b\x3a\x75\x20\xc1\xe6\x8b\xcf\x1a\xe9\x9a\x9f\xf2\xbf\x3a\x93\xd8\x0f\x8c\x1d\xfe\x8b\x85\x29\x35\x17\x89\x5c\x19\x2e\x3c\x9e\x89\x82\x95\xd6\x5b\xe3\x34\xf4\x4d\x62\xf5\x35\x3e\xb6\xc5\xa2\x9e\xdf\xb4\xdb\x23\x09",
            n,
            e,
            b"\xae\x05\x20\x4e\x40\x9d\x72\x7e\xb9\xe4\xdc\x24\xbe\x8f\x86\x33\x28\xc2\x81\x3d\xa4\xfc\xef\x28\x86\x6e\x21\xa5\xda\xb2\x1a\x48\x53\x21\xb7\x35\x27\x4a\xf0\x6b\xf1\x7e\x27\x15\x18\xe1\x11\x64\xd7\x22\xab\x07\x35\x48\xf0\x2e\x1b\x44\x19\x23\xdb\x6f\x1c\xee\x65\xa0\x17\xed\xfb\xaf\x33\x61\xc6\x7f\xbc\x2b\x39\xfe\x03\x8c\xb5\xcb\x65\xa6\x40\xf9\x58\x87\x38\x9c\xe8\xa5\xad\x2e\xc6\xe6\x9d\x3d\x60\x35\x05\xb0\x25\xf6\xd6\x33\x0c\x8b\x64\x88\x02\xca\xf7\xe6\xfa\x3f\xe7\xb3\x81\x41\x65\x99\x86\xcb\x89\xe6\x23\x2f\x10\x62\x22\x56\x4d\x5e\x51\x95\xed\xa6\xa2\x5f\x99\x06\x85\x72\xc2\xfa\xfe\x97\xf1\x47\xf7\xf2\xf4\x11\x9f\x21\x38\x5a\xf1\xfc\xed\x97\xf7\x86\x32\xd8\xbf\x4f\xd9\xa9\x05\x4d\x8b\x9a\xa2\xa9\xf4\xde\xd5\x87\x84\x7a\x91\xd4\x2c\x63\x91\x12\x5f\x10\x3a\xe2\x88\x54\x7e\x84\x89\x69\x3a\xe8\x68\x6b\x84\x89\x1b\x77\x2b\x10\xc4\x79\x68\x83\xf6\x6c\xd4\x59\xa8\xc1\xa6\xa4\x18\x7b\xd6\xb3\x87\xd3\x49\xe9\x2d\x7b\x60\x49\x53\x72\x7c\x9e\x9f\xdc\x44\x9e\x73\x45\xe7\xca\x6b\x33\x9e\x26\xb0\x86\xf5\x54\x88\x98\xcb\xe9",
        );
        test::<Sha256>(
            b"\xce\xc5\xc9\xb6\xf8\x44\x97\xac\x32\x7f\x68\xef\x88\x66\x41\xfe\xc9\x95\x17\x8b\x30\x71\x92\x30\x43\x74\x11\x5e\xfc\xc5\xee\x96\x27\x0c\x03\xdb\x0b\x84\x6d\x67\x4c\x52\x8f\x9d\x10\x15\x5a\x3f\x61\xbe\xcc\xe1\xd3\xa2\xb7\x9d\x66\xcd\xc4\x09\xad\x99\xb7\x66\x30\x80\xf5\x1a\x10\x2f\x43\x61\xe9\xdb\xd0\x3f\xfc\xd8\x76\xb9\x8e\x68\x3d\x44\x8b\xd1\x21\x7e\x6f\xb2\x15\x1c\x66\x96\x47\x23\xb2\xca\xa6\x5c\x4e\x6c\xa2\x01\xd1\xc5\x32\xbd\x94\xd9\x1c\xd4\x17\x3b\x71\x9d\xa1\x26\x56\x39\x27\xca\x0a\x7f\x6f\xe4\x25\x36",
            n,
            e,
            b"\xc4\x8a\x8e\x01\xd4\xbb\xfe\x0f\x2f\x05\x65\x93\x37\xea\x71\xd2\x1f\x38\xd7\xf7\xa1\x0b\x00\xb0\x6e\x1f\x89\x9e\xaf\x40\xa8\xe9\x7e\xad\x64\xbc\xa3\x7f\x13\xa5\x5e\xf1\xcf\x3f\xb5\x2c\xee\x27\x9c\xdc\xb0\x96\x08\x5a\x46\x7a\xfa\x97\xb0\x3d\x78\xd6\x07\x6e\x47\x2b\x12\xd6\xbe\x96\x47\xce\xc3\x2d\x8d\x91\xa2\x62\x47\x69\x37\x71\x68\x74\x60\xba\x52\x69\xde\x18\xe1\xed\xef\x60\x22\x53\x3a\x95\x79\xf9\x1d\x58\x4f\x9e\x0c\xee\x11\x00\xc4\x47\xb7\x75\x76\xb1\xb4\xee\x16\x3e\xd4\x70\x01\x47\xa9\xaa\x61\xbd\xc4\xe2\x31\x6d\x2d\x81\x8c\x10\x28\xed\x1c\x3e\x37\x2c\x9f\x6a\x17\x45\x57\x24\x44\x63\x72\x48\x09\x1b\x83\xf7\xb5\x39\xf9\xbd\x58\xb7\x67\x56\x76\x03\x4c\x20\xe4\xca\x11\x9b\x91\xc4\xca\x5d\xc7\x6a\xcb\xff\x3d\x04\x62\x89\x83\x52\xc5\x91\xc2\xca\x6f\x2d\x8b\x09\xe2\xe6\x33\x8a\x84\x33\x6e\x06\xf0\xcc\x02\x0e\x9e\xb8\xda\x78\x58\x89\xb4\x97\xf3\xb9\x8e\x82\x7e\xe7\xa7\xd3\xf1\xb0\xb7\x3c\x19\x58\xe1\x6a\xa9\x78\x61\xe6\x67\x59\x70\xce\x31\xd9\xd1\x19\xbb\x34\x0b\xe8\x0f\xd0\xf4\x3c\x3d\xbe\x64\xf2\xa5\x9d\x62\x9d",
        );
        test::<Sha256>(
            b"\x91\x93\xf8\xb9\x14\xdf\xe0\xe6\x25\x21\xf3\x5a\xfa\x4f\xa5\xd4\x28\x35\xe1\x98\xaf\x67\x38\x09\x37\x7a\x3e\x7a\x99\x73\x31\x42\xa1\x80\xdc\x0e\x13\xe6\xbb\x7c\xeb\x3b\x60\xe5\xe9\xd5\x15\x79\x4d\x82\xc3\x92\xe0\x79\x13\x42\x33\x91\xd2\x2e\x2b\xb1\x9a\xa0\xbd\x88\xaf\xd7\xf7\x7e\x27\xa2\x40\xea\x4e\x2d\xe0\x85\x48\x1a\xc3\x1f\xf8\xd3\x79\x90\x21\x1f\x82\xf2\xcb\xf4\xc9\x0d\xe9\x8d\x6e\x13\x38\xbb\xc8\x8e\x6a\x80\xab\x96\x84\xda\xe6\x47\x85\xdd\x10\x72\x48\x04\x85\x93\xab\xc9\xab\x03\xf1\x73\x7a\x6f\x65\x30",
            n,
            e,
            b"\x5c\x2f\xe4\x53\xa8\xb0\x8c\x90\xb0\x2e\xb2\xc9\x99\x42\x42\xd5\x18\xf3\xf2\x1b\x36\x88\x95\xcf\xfd\x62\x40\x50\xe4\x8a\xa7\x14\x00\x5a\xe6\x75\xfe\x79\xaa\x3c\xad\xd4\xdf\x55\xbd\xf1\x2b\xec\x5b\xe8\xa4\x1d\x87\x53\x8f\x7e\x03\x1b\x78\x2e\x34\xd3\x92\x46\x8e\x5f\x14\xbc\x61\x3b\x8f\x4d\x28\xc8\xfb\x79\xa2\x53\x7e\x1e\x60\x10\x31\xda\x72\x0a\xcd\x7b\x2c\x8d\xcb\xe9\x85\x86\x24\xa7\xa9\xa9\x2a\x06\xf9\x18\x45\xf7\x32\x37\x0d\x67\x36\x5c\x64\x64\xf7\xb6\x8f\x22\xeb\x3e\xdf\xee\xc9\x7e\x32\x85\x02\x4d\x7f\x69\x43\xb6\xd5\x0a\x16\xcc\x96\xd6\x0f\x68\x03\x51\xde\xaa\x25\xf0\xbc\x86\x89\x48\x60\x7a\x6b\xa7\xf1\x94\x9b\x85\x94\x3c\x6a\x92\xbd\x61\x72\xe8\x1b\xcc\x05\x50\x14\xb7\x8a\x73\x39\x72\xe3\xf3\x9d\x14\x09\x9d\x16\x07\xa2\x0f\xf8\x68\x1c\x29\xae\x1e\xf9\x9e\xf1\x15\xed\x6a\x10\x84\xb5\x14\xb8\x1a\x69\xd4\xa1\x5c\xe1\xe2\x57\x6f\xdc\xf2\xb2\xaf\x61\x5b\x52\xfe\xc7\x01\x32\x11\x2d\xcc\x5b\xc1\x9e\xc1\x7f\x32\x28\x14\x60\x62\x34\x20\x31\x73\x53\xe8\xa2\x55\xfd\xa5\x02\xbd\x1f\xb1\x1a\x58\x83\x2a\xe2\xc0\x4f\x9a",
        );
        test::<Sha256>(
            b"\x0e\x57\xef\x40\xb0\x21\xbf\x87\xf6\x42\xc5\x75\x6b\x65\x15\xa0\xe0\x6c\x15\xa0\x18\x56\xd7\x16\xc5\x66\xa6\xed\xb3\x81\xdf\xdf\x44\xd9\x03\x3b\x1c\xc8\x09\xe6\x1d\xfe\xf9\xa0\x96\xdf\xb6\x89\xb7\x27\x1b\xe4\x49\xd0\x4a\x1a\x9c\x35\x41\x02\xc0\x77\xaf\x5f\xf7\x20\x05\xab\x6b\x06\xcf\x13\x1d\x73\x45\xc2\x1e\x82\x1d\x62\x01\xcc\xa4\xe0\x90\x44\x0d\x70\xbe\x60\x09\xd2\xdd\x7a\x98\xd3\x11\x75\x1e\x16\x05\xa3\xb9\x14\xdc\xe6\xd2\x62\x6b\x16\xf2\x33\xa5\xa3\xd7\x1d\x56\x7c\xc8\x20\x15\x2f\x25\xe4\x73\x51\x42\x42",
            n,
            e,
            b"\x76\x43\xaa\x3f\xe6\x3e\x66\xf7\x9d\x6b\x40\x9d\x14\x5e\xa8\x20\xc9\xf7\x35\x6f\x71\xb4\xac\xdc\xbd\x43\xfe\x1e\x99\xf8\x80\x2c\xd1\x66\x2b\x16\x24\x0f\x5c\xfd\x94\xa7\x69\xb0\xb3\xf2\xcb\x0b\x11\x88\x7e\x88\x6e\x5b\xa4\x37\x33\x36\x74\x90\xb3\xfc\x18\x8f\x2f\xb3\xa0\xc0\xc8\xa6\x8b\x5d\x27\x26\xc8\xf7\xa3\x19\x02\xb6\xb8\x6c\xd4\x02\x28\x7d\x38\x5c\x3e\x3c\x06\x50\x3c\xe1\x7f\xd6\xe5\x4e\x58\x2f\x4a\x90\x7a\x91\xf9\x52\xd2\xa3\x60\xe2\xfb\xa0\x00\x28\xe4\xd3\xb0\x2a\xab\xf7\xd2\x20\xb3\x1d\x1f\x8e\xe7\xfa\xa0\x70\x14\x76\x82\xcc\xc8\xbc\xc7\x56\xca\x6a\x68\xfc\x20\x95\x45\x50\xc3\x17\xe8\x79\x18\x78\x1a\x3d\x1f\x19\x23\x50\x30\x91\x09\x0c\x3c\x60\xca\x1c\x0b\x1c\x69\x99\x06\xfb\xf8\x5a\xa7\x0a\xd9\xae\x48\x70\x9f\xf7\x43\xb8\x2d\xcc\x31\x07\x4c\xfc\xea\x62\x3e\xa4\x5e\x48\x64\x4b\x19\xa2\x17\x72\xca\x10\x7e\xd6\x42\x39\xc5\x65\x74\xa0\x87\xf1\xa6\xaa\xdf\x0f\x4b\x00\xff\xe5\x81\xc1\x41\x02\x74\xc8\x75\xe4\x59\x90\x63\xe4\x6e\x51\x68\x80\x3f\x0d\x28\xd2\x1f\xcd\x35\x09\xb4\xc6\x22\x29\x95\xad\xd7\x75\x3b\xf3",
        );
        test::<Sha256>(
            b"\x0c\x84\x91\xfc\x34\x8d\x34\x1f\xe8\x5c\x46\xa5\x61\x15\xf2\x60\x35\xc5\x9e\x6a\x2b\xe7\x65\xc4\x4e\x2e\xc8\x3d\x40\x7e\xa0\x96\xd1\x3b\x57\xe3\xd0\xc7\x58\x34\x22\x46\xc4\x75\x10\xa5\x67\x93\xe5\xda\xea\xe1\xb9\x6d\x4a\xb9\x88\x37\x89\x66\x87\x6a\xa3\x41\xb7\xd1\xc3\x1b\xba\x59\xb7\xdb\xe6\xd1\xa1\x68\x98\xee\xf0\xca\xca\x92\x8f\x8c\xe8\x4d\x5c\x64\xe0\x25\xdc\x16\x79\x92\x2d\x95\xe5\xcd\x3c\x6b\x99\x4a\x38\x5c\x5c\x83\x46\x46\x9e\xf8\x76\x4c\x0c\x74\xf5\x33\x61\x91\x85\x0c\x7f\x7e\x2b\x14\xbe\x00\x27\xd8",
            n,
            e,
            b"\xca\xcc\x8d\x9f\x5e\xcd\x34\xc1\x43\x48\x84\x61\x13\x5c\x49\x51\x67\x61\x45\xc6\xe4\x72\xb9\x2f\x12\xf7\x58\x04\x6f\x17\x21\x42\xfa\x38\x8f\x28\x5f\x3f\xff\x06\x82\x42\x02\x88\x29\x04\x7e\x24\x80\x59\xed\x4f\xd3\x9d\x2c\x5a\xde\x46\x9d\xc7\xc3\x93\x45\xe5\x11\x49\x50\xd2\x03\x1c\xc7\x46\x5f\xe7\x12\xc4\x04\x1d\x05\xc7\x56\xd3\xf2\xd8\x8a\x46\xce\xb9\x9f\x2e\x24\xa5\x2e\x95\x8a\x03\xcd\x25\x19\xa9\xb1\x37\xe6\x2d\x5c\xa2\xb3\x53\xf7\xb0\x47\xb6\x25\xc3\x60\x23\x13\xfd\xb5\x3c\x8d\xb2\x3d\x83\x95\x1a\x59\x9d\xb3\x28\xfe\xdc\x4a\xe0\x6d\xa8\x9c\xe7\xf5\x62\x59\xb5\xc8\x22\x2f\x7b\xd3\xd9\x74\x04\x78\xfd\x28\xe5\x81\x0d\xb7\x8a\xee\x86\x23\xfd\xd3\x9f\x60\x3f\x8d\xdf\x98\x08\x1d\x78\x73\x98\x0c\x4e\xb0\xe2\x2a\x9c\xd4\x08\xf7\xc4\x13\x4c\x12\xd2\x04\x9a\x2d\x12\x0f\x4b\x62\xe6\xb3\x82\xb9\x97\xfc\x37\x5e\xf7\xac\x95\x5f\xcf\x80\xb0\x45\xc3\xd6\x38\x5f\xf4\x22\xda\xd3\x50\xc6\x88\x70\x53\x90\x68\xa1\x62\xa2\xed\xbb\x93\xce\xef\xed\x96\x77\x93\x9b\x90\xbd\x3d\xfa\x0d\xc0\x53\x46\x0b\x4e\x23\x32\xef\xa6\x92\x17\x9a",
        );
        test::<Sha384>(
            b"\x6c\xd5\x9f\xdd\x3e\xfd\x89\x3d\x09\x1a\xfd\xc3\x15\x5d\x35\x4f\x10\xd6\xd8\x81\x67\x42\x7a\x2c\xf7\x24\x62\x07\xe5\x17\x91\xa6\xca\x62\x00\xa9\x14\xcd\x28\x34\xa9\xb3\xc7\x9f\xcd\x59\xe2\x6e\x45\x7e\x06\x83\xbc\x33\xd4\x92\x67\xed\xbd\xd6\xe5\xd9\x09\x02\x69\x6f\x1e\x7b\x1a\x4a\xff\xc4\xba\x37\x13\x39\x86\x8c\x28\x01\x5e\xbb\xb7\x3e\x26\x26\x69\x86\x6c\x35\xdb\x97\x4b\xa6\x9e\x46\x8f\x25\x83\xb9\x19\x1d\x15\xd6\x86\xcd\x66\xfb\x0b\x9e\x0f\xf0\xa3\xb4\x72\x1a\x6d\xc3\x42\xf1\x4f\x24\x46\xb4\xe0\x28\x59\x5b",
            n,
            e,
            b"\x39\x74\x90\x0b\xec\x3f\xcb\x08\x1f\x0e\x5a\x29\x9a\xdf\x30\xd0\x87\xaa\xba\xa6\x33\x91\x14\x10\xe8\x7a\x49\x79\xbb\xe3\xfa\x80\xc3\xab\xcf\x22\x16\x86\x39\x9a\x49\xbc\x2f\x1e\x5a\xc4\x0c\x35\xdf\x17\x00\xe4\xb9\xcb\x7c\x80\x5a\x89\x66\x46\x57\x3f\x4a\x57\x0a\x97\x04\xd2\xa2\xe6\xba\xee\x4b\x43\xd9\x16\x90\x68\x84\xad\x3c\xf2\x83\x52\x9e\xa2\x65\xe8\xfc\xb5\xcc\x1b\xdf\x7b\x7d\xee\x85\x94\x1e\x4b\x4f\xb2\x5c\x1f\xc7\xb9\x51\xfb\x12\x9a\xb3\x93\xcb\x06\x9b\xe2\x71\xc1\xd9\x54\xda\x3c\x43\x67\x43\x09\xf1\xd2\x12\x82\x6f\xab\xb8\xe8\x12\xde\x2d\x53\xd1\x25\x97\xde\x04\x0d\x32\xcb\x28\xc9\xf8\x13\x15\x9c\xb1\x8c\x1b\x51\xf7\xa8\x74\xcb\xf2\x29\xcc\x22\x2c\xae\xb9\x8e\x35\xec\x5e\x4b\xf5\xc5\xe2\x2c\xc8\x52\x86\x31\xf1\x51\x17\xe8\xc2\xbe\x6e\xac\x91\xf4\x07\x0e\xec\xdd\x07\xec\xc6\xdb\x6c\x46\xea\xa6\x5f\x47\x2f\x20\x06\x98\x8e\xfe\xf0\xb5\x1c\x53\x8c\x6e\x04\xd7\x51\x9c\x8e\x3d\xa4\xb1\x72\xb1\xe2\x76\x10\x89\xed\x3a\xd1\x19\x79\x92\xef\x37\xc1\x68\xdc\x88\x1c\x8b\x5f\x8b\xbf\xee\x91\x9f\x7c\x7a\xfd\x25\xb8\xfc",
        );
        test::<Sha384>(
            b"\xac\xb3\x0b\xe9\x09\x2b\x2f\x18\xf2\x59\x34\xa0\xd6\x78\xb6\xbc\xd6\xb6\x7c\x2b\x88\xe7\x58\x84\xf4\x7b\x4f\xca\xe3\xad\xfa\x40\x5a\xfe\x2c\x7e\x61\xe2\xd6\xc5\x08\xb9\x27\x90\xac\x00\xf7\x6b\x77\xc9\x65\x08\x26\x68\xbf\x90\x0f\x70\xa3\x37\x62\xde\x64\x13\xaf\x93\xaf\x2e\xa8\x08\x6f\xda\x29\x3d\xed\x44\x75\xf2\x3c\x4c\xc3\x1a\xd4\x94\xf9\x8d\x7d\xd7\xb7\xfd\x6f\x7d\x97\x2b\xb7\x6c\xb3\x5a\xdc\x20\x68\x04\xc3\xfe\x5a\xcd\xd0\xe5\xb8\xb5\x4e\x07\xc2\x91\x11\xf7\x88\xbc\x59\x02\xf4\x0a\xfa\xc3\x0a\xfd\xba\xf2",
            n,
            e,
            b"\xb5\xc6\x0d\x8d\xa9\xb3\x94\x38\x78\xcb\x23\x59\xcf\x65\xe4\x81\x7c\x07\x94\xf9\x50\x45\x3c\xa7\x7c\x81\xa5\xa1\xc1\x58\x55\x91\xaa\x50\xa6\x74\x68\xe3\xb3\x99\xe4\xfa\xf1\xd6\x06\xbe\xa0\xd9\xe6\xcc\x1d\x2d\x70\xdb\x80\x63\x73\x9e\x0c\x27\xd3\xdc\x9f\x9a\xfe\x88\xde\xa5\x2e\x73\x29\x8a\x07\xd0\x5c\x7d\x97\x07\x00\x2e\xfa\x53\x7c\x38\x9e\x38\xbd\x37\xbc\xa7\x4e\xb0\xaf\x62\x61\xa5\xda\x06\x13\x62\x02\xc8\xad\x48\x7e\xeb\xd5\x0b\xef\x74\x76\x70\x89\xc7\x08\x70\xbe\x1d\x8f\xab\x91\x56\xf9\xfd\xbc\x2f\x2e\x9c\xc3\x30\xa9\x50\x18\xce\x79\x43\x98\x4b\xec\xc2\x56\x21\xbf\xa6\x60\x18\xef\x83\x20\xb6\x00\x59\xf9\x41\x15\x6e\x9c\xdd\x87\xff\x0d\x82\xcf\x7b\xe7\x74\x65\xe0\x20\x3e\x71\x20\xaa\xec\xed\x84\xab\xd8\x18\x69\x47\xd4\xac\x3d\xaf\x3f\x99\x39\x02\xae\xc4\x7c\x30\x90\x47\x5c\x85\x7b\x5d\x35\x9f\x0a\x55\x72\xd4\x68\x8e\x5a\x76\xa4\x65\x38\x68\xff\x54\xce\x9f\x99\x9e\x6b\xb5\x59\xd1\xc1\x1c\x67\xc1\x5b\xe9\xd7\xfe\x5f\x8c\x17\x04\x30\x1d\x05\x5f\x3d\x29\x07\x72\x27\x79\xd6\x01\x20\x36\x08\x4e\x95\x0d\xe3\x6f\x4f",
        );
        test::<Sha384>(
            b"\x60\x1a\x6a\xad\x3f\xaa\x79\x88\xd5\xae\x52\x8a\x69\x69\x03\x1b\x10\xa6\xf3\x92\x16\x94\x6a\xa8\x9f\xd4\x53\x2c\x8e\xd1\x41\xf9\xa6\x50\xb1\x26\xef\x48\x8f\x7c\x5c\xf3\xfb\x2d\xaa\x25\x4c\xc2\x8b\xdd\x55\x56\x04\x19\xe8\x02\x14\xef\x99\x98\x96\xda\xc4\x94\x68\x52\xd2\x4f\xcd\x9f\xb7\x76\x10\xee\xbf\xbb\x6b\xa5\x8b\xca\x26\xf4\x56\x7f\x03\xac\x7e\x56\xda\x55\x3f\x23\x81\x7b\xc1\x03\xee\x48\x55\x92\xa0\x58\xfb\x5e\x3b\xc8\x29\x9c\x72\x90\xc7\x1a\x29\x13\x7e\x75\xdb\xf5\x32\x8c\x3a\x2d\xcd\x34\x16\x5b\x3f\x2e",
            n,
            e,
            b"\x30\x1d\x60\xd5\x65\x76\xf3\x66\x3a\x7f\xbe\x80\x36\xbb\xe4\xfb\xc0\xfb\xd8\x2c\xd6\xa4\x2e\x36\xd7\xbb\xc8\xb2\x06\x54\x3d\xc2\xd5\x6d\x31\x98\xe7\x91\x1a\xd1\x38\xca\xd2\x22\xdd\x99\x05\x0d\xd1\xf8\x5f\xe1\x9c\x8a\x88\xbf\x67\x13\x5e\x7f\x8f\x11\xb5\xf5\xe4\x85\xc9\x1f\xc7\xd4\x78\x06\x9b\x72\xf4\x6e\xbc\xdc\xf2\xd2\xae\x7d\xe6\xac\x8f\xe5\x3b\xb6\xc0\x49\x11\xd1\x22\xcc\x23\x1d\xc2\x10\xb2\x14\x7e\xbe\x8b\x05\x2e\x8b\x2c\xcc\x09\xf3\x38\xb3\x49\xde\x20\x25\xcc\x87\xb2\x61\x9a\x7b\x16\x33\x47\xca\x66\xa3\x47\x91\xa2\xe4\x6b\x4e\x2a\xc5\x7e\xb9\xf6\x02\x9c\xdb\xe0\x24\xe8\x96\xd5\x7f\x7d\x04\x91\xf7\x78\x33\x12\xf8\xf0\x6c\x79\x07\x70\x15\x0c\xd1\x39\xf6\x1f\xd2\xb3\xe7\x04\x1b\x37\x26\x1c\x6e\x7e\xa8\x6d\x4e\x06\xd9\x30\x0b\x1a\x56\x67\xcb\x02\x88\xc5\x50\xb2\xaf\xb3\x55\x94\x48\x34\xb4\x61\xce\xad\x13\x79\x42\x76\xbb\x46\xe5\xe2\x0a\xec\x7b\x63\xaa\xca\x4d\x49\x1a\x50\x0f\xac\xd5\x9a\x37\xc5\x27\x79\xcf\x46\x7d\x74\xaf\x1e\x62\xb1\xeb\xe0\xfd\x0b\xe1\xca\xcb\x7c\xe6\xd0\x50\xd8\x6e\x4e\xb7\x6c\xde\x06\x93",
        );
        test::<Sha384>(
            b"\x44\xd3\xe0\xfc\x90\x10\x0a\x1c\x93\x16\x06\x3f\x26\xb1\x80\x32\x6c\xc2\xe3\x83\x4c\xe5\x6e\x43\x24\x52\x8a\x0b\xbb\x01\x5b\x3d\x78\x12\x95\x8c\xd2\x6b\x91\xbf\x08\xa3\xa0\xb1\x12\x1f\x9f\x9d\xd7\x7a\xcb\x98\xa0\x2a\xd7\x5f\xcd\x61\x3c\x53\xc7\x32\xd1\xc2\x35\xf5\x9b\x68\x73\xec\xe6\x36\x3f\x27\x94\x52\xb6\xa4\xb6\x5e\x80\xbb\x59\xfd\x47\xb9\xa2\x93\x6d\xcc\x1e\x4d\xfe\x1f\x53\x62\xe3\x45\x9b\x98\x59\xdb\x32\x09\xa2\x69\x8d\x27\xfa\x8a\xed\xfe\xcd\x4d\x35\xb9\x27\xda\xf8\x68\x6c\x59\xd7\x00\x49\x0f\x0a\xa3",
            n,
            e,
            b"\xaf\x22\x29\xe9\x4a\x85\x7b\x89\xe0\xe8\x90\xda\xca\x3a\x8f\xe1\x2e\xbd\xba\x04\x94\x8d\x18\x83\xa7\xd7\x81\x6a\x3b\x68\x2f\x7d\xa3\x03\x25\x40\xa8\x76\x9f\x9c\xca\xc9\x58\x6c\xf2\x4e\x8c\x20\x4b\x45\xb8\x5d\x1b\xdc\xc5\xa5\x45\x0a\x21\x5b\x40\x48\xea\x42\x98\x3b\x34\x56\xfa\x8c\x76\xc6\x78\x6e\x02\x4f\x70\x5e\x08\x8d\x69\x45\x59\xd6\x68\xca\xa8\x68\x4c\xad\x0f\xc5\x78\x50\xfc\xaf\x34\xe4\x58\xae\xe8\xfa\xd4\xe0\x9e\x6f\x19\x65\x57\xd4\xe8\x86\x02\x84\xd9\x82\xc0\x10\x5d\x98\xce\x49\x12\xe9\x6c\x35\x50\xe2\xa0\xc7\xe8\xba\xd5\xab\xc2\x9a\x9a\x54\x2f\x57\xa8\xc6\x05\x79\x03\x80\x67\xb3\xd5\x39\x1a\xbc\x21\xb4\xf9\xde\xb0\x24\xca\x58\xf9\xb0\xc3\x8c\x0d\x1f\x82\x37\x3f\x52\x8e\x93\x9b\xd7\x3a\x24\xd5\x01\xc5\x91\x16\x88\x14\xc8\x72\xc5\x25\xdb\x0e\x56\xca\xe4\x7d\xf0\x0f\xa3\x72\x8d\xc3\xa0\x97\x69\x65\x32\x3c\xe8\xd2\xde\xe2\xb1\x38\xb5\x0a\xb7\xaf\xd4\x84\x95\x11\x46\x73\xe9\x1b\xb3\xed\x22\x05\xe2\x6a\x84\x55\x47\x4c\x3d\x4e\xc8\x73\x9b\xbf\xf6\xdf\x39\xb2\xb7\x2e\xe0\x50\x41\x09\x30\x42\x3b\x14\x72\xb6\xed",
        );
        test::<Sha384>(
            b"\x5a\xf0\x90\x77\xa1\xf5\x34\xb8\x98\x22\xb2\x6c\x32\x72\xad\xf8\x50\x0d\x3c\x6b\xd9\x0f\x9b\x5e\x0d\x8b\x21\x1f\x16\xd0\x72\x0e\xe0\xea\xf6\x46\x2b\x6c\x8a\x80\xdf\x6d\x75\x35\x9f\xd1\x9d\x03\xa0\xca\xfb\x52\xbc\x9d\x4c\x37\xc2\xaa\x09\x99\x11\xa7\x9a\x92\x65\x2c\xc7\x17\xf0\x74\x6f\xdc\xad\x62\x7c\x72\xf1\xc2\x16\xb2\x43\xd2\x17\x5f\x6d\x00\xbf\x07\xd3\xf6\xaa\x2a\x04\xd4\xfe\x9f\x8f\xbc\xe9\x32\x18\x94\x4b\x92\xaa\x07\xaf\x6b\x4f\xcd\x80\xcf\xde\x2d\x7a\xda\x15\xc0\x5e\x96\xe7\x77\xea\x1c\x17\xdf\x08\xfc",
            n,
            e,
            b"\xa5\x68\x23\xfa\x57\x7e\x89\x46\xf1\xd2\xf6\xe3\x51\xb7\x38\xb5\x35\x92\x54\x43\x58\x52\x8a\xf8\x88\x07\xea\x4f\x19\x01\x7d\xfe\x81\xa3\xd6\x9f\x62\xfb\xff\x64\x95\x50\xd9\xb3\x10\xfa\xf2\x7a\x04\x1f\xe6\x24\xf0\xa0\x2b\xdc\xdd\xb7\x9b\xfb\x0a\x46\x57\x39\xec\x8b\x64\xb7\x48\xcc\x29\xe5\xa0\x2c\x77\x7e\x18\x26\xd3\xe2\xf1\xee\xe6\xfe\x2e\xde\xe4\xa8\xbc\xac\x51\x9c\x7c\x7c\xa5\xc0\x39\xe7\x6d\x63\x06\x68\x94\x5a\x1e\x5e\x86\x18\xe2\x35\x86\x45\x61\xa4\x40\xe7\x3e\x39\xf6\xd6\x84\x2a\xd7\xda\x64\xef\x5b\x0c\xe1\xc4\xab\x88\xdb\x15\x7b\x68\x10\x71\x74\xad\x7d\x5c\x9a\x60\x65\x06\x87\x68\xc1\x1c\x4c\x96\xff\x67\x05\x0b\x5d\x07\xb8\xcd\x02\x7f\xcd\x0d\x34\x7e\xc7\x9a\x19\x7c\xf4\x34\x35\x98\x5b\xc1\xae\xb4\x79\xdb\x00\x22\x28\x9e\x8d\xd3\xb3\x1b\xb7\xc6\x2d\x88\x31\xcf\xe6\x95\x2f\x41\xd2\x4f\x89\xd7\x53\x78\x95\x35\xf9\x18\xff\x68\xb3\x69\x50\xaf\x6f\xd3\x1d\xee\x1a\xc4\x76\xa0\xcf\x93\xaf\xe9\xf4\xa7\x66\xf3\xc4\xd2\xc0\xc3\xf9\x28\x25\xd5\x57\x2e\xb2\xeb\x8a\x2b\x64\x4e\x32\x9e\xea\x16\x83\xf9\x08\x10\xed\x77",
        );
        test::<Sha384>(
            b"\xf6\x0a\x3a\x54\x37\x68\xfa\xbe\x37\xf0\x03\x00\x9a\x8c\x26\xf7\xdc\x91\xf1\x42\x2d\x44\x29\xed\x7f\x9d\x74\x4c\xdd\x4b\x55\x2a\xfe\xf7\x5d\x24\x1a\xcd\xa0\x4f\xfc\x39\x67\x21\x59\xee\x24\x8e\x60\x2d\xab\x71\x92\x44\x9e\x2e\xd4\x55\x29\x95\xc2\x58\xf0\x0a\x47\x63\x46\xe3\x6a\x29\xa0\x12\x6b\xc2\x49\x04\x0f\xaa\x57\xc9\x38\x0b\xdd\x74\xb8\x3f\x62\xc5\x67\x90\x92\x05\x74\x43\x34\x32\xf8\xd6\x5c\x5c\xd1\x85\xe2\x4f\xad\x13\x12\x72\x65\xc6\xa5\xef\x8d\xb4\xf1\x14\x49\x3d\x5c\xfa\x61\xd9\x16\x64\x98\x14\x08\xe9",
            n,
            e,
            b"\x08\xd3\x96\x48\x1d\xee\xf1\x8c\xb0\xbe\xf7\xc3\xe8\x26\xfe\x6e\x5c\x9e\xcc\x85\xe5\x23\x0d\x35\xd6\x67\x72\xb8\xd2\xd0\x15\xd4\xe5\xf5\x79\x4f\xbe\x05\x50\xdf\x2f\x74\x57\x30\xd6\xf8\xd1\xd3\xb8\x50\xd1\x64\xfc\xe4\x63\x08\x05\xe7\x11\xb5\x93\x08\xf8\x60\x85\x06\xb7\xe0\x1e\x8e\x92\x94\xed\x8b\x7e\x75\x82\x16\x56\x77\xf1\x80\xe9\x65\x16\x9d\xca\x81\xb3\xda\xf2\x4d\x7b\x92\xfe\x32\xd6\xa9\xac\x63\x82\x1d\x48\xb1\xa0\xa1\x44\xfc\x7a\x04\xb0\xbf\xc6\x3a\x3b\xc1\x6a\x0f\xd8\x37\xb0\x20\x37\xed\x76\xe5\x0d\x46\xcb\xfa\x38\x57\xe6\x58\xe3\x70\xc5\x86\xab\x1e\xed\x82\x50\x76\x32\x1a\xc8\xe8\x2b\xe3\x74\xba\xcb\x29\x5e\x4d\x34\x08\xf0\xcc\x1f\xc4\xc3\x00\xb8\x42\x75\xa5\x1c\x35\x73\xe9\xca\xbf\xdb\xe3\xdc\x51\xe4\xa6\xf5\x81\x1d\x86\x0d\x72\x5a\xaf\x8f\xd0\xaf\x19\xa2\x43\x7b\x0f\x1c\x80\xf5\xac\x22\x2f\x6b\x25\xf1\xfa\x09\xe9\x33\x99\xa6\x97\x6b\x1b\x3c\xa7\x6a\xfe\x60\x86\xe9\xb2\x32\xaa\xe6\xc7\xb8\x18\x25\x5b\xf9\x63\xf3\x1c\x04\xae\x3f\xa2\x13\x6c\x0a\x44\x29\x97\xd4\xcf\x12\xf3\x95\xfb\x80\x4a\x47\x55\xb5\x6b",
        );
        test::<Sha384>(
            b"\x2c\x07\xa8\x1d\xe5\x89\x55\xb6\x76\xfe\xc0\x57\x2d\x48\xd1\x95\x5b\x48\x75\xff\x62\xa4\x4b\x00\x10\xc7\xa1\x07\x2b\x29\x9e\xe4\x4d\xd0\xc0\x76\xf2\x17\x8a\x83\xd0\xae\x76\xe7\x67\xe2\x31\xf1\xd8\x1e\x07\x0a\xfa\xb2\x9c\x97\xab\xd4\xde\x21\x64\xe4\x37\xb3\x11\xf5\x07\x84\x1f\x88\x51\xd6\xd6\x9a\xb5\x1e\xe9\xe2\x9e\x65\x4b\x54\xbc\xee\x45\xe9\xb5\x19\xc6\xa2\x17\x87\xfa\xcb\x92\x7f\x1d\x7d\x64\x91\x92\x66\x14\x79\x2f\xcc\x63\x46\xdc\xd0\x80\xbb\x5c\xf0\x7b\xf5\x6a\xd0\xfc\x4e\x08\x3a\x35\x82\x14\x63\x15\x10",
            n,
            e,
            b"\x9a\xa3\x91\xe7\xc2\xf0\xe9\x20\xaa\xc2\x7e\xd9\xfc\x20\x81\xd3\xc9\xca\xa3\x73\x58\x83\xd0\x1a\xd7\xa7\xe3\xb1\x18\x67\xd0\xad\x62\x41\x56\x47\x7b\xbb\xdd\xe6\x59\xf4\x74\x68\x2d\x0d\x77\x44\x89\xe2\xb5\xb0\x39\xd1\xeb\x35\x45\x4c\x9e\x3e\xed\x78\xcf\xf9\xc4\x26\x2e\x3a\xec\xfc\xa1\xd8\x17\x54\x2b\x48\x60\x96\x59\x8e\x11\x14\xbf\xc0\x3f\x20\xa4\x5d\xe3\x6f\x6d\xf7\x0d\x14\x4d\x01\xdc\x48\x66\xa0\xf8\x33\x19\xe7\xc2\xb8\x53\x0f\x8c\x27\xa4\x1b\x7a\xdd\x9f\x69\x2d\x8a\x8e\x64\x64\x55\xb6\x7c\x9e\xc4\x7a\x4d\x2c\xe3\xdf\xe3\x5d\x6a\x2e\x89\xd9\xbe\x50\xc5\xb6\xda\x39\xbb\x02\x54\xbd\x23\xa8\x09\xab\x97\xb2\xb4\x8a\x06\x8a\x87\xab\xde\x6b\x6a\x6e\x35\x95\x5f\xc9\x2a\x96\x26\xf9\x60\x7d\x5b\x3f\x40\x15\x17\x27\x15\x94\xbe\xf7\x38\x59\x81\x2b\x6a\x62\x1e\xd6\xbd\xaf\x3c\x5f\x2a\x90\xb1\xe1\x68\x0f\x68\xdc\xfc\xca\xcb\x65\xe0\x08\x1f\x1c\xcb\x6a\x20\x73\x70\x9d\x1b\xa0\x67\x06\x50\x16\xed\x73\xeb\xd7\xeb\xe9\xe7\xa7\xb6\x0c\x8c\x9d\xd0\x4a\x56\xfa\xb3\x07\x02\xc8\xa6\xdf\x6a\x35\x3a\x30\x10\x47\xdf\x4c\x7a\xff\x62",
        );
        test::<Sha384>(
            b"\x35\xec\x92\xaf\xdb\xc2\xfc\xef\xe4\x8f\x1e\x2f\x6e\x48\x29\xae\x53\xb3\xda\x04\x59\xcc\x4e\xa8\xa9\x68\x18\xb5\x83\x18\x91\xee\x2f\x50\x6f\xff\x37\xc8\x99\x06\xd3\x23\x3a\x51\xa5\xcf\x14\x69\xa6\x2c\x18\x50\x61\xf0\x33\x08\x5f\xca\x6a\x54\xe2\x45\x29\xc3\xd6\xf0\xd8\xe9\x04\xbc\xb0\xf0\x89\xa5\xcd\x50\x86\x94\x84\xda\x1a\x84\xf6\xfb\x8d\xe4\xe5\x3f\xce\x3d\xc7\x14\x20\x15\x19\xd1\x10\x13\xf6\xf6\xaa\x64\xe8\xb5\xec\x5c\xfe\xb2\x7b\x61\x1f\x08\x95\x05\x9d\x8c\x47\x72\x0d\x55\xe0\x0b\x57\x7c\xa5\x50\x09\x20",
            n,
            e,
            b"\x6b\x0f\x5b\x50\xe6\x78\xda\x08\x3e\xd0\xf1\xb6\x4e\x94\x3e\x8c\x62\x79\xc7\x24\x6a\xf5\xad\x07\x9c\xdb\xf2\x23\xe4\x2a\x0d\x47\x1e\x56\x31\x4b\xc0\xd5\x8f\x20\x2a\xa6\xc5\xe1\xe5\x25\x59\x85\xb0\x79\x5d\x48\xeb\x3d\x4b\x8e\x3f\xc9\x22\x40\xae\x02\xb4\x08\x8c\x6c\xe8\xab\x0e\x8c\x79\xc6\x8d\xfd\xc4\x86\x57\xd6\xa2\x82\x95\x39\x1b\x9a\x5a\x5f\x35\x25\x51\x26\xbf\x8c\xa5\x3c\xbc\xc0\x08\x2e\xab\x52\xec\x10\x9d\x22\xa1\x18\x5f\x6d\xc7\x92\xfc\x29\x0a\xa8\xdb\xae\xbb\x2f\xbe\x40\x4f\x1d\x03\x9a\xa6\x34\x3c\xd7\xaf\x9f\xcb\x2d\x1e\x05\xde\xf4\x80\x96\xc2\x37\xe1\x0d\xaa\x7c\xfa\xc5\xae\x9b\x3b\x30\x22\x00\x5d\x0d\x2d\x5c\x9c\x5c\x50\x2b\x2f\x23\x59\x4e\x80\xd1\x60\x4b\xbb\x8f\x5d\xec\x07\xcd\x3a\xfe\x1f\x77\x77\x43\xb0\xb5\x8a\x4e\x0e\x4e\x5c\xaa\x14\x88\x30\xee\xe0\x47\x96\x8e\x7f\x40\x66\x1f\x9f\x1a\x02\xe1\xa7\xfd\x2b\x6c\xaf\x19\x32\x6a\x75\xe9\x56\x5e\xfd\xc0\x11\x4b\xce\xcb\x14\xdd\xa0\x6c\x32\x9c\xf3\x22\xa5\xbd\x3e\x6a\xb4\x8d\x95\xf2\xd2\xa9\xc1\xc1\x23\x3a\x0a\xa0\x15\xa7\x38\xf9\x01\xf1\x31\x48\xb4\x54",
        );
        test::<Sha384>(
            b"\x80\xc9\xde\xbd\xf9\x31\x74\xd7\x57\x50\xa6\xcf\x09\xaf\x71\xfc\x18\xfd\x51\x3b\xff\x9c\xb4\x91\xbe\x60\xaf\x11\x2a\x93\xf0\x00\x87\x3c\xf4\x38\x58\xa0\x7a\xca\x76\x0a\x37\xe7\x60\xc8\xcb\x01\xd2\x76\xf4\x2d\x99\x7f\x01\xcc\xa5\xe0\x8a\x6a\x60\x2f\x5f\xe6\x3e\xdc\xbe\xd3\x95\xb8\xc9\x1f\xb0\xb3\x36\xf2\x1f\xea\x49\xd9\x50\xe1\xff\x24\x64\x0c\x8d\x8d\x3b\x95\x08\x1a\xd1\x59\x66\x44\xce\x34\xa5\x58\x58\x7e\x4a\x1e\x2c\xd5\x0d\xb9\xed\x1d\xd3\xce\xbb\xc6\xdc\xe8\x08\x4d\x3e\x1b\xa7\x06\x92\xe8\x26\x18\xed\x61",
            n,
            e,
            b"\x4a\x15\xa7\x83\xad\xbf\x27\x46\x22\xd5\xa6\x10\xbb\x6f\xc7\x33\x37\x99\x9e\x44\x5d\xc2\x13\x3a\xcc\xb7\x88\xd6\x20\x3d\x70\xf3\xcd\xc6\x3e\x67\xda\xa4\x17\x1a\x79\x52\xa4\x98\x64\x56\xfa\xb3\xc0\x77\xa8\x94\x1f\xb2\x59\xe3\x7a\x5c\x0c\xbb\x20\xc4\x08\xfa\x24\xad\x0e\xc8\x50\xe9\xbf\x02\x8c\x36\x04\x60\x99\x41\xf5\xae\x2f\x18\xbf\x1a\xc3\x7a\x24\xf7\x55\xab\xb9\xc8\x5d\xdc\xd0\xbf\x4a\x12\xfa\xbd\x9d\x25\x30\x29\xe0\x81\xf6\x28\xe2\xbb\xe9\xf9\xaf\xe9\x22\x49\x54\xd8\x31\x5d\xb8\x6c\x21\x25\x51\x2b\xb9\x8c\xe9\xb3\x69\x30\x99\x4b\x09\x1a\x8a\x1d\x7d\x4e\x2f\x4a\x0e\x58\xd0\xa3\x58\x76\xad\xad\x14\x30\x05\x30\xb3\x9c\x8d\xc1\x1d\xed\x3e\xf2\xfa\x95\xd5\xf2\x2e\x67\xca\xe3\x4c\xc2\x1a\xd5\xe2\x3f\x91\x22\xb5\x3d\xfb\x79\xf1\xa2\xac\x63\xc1\x84\x4e\x9e\xf0\x69\xa2\xe4\x1f\x17\x8d\x6d\xce\xdc\x51\x8a\xaf\xcf\x81\xe0\xeb\xd8\x82\x55\x6e\x73\x1c\xb0\xab\x41\xd9\x57\x27\x4a\x3f\xbb\xb7\xce\xf2\x60\x87\x91\x00\x0c\x6b\x86\x08\x68\xcb\x73\x93\xe7\xd0\x3d\x94\x56\x89\xff\xb7\x75\x55\xef\xe0\x8f\x46\x14\x51\xd3\x3c\x11",
        );
        test::<Sha384>(
            b"\x31\x39\x5c\xef\x34\x95\x51\x34\x3a\x49\x27\x1a\x8d\x81\x2b\x4c\x7b\x65\xb4\x55\xb7\xed\xa8\x11\xfc\xf7\x41\x61\xf3\x97\x11\x23\x57\xae\x44\x62\x57\xbe\x26\xc9\x3c\xfc\xe5\x5e\x4b\xa7\x97\x6d\xed\x99\x7e\xc1\x0d\x1c\x8b\x1a\xc2\xfe\x22\xdc\x2e\xe8\x1d\x05\xa6\xeb\x13\x61\x12\x5c\xda\x01\x97\xe2\x4a\xe9\x74\xcd\x44\x09\x2a\xa9\xf3\x6f\xe0\x13\x52\xba\x05\xcc\xef\xd2\x37\x0c\xee\xd6\x64\x19\x50\x56\x2f\x17\x76\xc3\x95\x22\xe0\x23\xd0\x9a\x3b\x09\x7b\xbe\x9b\xc5\xf8\x7d\x05\xd8\x0f\x88\x30\xab\xd7\xac\x8c\x80",
            n,
            e,
            b"\x16\x2f\x38\x76\x95\xcf\x9d\x82\xdd\xa8\x9c\x74\x93\x18\xe4\x6c\x9b\xe8\x95\xec\x36\x4e\xa4\xae\xce\x97\xcc\xfa\x63\x92\x5a\xf3\x71\x08\x94\xda\x2b\x7b\x59\x67\xe4\x6f\x4e\xfa\x80\xca\x25\xd2\xa9\x65\xa7\xe1\x5f\x75\xe0\xaa\x1b\xd4\x25\x0f\x8f\x41\x09\x9e\x6e\x97\x14\xc3\xfc\x43\x11\x07\x7a\xe9\xbd\xdf\xe3\x5b\xa4\x72\x75\x31\x52\x9c\x23\x9d\x54\x6a\xb1\xc2\x98\x18\x7f\x16\x5f\x70\x8c\xcc\x0a\xe3\x97\x9a\x8d\xa1\x93\xe3\x48\x59\xa5\x9c\x2c\x3b\xc4\x22\x53\xc8\x34\x66\x88\xe6\xbb\xa6\xfb\x1b\x01\xb1\x0c\x1e\xc2\xc6\x49\x3d\xed\xcc\x26\x96\x26\x9d\x85\x1b\xde\x63\xe2\x7e\x37\xbe\xd3\x57\x45\x5c\x8f\xee\x56\x29\xf9\x4a\xfa\x7a\x98\x66\x95\xcf\xd5\xb9\x92\x12\x65\x7a\x6c\x88\x46\x44\x59\x60\x86\xb8\x9e\x0c\x7c\x05\xe8\x19\xfa\xeb\xeb\xef\x74\x5f\xd2\x95\xaf\x88\x66\xe0\x75\x0f\x54\x79\xba\xed\x50\xcb\xb3\xd0\x59\xf8\xa5\xeb\x7e\x0e\x61\xe2\x73\x3a\xe5\x0f\x0c\x1e\xc4\x2b\xe7\x1f\x5d\xff\x32\x41\x95\xcb\x4f\x0e\x94\x1a\x21\x56\x15\x13\xc3\x03\x7d\xb9\x2f\xec\x95\x56\xb7\x72\xcc\xab\x23\x9e\x34\xb1\x87\x6c\x56\xb1",
        );
        test::<Sha512>(
            b"\xa7\xc3\x09\xd4\x4a\x57\x18\x8b\xbd\x7b\x72\x6b\x98\xb9\x8c\xe1\x25\x82\x22\x8e\x14\x15\x86\x48\x70\xa2\x39\x61\xd2\xaf\xb8\x2c\xd5\xbc\x98\xbe\xc9\x22\xd5\xf2\xac\x41\x68\xb0\x56\xda\x17\x6e\xf3\xba\x91\xf6\xb6\x99\xba\x6a\xcc\x41\x44\x86\x8f\xf3\x7f\x26\xfd\x06\x72\x08\x68\xd1\x2a\xd2\x6e\xcb\x52\x57\x2c\xf1\x04\x16\xaf\x68\xdf\x03\xab\x64\x5a\x8b\x70\x48\x57\xd2\x19\x0f\xfc\x3f\x07\xea\xbe\x3a\x8e\x2a\xbe\x34\xed\x61\x59\xe8\x84\xc4\xfa\xe1\x41\xd4\x33\x3d\x5c\x3e\x0d\xb0\x44\xff\x9c\xcc\xd9\xcb\xd6\x7f",
            n,
            e,
            b"\x14\x8a\xf6\x1e\xd5\xea\x8a\x87\xa0\x8b\x3f\x40\x39\x29\xbf\x80\x31\xdb\x4f\xd3\x99\x9b\x64\x40\x9b\xa4\x89\xf9\x7a\x3e\xe5\x20\x8e\xa4\x20\x2d\x2e\xc1\x87\x34\xf6\x15\x00\x3a\x51\xf7\x74\x41\x08\x5b\xe6\xac\x0f\x11\x81\x0f\xfa\x2d\xad\x58\xf0\xe1\x86\xd5\x52\x0a\xc2\xb8\xa5\xd3\x96\x6e\x8d\x2a\xbb\x80\x74\xe1\x3b\x50\xa4\xe7\xde\x83\xbe\x10\xa6\x6f\xdc\x7c\xa1\x81\x18\xc5\x77\x4f\x78\x12\x12\xde\x9e\xfe\xbc\x63\x76\xfc\xdd\xdc\x65\xa3\xb1\xb8\xf1\xab\x31\x49\x2f\xe4\x78\x25\x9c\xe7\x19\xb3\xdb\x58\x74\x98\xd8\x79\xa0\x1d\xec\x96\xe8\xea\xbe\xb0\x7f\xf7\x07\x3f\x3f\x3e\xb4\x46\x08\x49\x55\xca\x26\x32\x9a\x79\x13\x15\xa2\xc2\x59\xd2\x25\xe2\x6b\x21\x54\xb2\x04\x7b\x21\xfa\xba\x68\x11\x5b\xfd\x96\x2e\x5e\x24\xec\x52\xd7\xc5\xd2\x31\xe3\x04\x4c\xbc\xd8\xc8\x80\x48\x55\x70\x3c\xba\xa6\x22\xb1\x5b\x6e\xf7\x8c\x74\x21\xa3\x67\x16\x6f\x1b\x02\x57\x6c\x87\x36\x05\x93\xda\x75\xb7\x18\x9e\xfa\xfd\x10\x82\xbd\x59\xf6\x85\x7f\x17\x01\xf6\x46\xc2\x4d\x70\xc9\x52\x73\xc4\x9d\x5b\x11\xe6\xaf\xe2\x58\x82\x1b\x55\xc1\x68\x0c",
        );
        test::<Sha512>(
            b"\xca\x50\x5d\x45\x91\x12\x16\x64\x99\x07\x47\xd9\x5d\x95\x55\xcc\x75\xbf\xc3\xfd\xae\xec\xee\xaa\x60\xea\xfa\xb3\xfc\x32\x0c\xfc\xe5\x6e\xb9\x13\x81\x38\xbf\x13\x8f\x25\xf3\xc8\xbb\x02\x7b\x13\x6f\x5d\x3d\x90\xed\x48\x97\x77\x9b\x59\x51\xc0\x9d\xf5\xd0\x8b\xa9\xce\x8c\xbe\x17\xab\xc4\xf0\x38\x68\x70\x86\xe9\x3d\x77\x1b\x68\x43\x22\x26\x66\x33\xd0\xd6\x5d\x71\xec\x41\x23\x4a\x1d\xbe\xc0\x7a\xbc\x8f\x7d\xf2\x8b\xc4\x3d\xd8\xa4\x5b\x10\xce\xaf\xac\x06\x77\x58\x05\x41\x37\x01\x91\x4e\x3b\xb3\x7e\xb6\xba\x5b\x5e",
            n,
            e,
            b"\x58\x9c\xcd\x4e\xbf\x97\x64\xf8\x7e\x6a\xfa\x7f\x13\xc4\x06\x25\x79\xb0\x22\x28\x11\x7b\x15\xa8\x73\x8a\xb3\x9c\xd6\x44\x77\x06\x9c\xb4\xf5\x2c\xd8\xd5\xf4\x57\x4c\x65\x7b\x45\x38\x35\xca\x3c\xed\xb8\x24\xf0\x3b\x92\xa5\x73\xd6\xd3\xd9\x13\x61\x31\x3f\x11\xbd\xcb\x34\xd2\x05\x9f\xe2\xe6\xce\x2b\x85\x44\x61\xaf\x58\xa9\x29\x4c\x88\xcb\xfb\x2a\x63\x99\x76\xb5\x6e\x47\x48\x02\x6f\x30\x40\xe2\xfd\x71\x12\xd6\xad\x44\x50\x06\x89\xac\x77\x7c\x07\x1d\x17\x39\x19\x69\x76\x2e\x18\x64\x17\xc4\x40\x0a\xbd\xda\x5c\x16\xdc\xe0\x07\x76\x42\xf1\xfc\x13\x54\xe0\xe8\xc1\x4e\x55\x8c\x92\x3c\x1b\xfb\x85\x48\x8b\x83\x50\xf4\x15\x86\x6a\x60\x87\x1e\xd7\x15\x1f\x5f\xbc\x5b\x88\x05\x00\x01\x19\x77\xc7\x78\xe1\x7f\xe8\x91\x8c\x5d\x34\x3f\x70\xb0\x0d\x58\xf7\x18\x95\x61\x25\xfe\x28\xb3\xa5\xe2\xd0\x76\x04\xa2\xb8\xa8\x77\x20\x44\x34\xce\x90\x3b\x35\xa0\x30\x93\x6b\xc7\x19\x51\xca\x59\x3d\xf9\x7d\x24\xe8\xe8\xad\x8f\x2d\xc9\xb7\x8f\x76\xef\x13\xa1\xd3\x86\xca\x85\x7c\xed\x48\xf1\x9f\x3e\xbe\x39\x10\x8f\x9b\x33\xff\x59\xeb\x05\x56\xb1",
        );
        test::<Sha512>(
            b"\x23\x7a\x7e\x44\xb0\xa6\xc2\x68\xbb\x63\x36\x4b\x95\x8a\xe0\x2b\x95\xe7\xee\xd3\x6b\x3e\xa5\xbf\xb1\x8b\x9b\x81\xc3\x8e\x26\x63\xd1\x87\x14\x4e\x32\x3f\x9c\xea\xfb\x47\x95\x07\xd1\x84\xe6\x3c\xfb\xec\x3e\xcd\xbb\x8a\x05\xd2\xdf\xc8\x92\x96\x93\xed\x9e\x3e\x79\xe5\xf8\xab\xfc\x41\x7b\xa1\xe1\x7e\x3e\x28\x1e\x8a\x0a\x32\xf0\x84\x11\x7f\x28\xc3\xdc\xbe\xc5\x1b\x86\xf5\xc8\x5b\x28\x22\x44\x1a\x94\x23\xb5\xb4\x46\xd3\x92\x8f\x97\x76\x26\xa3\x34\x57\x9b\x39\xcf\xaf\x58\xf2\x14\xc9\x8d\x0c\xdf\x64\x0b\xe1\xac\x59",
            n,
            e,
            b"\xaf\x07\x6b\xc2\x13\xca\xf7\x56\x19\xf4\xbd\x1d\x78\x7c\xc1\x98\xf7\xdf\x33\x24\xa0\xdd\x87\xa8\x84\x16\xe0\xa4\xb8\x1c\x2f\xb9\xa9\xdb\x5f\x98\xae\xd4\x3b\xc1\x5f\xe2\x35\x71\x43\xa6\xe4\xff\x70\x1d\x9c\x48\xf5\x1d\xe9\xeb\x80\x36\x70\xbb\xc4\xb0\xae\xa7\x22\x0b\xe2\xf8\x4b\x83\x00\x31\x8c\x77\xa9\xf6\x15\x98\x6c\x49\x80\xab\xda\x85\xe3\xad\x00\x89\x56\x4d\xba\xf7\xf4\x4d\x81\xb6\x66\x4e\xec\x03\x11\xad\xb1\x94\xd4\x6d\xe9\x6b\xb1\x7d\x5a\x5d\x47\x42\x68\x45\x80\x2c\xa0\xf4\x9a\x16\x9e\xb8\x2b\x75\xaf\xa1\x91\x02\x7a\x0c\xc8\xfc\xe9\xdd\x16\x05\x53\x50\xdf\x97\x45\xfc\x72\x00\xff\x9f\x4e\xa3\xcf\xbf\xc6\x6c\x42\x84\x81\x13\xe3\xbe\x32\x93\xd5\x10\x38\x2d\x09\x99\xf0\x32\x51\x55\x27\xbd\x99\xf6\x6e\xfa\x2a\x75\x5e\x01\x12\x47\xb2\x23\xa6\x8e\x51\x25\x8b\x6b\xc3\x19\xa7\xcd\xef\x4a\xec\x53\x3e\x9d\xcd\x8a\xe2\x6e\x34\x9e\x5b\x33\xc7\x91\x21\x90\x7d\xe5\x09\xa1\xcb\x83\xc2\xe5\x9a\x47\xc1\xa8\x84\xbf\x68\xe7\x22\x93\x16\xa6\x2e\x3c\x49\xd1\xf5\x42\xeb\xe7\x10\x5c\xfc\x27\x09\x92\x68\x12\x0a\x77\x43\x90\x84\x71",
        );
        test::<Sha512>(
            b"\xab\x18\x93\x92\x30\xb0\x96\x64\x6a\x37\xa7\x81\x62\x9f\xbd\x92\x70\xf3\x89\x1a\x5c\xea\xb4\xa8\xc3\xbc\x68\x51\xbc\x34\x11\x5d\xbc\x06\x65\x41\xb7\x64\xa2\xce\x88\xcc\x16\xa7\x93\x24\xe5\xf8\xa9\x08\x07\x65\x2c\x63\x90\x41\x73\x3c\x34\x01\x6f\xd3\x0a\xf0\x8f\xed\x90\x24\xe2\x6c\xf0\xb0\x7c\x22\x81\x1b\x1a\xe7\x91\x11\x09\xe9\x62\x59\x43\x44\x72\x07\xdc\xd3\xff\xf3\x9c\x45\xcb\x69\xee\x73\x1d\x22\xf8\xf0\x08\x73\x0c\xe2\xef\xc5\x3f\x11\x49\x45\x57\x3e\xa2\xdd\xeb\xb6\xe2\x62\xc5\x27\xd2\x0f\x8b\xb1\xdc\x32",
            n,
            e,
            b"\x95\xbd\x0b\xf2\x36\x2f\x34\xb2\xe0\x40\x75\xb2\x93\x4f\x40\x47\x98\x70\x3e\xa4\x72\xb8\x1a\xc3\xcc\x22\x3a\xec\x48\x6e\x4c\x3d\x9c\x5d\x1c\x2f\x9e\xe2\x24\x17\x13\x29\x64\xed\x58\xe4\x99\x37\xf5\xb2\x57\xd3\x16\xca\x7f\xff\xe2\x90\xb1\x9f\x5b\x58\x10\x38\x36\x81\x2b\xef\x30\xca\x03\x27\x03\x9d\x8b\x9e\xa9\x12\x95\x39\x2f\xc3\x94\xb8\x81\xe2\xd2\xac\x9e\x30\xc5\xa4\x42\x56\x70\x0f\xc9\xde\x0d\xba\x29\x82\x73\xae\xc3\x0c\x4f\x77\x8d\x2e\x71\x27\xe8\xb8\xa8\x8b\x02\x74\xfc\xe0\x40\x81\xcc\x13\xad\xbe\xfe\x55\x50\x14\xe1\xb5\xd5\xdc\xf6\x22\x4c\x5a\xe2\x77\x54\x23\xa6\x6c\x81\x81\x8e\xec\x01\x4a\x3f\xaf\x9e\xe7\x5a\x3f\x6c\x3e\x51\xc5\x56\xb0\xa2\x88\xe8\xc2\x62\x94\x66\x84\xeb\x62\x8b\x88\xe3\xf8\x75\xe6\x2e\xf6\xe8\x01\xca\xe7\x5f\x61\xce\xe4\x04\x97\x1c\x39\xd2\x4a\x97\x12\xeb\x34\x2d\xdc\x66\x35\x15\xde\xc1\x03\xb1\x8d\x97\xd7\x8e\xd6\x82\x12\xf2\x79\x00\xe7\x7c\x04\x9b\x60\xc8\x53\x00\x2b\x08\x02\x2d\xf5\x6f\x70\x7e\xfa\x71\x02\x75\x89\xe1\xa3\xca\x6e\x41\x5b\xa5\xf4\x43\x7e\x97\x8b\x07\xaf\x3b\x73\xba\x0d",
        );
        test::<Sha512>(
            b"\xa2\x80\xe8\x9c\xeb\x2c\x8c\xf2\x62\x97\x19\x1b\xaf\x9a\x95\x5d\x0d\x52\x37\x5d\xa0\x23\x63\x3e\x0a\xfc\xdb\x0d\x39\xdc\x33\x5d\x82\x95\x85\x2e\xf4\xd0\x67\x14\xe6\x51\x1a\x95\xd3\x7c\x04\xd2\x68\x18\x60\x6a\xda\x54\x35\x9b\x7d\x07\x84\xaa\x93\x3c\xc6\x85\x61\xee\x96\xa8\x89\x10\xaa\x3d\x93\xd1\x07\x87\xcd\x1d\x75\x80\x55\x67\x31\xc1\x74\xa6\xe3\xa3\x2d\x9d\xcf\xa4\x16\x60\x4f\x0c\x67\x14\x81\xd0\x51\xf6\x3d\xb6\x91\x9f\x4a\xba\x44\x86\xd1\xb0\xfd\xc6\x11\x2c\x15\x21\x55\x9f\x42\x45\x23\xc2\x6b\x4f\xb7\x38",
            n,
            e,
            b"\xcd\x60\xde\x3b\x4a\x12\x89\xa8\x4c\xa7\x61\xf9\x0f\xa6\x3f\x4d\x56\x88\xbd\x88\x5f\x4b\x53\x1c\x85\x15\xad\xd2\xde\x12\x51\xf9\x93\xff\x7f\x98\x6b\xef\x3f\xba\x69\x2e\xcd\xeb\xc8\x19\x42\xd7\x42\x9c\x7a\x59\xc5\xd3\xf1\xfb\x87\x2f\xc1\xda\x19\x15\xe9\x45\x86\xa5\xc3\xd9\x63\x60\x36\x19\x00\x8f\x7e\xfe\xde\xd1\xd7\x0b\x0a\x11\xce\x2c\xd8\x1b\x5b\x0d\x86\xb3\x76\x0c\x94\x83\x67\x4f\x55\xe9\xfa\x47\xf2\xf3\x10\xd5\x88\xfb\x21\x60\xe8\xb5\xc3\x2b\xe4\xe7\xa9\x68\xd5\xa8\xd4\xac\x65\x76\xb7\x1a\x2b\x91\xcd\x6a\xf0\x01\x6c\xbc\x81\x6d\x4a\xae\x8c\x70\x64\x9e\x08\xdc\xe9\x0b\x3c\xe5\x2a\xb4\x9c\xe2\xcb\x5b\x0e\xd8\xa4\x5e\x33\xd9\x4c\xf2\xd4\xcf\xde\xe1\x15\x12\x70\xb2\x07\x3a\xef\xfe\xaf\x71\x7d\x39\xe0\x41\x92\xb8\xb6\x93\xc5\x3f\x21\xa6\x12\x38\x13\x28\x08\x06\x92\x0b\x7d\xc5\x82\x20\x1c\x9d\x11\x70\x50\x32\x06\x71\xe8\x61\x39\xa0\x27\x97\x6b\x7e\xcf\x41\x33\x69\xa9\xfc\x28\xe0\xbd\x71\x9c\xeb\x5e\x10\x7d\xe7\x99\xf1\xbc\x2e\x25\x5a\x9f\x29\x47\x6d\x45\x74\xd1\x33\x2f\x66\x46\x8a\xfb\x90\x04\xff\x7b\x53\x53\x02",
        );
        test::<Sha512>(
            b"\x85\xed\x1e\x3d\xfc\xd5\xbc\xa2\x4c\xad\x1d\x01\xeb\xe1\x92\xb7\xd0\x59\xec\x9b\x88\x44\x36\xe1\x87\x14\xa4\x3f\xbc\xc9\xc6\x4f\x68\x73\x01\x35\x2f\xf2\x40\x81\x70\x01\xe7\x57\xd2\x73\x09\xcd\x1f\xbb\xda\x94\x56\xb2\x67\xdb\xfb\x95\x84\x70\xb2\x4d\x06\x28\x0c\xf4\x33\x82\xa1\x94\x77\x87\x5f\x32\x59\xf4\x21\x0b\xac\x9b\x83\x1d\x0a\x07\xf5\xe9\x7e\x5f\x0f\x78\x81\x8c\x25\x9c\x28\x9e\x1a\x78\x9b\x6c\x79\x42\xc9\x7b\xc1\x48\x5a\x22\x01\x31\xe5\xeb\xa5\x86\x64\x3b\x90\x71\xe5\x36\x6b\xc4\x82\xdd\x3c\x3c\x92\x79",
            n,
            e,
            b"\x13\x81\x34\xbb\xec\xef\xaf\xc7\xca\x8b\x10\x2c\xbe\x87\xb0\x12\xf8\xaa\xda\x88\x78\x99\x50\x02\xcf\x18\x87\x69\x4b\x5b\xe3\xb8\xf0\xbb\x61\x6b\xc6\xe0\x79\x62\xd5\x48\x2d\x3a\x52\xc5\x2a\xb9\x1b\x3e\xe0\x06\x4d\x24\x55\x8e\x13\xc7\x5c\x80\xf6\xa9\x5b\x7d\xc4\x98\x44\x28\x79\xd5\xba\xf8\xff\xa7\xe2\xf6\x38\x80\x8b\x97\xff\x70\x13\x6b\xb6\x45\xe3\x09\x44\xdd\x97\xa9\x97\xa0\x20\x51\x69\x55\x3a\x5b\x9e\x87\x4c\x5a\x94\x41\xe1\x8c\x15\xeb\xed\x76\x04\x3b\x63\x9d\xfd\x64\xdb\x79\xe1\x74\x84\x7a\x10\x27\x24\xa2\xa0\x5c\x64\x94\x73\xcc\x7d\xac\xd3\x9e\x2e\x1d\x56\x66\xbb\xb5\xf0\x12\x46\x74\x70\x48\xff\xfc\xdf\xcd\xdf\x78\x2d\xa2\x4a\x6d\xcc\x02\x2b\x26\x95\xf7\x07\x81\xbd\x9f\x8f\xf7\xd0\x3b\xe2\x2e\xb8\xfc\x79\x3f\x5c\x07\x1a\x66\xd9\xa6\xea\x46\xc6\xa2\xcf\x05\x56\x52\x6b\xa8\xb0\x85\x07\x35\x46\x44\x80\x81\x73\x2a\xc1\x5f\x12\x83\x3c\x1d\xb1\x70\x1f\xf7\xf6\x83\x44\xca\x65\xdf\xf8\x62\x11\xa0\x03\xad\xbf\x51\x89\xcf\xae\x79\xea\xa8\xc8\xb7\x14\x1e\xa3\x78\xe4\x4c\xc9\xc5\xbf\x02\x4d\x2c\x71\x0f\xf5\xcd\x68\xaf",
        );
        test::<Sha512>(
            b"\x0b\xdb\xa3\x4e\x35\xfc\xa6\x5a\x17\x81\xd4\xd7\xc9\x33\xa5\xf2\x10\xd3\xa5\x94\x83\xae\xbc\x95\xec\x71\xb3\x2d\xf1\x3f\xf4\xab\xf4\x01\x91\x69\x37\xfd\x88\xff\x44\xab\x46\xb7\x8c\xc3\x69\x41\x4e\x9b\xca\xa8\xba\xb0\xbb\x85\x57\x82\x8d\x73\xa2\xa6\x56\xc2\xf8\x16\xf0\x70\xb5\xcb\x45\x54\x9e\x8e\xca\x9d\x7c\x0b\x4a\x7b\x0a\x27\xe5\x1c\x11\x93\x58\xda\xd2\xa1\x7f\xb3\xa4\x57\x18\xf9\xde\xc3\xc9\x4a\xf7\x8d\x65\xc3\xec\xd3\x6b\x71\xe2\x30\xcf\x08\x0d\x1e\xfd\xd8\xd0\x7f\x1c\xfc\x26\x76\x8f\xd5\x40\x7b\xc2\xb7",
            n,
            e,
            b"\x9f\x48\xde\xb9\x6b\xec\x0b\x72\xfb\xc4\xf1\x2f\x08\xaf\xb4\x6b\xcc\xf1\x9d\x9e\x0c\xd0\x36\x8e\xbe\xb3\x12\xd8\x38\x72\x62\x63\x80\xac\x92\x8b\x61\x2c\x5c\xd7\x74\x38\xd4\x7a\xa9\xce\xea\x90\x5a\x9d\xe7\x18\x2c\x8e\xf7\x6e\x8a\x7a\x03\xd6\xef\xec\x84\x00\xb6\x49\x63\x62\xbf\x6a\x30\xce\xb1\xce\xd2\x18\x5f\xc7\xc2\x11\x7b\x6a\x6d\x88\x8a\xc2\x0c\x16\x87\xb0\xf2\xaa\x9b\x76\x70\x5f\xd3\x15\x48\x89\xb6\xac\xaf\x4e\x63\xbe\x25\x88\x0c\x71\xe6\xc2\x39\xec\xfb\x96\x50\x04\xcd\x63\x21\x25\x7f\x84\x6a\xfd\x2a\x65\x90\xc7\x2a\xd8\x31\x46\xee\xfc\x7b\x0d\xc4\x79\x63\x39\xa7\xf6\x4d\xa0\xfb\xe3\x59\xf9\x4a\xce\x1f\xd1\x51\xc5\xac\x7b\xb5\x70\x7b\x32\xea\xcf\x56\x4f\xe1\x62\x2e\x66\xe1\x84\x4e\x63\x96\x02\xca\x36\x27\x4a\xe0\x1f\x93\xe6\xb2\xbd\x1e\xff\xd3\x4a\xb6\x3d\x85\x2c\xc9\xca\xf3\xce\x84\x46\xc2\x9c\x8a\xe3\xc6\x11\x0f\xb7\x53\x8c\xc8\x37\x1c\x2a\x39\x81\x24\x9c\xdc\x1b\xe2\xb2\x4b\x6a\x0c\x95\x17\x64\xd0\xb7\xef\xa9\x2a\x22\xcd\x8e\xd1\x65\xe1\x82\x86\x35\x79\x37\x79\x97\xa9\xee\x50\xc8\xac\x3a\xa4\xdf\x1a\xca",
        );
        test::<Sha512>(
            b"\x9a\xee\xd8\x5b\x40\xba\x7f\x86\xa2\x28\xb5\xa1\x51\x5b\xa1\x90\xb2\xef\xff\x66\x99\x3a\x5e\xce\x19\xd1\x8b\xaa\x9b\x4e\x4d\xf9\x2e\x51\x52\xfe\x1e\xc5\x6a\x9f\xc8\x65\xf3\x0b\xac\x7e\x94\x9f\xc4\xf6\x2f\x0b\x15\x8d\x10\xb0\x83\x63\x6b\x4d\xe9\xbb\x05\xdb\x69\xfe\x31\xb5\x01\x03\xfe\xfc\x5f\x8d\xaf\x3a\xf7\x15\x6b\x45\x52\xca\x36\x67\xa9\xd7\x20\xbb\xb2\xe4\xbc\xda\xba\xdf\xd4\xb7\xf4\xfc\x5b\xc8\x11\xfa\xa3\x67\x10\xa9\xd1\x77\x58\xa9\x8d\x4a\x04\x74\xfe\xc2\x7e\x9e\xf5\xb7\x4f\x5c\x68\x99\x35\x44\x23\x57",
            n,
            e,
            b"\x9e\xec\xdb\xd7\xfb\xf6\x18\xdd\xdd\xfb\x6e\x75\xd6\x44\x40\xf6\x04\x45\xb8\x53\xc5\x42\xfe\x0f\xba\xaa\x6a\x43\x12\x94\xe6\xcb\x66\x83\xae\x1a\x71\xea\x05\x5e\xb4\x9c\xd2\xa3\xcb\x51\x54\xdc\x93\xd9\xaa\x16\x63\x99\xf4\xe6\x29\x4f\x0e\xb0\x65\x28\x00\xd7\x1e\x04\x1c\x1c\xe1\xad\x84\x9c\x03\xc9\x63\xbc\x09\x29\xdc\xdd\x11\xbe\x5d\x67\xa0\x50\xd0\x2b\x64\xb2\x9e\xab\xa6\x55\x64\x2b\x64\x36\xfb\xfb\x16\x36\x90\xbf\x43\x2f\xdc\xee\xdd\x10\x6c\x2f\x49\x72\xec\xbf\x30\x77\xed\x8b\x75\x3b\xb6\x05\xec\x1e\xa0\x30\x20\x83\x9a\x31\x8a\x24\xf8\xd4\xc1\xd7\xd8\xdf\x99\xa7\xf0\x01\x0a\xe4\x1a\x8b\x06\x8e\x28\x88\x53\x10\x56\xa7\xda\xbb\xe9\x21\x87\x8d\xcd\x3c\x7d\x69\x41\x68\x67\xf4\x01\x2a\x60\x6a\xe8\x68\x55\xf1\x5a\xed\x0d\xa1\x25\x0e\x59\x68\x77\x06\xe8\x9c\x94\x94\xba\xf3\x7f\x61\xfb\x17\x03\xb7\x99\x28\x79\x5f\x90\xcc\xbe\x29\x3a\x1e\x94\x72\xf6\xe0\xf4\xb8\x90\xfd\xda\x3e\xa2\x52\x2e\x3d\x11\xd5\xab\xdf\x00\x69\x51\x94\x24\xd1\x47\xb5\x64\x6a\x5a\x60\x1f\x19\xec\x89\x72\x9a\x8b\x48\x46\x1e\x71\xc0\x8b\xbe\x9c\xda",
        );
        test::<Sha512>(
            b"\x65\x4e\x18\x9f\x06\xc7\xd4\x2d\x55\x39\xa5\x87\x21\x84\xf8\x33\x6c\xf1\x00\x69\x1f\x19\x08\x18\xfd\x02\x08\x2a\xd6\x8a\x76\x09\xfd\x09\x5e\x62\xfc\x32\xb5\x29\x85\x3a\xeb\xdd\xac\x3d\xbf\x0d\x54\xdd\x57\x1b\xe7\x2c\x90\x40\x4b\xcc\x93\xd0\x11\x54\xa9\xbf\xef\xf6\x50\x65\x70\x5f\x8e\x7e\xea\xdf\x85\x75\xb1\xca\x48\xe2\x8a\x1e\xed\x51\x62\x65\xe3\x45\x40\xdd\x86\x7c\x79\xd7\xf1\x75\x23\x5d\x13\x30\xcb\x17\x06\x35\x6b\x70\x9b\xd7\x96\xf4\x3a\xba\xf6\xfc\xe9\x93\xf8\x8e\xaa\x2f\xc6\x7f\x0a\xb7\x76\xda\xf7\x32",
            n,
            e,
            b"\xaf\x90\x29\x8b\xce\xf6\x15\x30\x9f\x23\x5d\x5c\x33\x60\xf0\xdf\x11\xf5\xfb\x98\x87\x89\xf2\x13\xd4\xc4\x61\x34\xfe\xe5\xeb\x10\x4a\xa1\xfa\xbb\x13\x07\xc9\xa9\x04\x70\x9d\xe8\x86\x73\xed\x99\x51\xcb\xa9\x31\x67\xc6\x7c\x09\xd8\x27\x02\x1b\x08\xa2\x2c\x05\x05\x82\x8a\xb4\xbe\xb4\x2e\x59\xa3\x88\x32\xcb\x4d\xa2\x4e\xcf\x91\xf4\x70\xa3\xb4\x12\xc0\x71\x2a\x8a\x59\xf6\xf2\x73\x9d\x4e\x9e\xb4\xcc\x58\xd2\xc5\x25\x92\xf1\x45\x2d\xc6\x57\x59\xab\xe4\x3e\x8d\x2b\xc8\x04\xe2\xef\xb3\xef\xc9\xb2\x3c\xc1\x73\x4f\xf7\xca\xef\xa4\x6b\x03\xba\x4b\x39\x7d\x07\x14\xcd\xb8\x50\x1a\x81\x2c\x1b\x9f\x47\x41\x1c\x91\xcb\xa5\x3a\x3d\x3b\x13\x9e\xdb\xd7\xcb\xb5\x43\xf5\xbf\x38\x29\xba\x7f\x5f\xaf\xd8\xa7\x12\xc0\xb1\x11\x94\x3f\x53\x20\x93\x53\xaf\xab\xa1\x76\xb3\xf5\xdc\x06\x03\x39\xd0\x9b\x1f\xb3\xc2\x13\xda\xe5\xd0\xf0\x04\xd3\x02\x82\x85\x60\xfb\x5d\xeb\xf9\xfe\x49\x1e\xaa\x66\xf5\x97\xaa\x4d\xe2\x3e\xee\xf9\x17\x63\x58\x75\x5c\x95\x2e\xf9\x6e\x36\x72\x58\x3b\x6e\xcd\x95\xa0\x2e\x8c\xa7\xb2\x1d\x7c\x20\xcb\xb7\xa7\x57\xaf\x71",
        );
        test::<Sha512>(
            b"\x12\x1f\x80\xb4\x3f\x97\x57\xb3\xfa\x80\x90\x6a\xea\xb2\x32\x19\x5f\x0e\x2c\x41\xe5\xbf\x8c\x09\x1a\xc0\xf1\xe0\xbc\x9e\x43\x64\x06\x80\xa1\x82\x3d\x64\x9b\xdf\x86\xab\xa2\x77\xfa\xd8\xbc\x85\xfc\x95\x7d\xa2\xca\xf7\x32\x30\x53\x02\x5f\xf9\x49\x70\x6c\x14\x76\xae\x9b\x09\x53\x28\x3d\x34\xd7\xc6\x26\x6f\x8d\xb6\x5e\xeb\xe9\x6d\x19\x5f\xdc\xe8\xe9\x65\xa6\x38\x33\x20\xec\x3d\xe0\x23\x0a\xb2\x54\x8e\xaa\x69\xa4\x7a\x96\xd8\x03\x98\xca\xd5\x7e\x14\xce\x9e\xea\xc0\x42\x1c\x1a\x6e\xba\x69\x55\x9d\xcd\x8f\x06\x59",
            n,
            e,
            b"\x06\xa2\xd7\x45\x85\xf1\x2e\xa7\xa8\x05\x27\xb8\xc6\x35\xa2\x1c\xc1\x1b\x45\xdb\xb0\x88\x5a\x12\x72\x21\x26\x81\x1d\xd2\x5d\x65\x7b\xfa\x9f\xda\x77\x43\x01\xca\x34\x98\xd0\x5d\xfd\xfb\x78\xa6\xaa\x16\xa9\xf8\xa9\x5f\x40\xf1\xf0\x4b\xd3\x54\xa5\x22\xf6\xa2\xd6\x2b\x32\x4e\xfa\x3c\x00\x6c\x22\xc2\x31\x4b\x01\xfa\x0e\x91\xa3\xdb\xa4\x9a\xa3\x5b\x46\xb1\x98\x04\xb0\x7a\xd9\x8f\xe4\xbc\x99\x03\x93\xa4\xa2\x73\xce\x8f\x1c\x85\xfc\x19\xcd\x5e\xae\x9a\xf0\xb7\xd1\x95\x7b\xb2\x34\x09\x77\x8a\x01\x0b\x00\xc6\x95\x9e\x1b\x67\x06\x6f\xdb\x9f\x84\x95\xb4\xde\x4d\xcb\xb9\x87\x35\x81\x45\xb1\xff\x6a\x39\xef\x6f\xc5\x88\xcd\xa1\x74\x4e\x0a\xb9\xe7\xeb\x00\x2c\x29\xa7\x85\x31\xd2\x51\x57\xc5\xc2\xcd\x64\x70\x55\x15\x60\xa0\x28\x45\xdb\x6d\xbe\xe2\x42\xf9\x65\xa2\x55\x40\x6f\x6e\xf4\x7b\x32\x21\xa5\x11\x0e\xdb\x44\xd3\x8b\x94\x19\x1a\xea\xf4\x33\xc0\xec\xe3\x48\x0b\x9d\x1b\x06\xd8\xb8\xb6\xc0\xa2\x32\xa0\x4c\x56\x78\x88\xe6\x37\x2f\x2e\x94\xbc\x2b\xe6\xb8\x27\xf8\x71\x2a\xf4\x8c\x6f\x1e\x4f\x22\x3f\x55\x28\xfc\xf3\x48\x79\x9d",
        );

        // [mod = 3072]
        let n = b"\xdc\xa9\x83\x04\xb7\x29\xe8\x19\xb3\x40\xe2\x6c\xec\xb7\x30\xae\xcb\xd8\x93\x0e\x33\x4c\x73\x14\x93\xb1\x80\xde\x97\x0e\x6d\x3b\xc5\x79\xf8\x6c\x8d\x5d\x03\x2f\x8c\xd3\x3c\x43\x97\xee\x7f\xfd\x01\x9d\x51\xb0\xa7\xdb\xe4\xf5\x25\x05\xa1\xa3\x4a\xe3\x5d\x23\xcf\xaa\xf5\x94\x41\x9d\x50\x9f\x46\x9b\x13\x69\x58\x9f\x9c\x86\x16\xa7\xd6\x98\x51\x3b\xc1\xd4\x23\xd7\x00\x70\xd3\xd7\x2b\x99\x6c\x23\xab\xe6\x8b\x22\xcc\xc3\x9a\xab\xd1\x65\x07\x12\x40\x42\xc8\x8d\x4d\xa6\xa7\x45\x12\x88\xec\x87\xc9\x24\x4b\xe2\x26\xaa\xc0\x2d\x18\x17\x68\x2f\x80\xcc\x34\xc6\xea\xf3\x7e\xc8\x4d\x24\x7a\xae\xde\xbb\x56\xc3\xbb\xca\xff\xb5\xcf\x42\xf6\x1f\xe1\xb7\xf3\xfc\x89\x74\x8e\x21\x39\x73\xbf\x5f\x67\x9d\x8b\x8b\x42\xa4\x7a\xc4\xaf\xd9\xe5\x1e\x1d\x12\x14\xdf\xe1\xa7\xe1\x16\x90\x80\xbd\x9a\xd9\x17\x58\xf6\xc0\xf9\xb2\x2a\xe4\x0a\xf6\xb4\x14\x03\xd8\xf2\xd9\x6d\xb5\xa0\x88\xda\xa5\xef\x86\x83\xf8\x6f\x50\x1f\x7a\xd3\xf3\x58\xb6\x33\x7d\xa5\x5c\x6c\xfc\x00\x31\x97\x42\x0c\x1c\x75\xab\xdb\x7b\xe1\x40\x3e\xa4\xf3\xe6\x42\x59\xf5\xc6\xda\x33\x25\xbb\x87\xd6\x05\xb6\xe1\x4b\x53\x50\xe6\xe1\x45\x5c\x9d\x49\x7d\x81\x04\x66\x08\xe3\x87\x95\xdc\x85\xab\xa4\x06\xc9\xde\x1f\x4f\x99\x90\xd5\x15\x3b\x98\xbb\xab\xbd\xcb\xd6\xbb\x18\x85\x43\x12\xb2\xda\x48\xb4\x11\xe8\x38\xf2\x6a\xe3\x10\x9f\x10\x4d\xfd\x16\x19\xf9\x91\x82\x4e\xc8\x19\x86\x1e\x51\x99\xf2\x6b\xb9\xb3\xb2\x99\xbf\xa9\xec\x2f\xd6\x91\x27\x1b\x58\xa8\xad\xec\xbf\x0f\xf6\x27\xb5\x43\x36\xf3\xdf\x70\x03\xd7\x0e\x37\xd1\x1d\xdb\xd9\x30\xd9\xab\xa7\xe8\x8e\xd4\x01\xac\xb4\x40\x92\xfd\x53\xd5";
        let e = b"\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xea\xf0\x5d";
        test::<Sha224>(
            b"\x25\x4c\xe3\x6e\x8e\xd6\x2e\x08\x87\xd4\xbe\x00\xee\xfa\x82\x51\x5a\xce\xf9\x56\x54\x0c\xff\x45\xc4\x48\xe7\xf9\xa9\xd5\xc9\xf4\x0d\xe6\x1d\xa4\x39\xf3\x89\xe5\x25\x5e\xf8\xc8\x32\x57\xec\x92\x1b\xfd\x15\x08\x29\xc5\x22\xea\xa7\x20\xd7\xbe\x96\x58\x60\xce\xa2\xbb\xe5\x74\x54\xfc\x5e\x95\x88\xd6\xa9\x6c\x22\xf2\xd9\x89\xfd\x0b\xd2\x19\x24\x50\x13\x67\x45\x0a\xd2\xa3\x62\x7e\x4e\xe3\xca\x15\x61\x67\x48\xba\x54\x21\x9a\x84\xf8\x74\x24\x95\xf2\x3d\xe6\x42\x57\x10\xac\x74\x79\xc4\x84\x4d\x00\x31\x75\x0f\x3c\x38",
            n,
            e,
            b"\x9d\xfd\x3f\x32\x09\x1b\x91\x6a\x56\xf4\xf3\x57\xa9\x61\xa5\x25\xa5\x27\xfd\x29\xb1\x14\xb3\xf0\x38\x29\xdf\x1b\x25\xc0\xc5\x97\x3b\x1c\x51\xaf\x36\x63\x31\x16\xf4\xc7\x7a\xa2\xf6\x77\xa3\xb0\xf8\x23\x68\xa5\x38\xbd\xb3\x3e\x49\xdb\x9f\xa7\x04\xbd\x51\x23\xed\xef\xbd\x3a\x86\xdc\xc3\x92\x83\xc2\xa0\x3c\x96\xc6\x90\x30\xf0\x4c\x64\x84\x17\xf5\x44\xf0\x8a\x9b\x4e\x01\xae\x4d\x42\x9e\xf2\x14\x22\xdd\xfe\x76\x83\x4f\x92\x5c\x56\x53\xb1\x22\x78\x35\xa9\xa4\x41\x3d\xa7\x94\x2b\x0a\x01\x51\x96\xfa\xec\x31\x50\x41\x11\xc9\xf0\x84\xd6\xdd\x6d\x5a\x69\x56\xc5\x54\x12\xc1\xbd\x14\xaa\xf9\x5d\x82\x8e\x84\x49\x61\xfd\xd6\x0c\xc0\x78\xf1\x69\xf6\xe1\x18\x6c\xb0\xcb\x6b\xa3\xd2\x11\x68\xc5\xbf\xd0\x67\xdc\x6e\x46\x07\x84\xe7\xc6\xa7\x6e\xe3\xd8\xb3\x32\xac\xfa\x97\xe5\xd4\xb6\x56\xec\x8c\x61\x1e\xbd\x89\x6f\xe9\x0e\x61\x9b\x58\x8d\x0c\x61\x54\x92\x46\x4a\x1b\x3d\x05\xd3\xa9\x63\xf4\x51\x05\x1c\x65\xd8\xf8\x1f\xee\xa9\x25\xbc\xbe\xe9\xce\x7a\x39\xba\x3c\x91\x5a\x18\xa2\x4a\x45\x1e\x47\x0e\x76\x1d\x09\x85\x5a\x96\x5e\x83\xed\xae\x3f\xca\x41\x67\x8c\xc9\xd0\x98\xba\x99\x28\xb5\x25\xb5\x0e\x48\xcb\x03\x0c\x51\x0c\x4c\xe7\x27\xc6\xb9\x3b\xd0\x91\xb7\xd2\x0b\x4b\x96\x11\x65\xae\x0e\x28\x48\xaa\x99\x5b\xb7\x3a\xbe\x9a\x26\x34\x37\x8d\x22\x41\x28\x54\x1a\xb0\x56\xa3\x1b\x78\x48\x85\xae\xf8\x03\x4d\xed\xac\x13\x16\x74\x02\xf9\xf6\x2b\x55\x74\x12\x20\xdf\x8a\xff\x5d\xef\xb6\x9c\x03\x5d\x9a\x31\xe2\xa5\xb8\x81\x70\x57\x24\x1b\xcf\x85\x49\x32\xf5\xed\xee\x7e\xe6\x6e\x89\x17\xaa\x4a\x71\x8b\x6c\x44\x6b\xdd\xf0\x84\xf5\xcd\x76\x9c\xae\xff",
        );
        test::<Sha224>(
            b"\x35\xad\xcd\x3f\x24\xb6\x72\x55\x18\x81\x5c\xf4\x60\x6f\x7b\x1d\x94\x0c\x39\x63\x84\x37\x0a\x37\x6e\x84\x45\x69\x74\xde\x32\xec\x4c\x71\x64\xda\x3a\xc7\x49\xb7\x3b\x30\xff\xfa\x83\x68\x69\xc9\x2a\x74\x83\x05\x23\xcd\xf2\x86\x6d\xc0\xe0\xa8\x8d\x15\x06\x06\x3b\xef\x0a\x85\x5c\xf3\x0c\x95\x30\xac\x7c\xb3\xcd\x2e\x2e\x32\xcc\xfa\xb0\x3c\x42\x22\xdb\x14\xf2\xae\xa8\x9d\xc0\x3d\x45\x20\x69\xf0\x68\x4a\x72\x83\xe4\x74\x5e\xbd\x7a\x28\x24\x0b\xf1\xe0\xe0\x68\x68\x10\xc9\x7f\xec\x67\x63\x14\x46\x52\xf6\xa0\x16\xc3",
            n,
            e,
            b"\xb5\x12\x0b\xe9\x8b\xcd\xfd\xc1\xe1\xe3\x31\x2d\xd7\xb5\x91\x0f\x07\x31\x32\xa4\x27\x76\xc4\xda\x75\x69\x0c\x64\x1f\x32\xd2\x89\x91\x87\xd9\xb3\x9b\x55\xf9\x9e\xbe\x6c\xa0\xa0\x80\x36\x37\x27\x49\x81\x67\x06\x66\x4f\x39\xb2\x73\x12\x13\x5c\x50\x33\x9f\x2b\xb1\x7b\x2c\xee\xe2\x57\x68\xc2\xbc\x0a\xc3\x7d\x6c\xa6\xee\x90\x3c\x84\xe8\x2e\x2f\x4d\x00\x5d\x73\xbd\xc3\x35\xf1\x35\x39\x9c\x49\x12\x36\x62\xe8\x90\x81\x19\x91\x84\x37\xed\xb6\x15\xb1\x4e\x90\x6c\x9f\x8b\xa1\xb8\x5d\x5b\x45\x90\x9f\x43\x9c\xc8\x99\x29\x51\xbe\x16\x84\xa9\x9e\xba\x04\xec\xb0\xf6\xdf\x92\x33\x53\x51\x69\x77\x77\x4f\x69\xe8\x26\x65\x11\x90\xaf\xfa\x86\xa4\x0b\xe7\x5b\x06\xa4\x12\x8e\x55\x09\xc5\x15\x57\xae\x4f\xb4\x10\xc7\xe5\x84\x1a\xc9\xfd\xc4\xbc\x1f\x97\xe2\x86\x24\x29\xf3\x71\xaa\xaf\x99\x82\x4d\xac\xfe\xe0\xbc\x39\x61\xfb\x98\xb3\xff\xc0\x91\xf7\x79\x56\x22\x3e\xbf\x5b\xb5\x46\x55\x23\x58\x20\x8a\x32\xef\x9c\x37\x82\x5e\x81\x66\x8f\xd2\xc2\x30\xf7\x88\xca\x16\xff\xbc\xc0\xf1\xd8\x84\xb3\x0f\xe8\xef\xe6\x49\x82\x95\x00\x4c\xa7\xc7\xf2\xb1\x73\xe5\x66\x6b\x8b\x0f\xdf\x9d\x32\x75\x65\x59\xf9\x9d\x10\x5c\x1e\x80\x42\xa7\xae\xd7\x26\x2c\xa9\xa1\x70\x25\xaa\x09\x60\x75\xfe\x44\x33\xf3\x4d\xb6\xb0\xf1\x97\x77\x6c\x21\xfb\xe0\x0e\x83\x2e\xba\x02\x8e\x66\x52\x65\x30\x18\x07\x9f\xee\x04\xeb\x3e\x3c\x12\x80\x3c\x39\x83\x0d\x07\x2a\xb4\x97\x1b\xca\xb4\xb7\x97\x58\x69\x4b\x5d\x3d\x8a\xb2\x1c\xe8\x74\xb7\xc4\x2b\xed\xd5\x26\x52\x21\x9f\xf5\x16\xfd\x69\x4c\x3d\x7c\xb0\xbe\xf0\x18\x1b\xb8\x5e\xb4\xb1\x31\x84\xea\x3a\xef\xe3\xcc\xee\xa5\xc5\x75\x96\xf7",
        );
        test::<Sha224>(
            b"\x0b\xa5\x73\xa9\xdb\xb7\xf6\x2e\x5a\x4d\x3d\x84\x1b\xfd\x92\x98\xe8\xbb\x29\x9e\xb4\xfd\xb2\x56\xd1\x1d\x2f\x8d\x64\xfe\x03\xe6\x15\xf2\x4c\xda\x0b\xdb\x73\xfe\x17\x91\x02\x84\x2f\x84\xb5\x05\x1f\xa3\xd3\x7e\x7b\x7c\xbe\x98\xd8\xa4\xc9\x2c\x3b\x59\x4b\x06\xd2\x66\xf2\xe9\xe2\x47\x59\xd4\x01\x8e\xdc\x84\x85\x85\xab\x3a\x3c\x15\x1d\xbe\x5e\xe6\x47\xa4\xbf\xc8\xce\xce\x49\x52\xf9\x32\xaa\xc8\x0a\xdd\x4a\x42\xcf\x38\x80\x0b\x74\x8b\x05\x48\x9b\xbf\xa9\xda\xae\x68\x44\x85\x74\x03\xf0\x51\xe3\x7b\x75\x30\x36\xf3",
            n,
            e,
            b"\x36\xfd\x68\x13\xab\x41\x1c\x4d\xcb\x2d\x7b\xe1\xed\x61\x6c\x1e\x40\xde\x29\x1a\x00\xac\xd8\x7d\x2b\x4d\x9d\x4b\x73\xc8\x86\x4a\x44\x41\x3c\x51\xf0\x9b\x37\x84\x4a\x98\x04\xf8\x23\xb2\x7a\x90\x94\x62\x7a\xaa\xf0\x0a\x6b\xe9\x42\xd7\x55\x8b\xe1\x1b\x84\xa7\x3d\x98\x02\x9c\x2e\x26\xeb\x8f\x65\x05\x80\xec\xb1\x1b\x4e\xc2\x36\x35\x97\x33\x34\x44\x56\x96\x34\x35\x16\x00\x21\x29\x62\xfe\xf5\x35\x2b\xdb\xa3\x67\x83\x28\x99\xd3\x18\x8a\x74\x72\x36\xf0\x85\x28\xf0\x86\xd9\x3c\xa3\x3a\x06\xb1\x03\x92\xbb\xbd\x62\x5c\x86\x7d\xdb\xa7\x4b\xb1\x51\xdc\xc6\xaf\xdd\x4c\xe4\x10\x16\xdc\x2e\xf0\xce\xea\x2c\xa2\x09\x17\xfb\xdb\x07\x77\xe2\x35\x03\x46\x4d\x0b\xb5\x9c\xd4\xe1\x2c\x10\x94\x52\x50\x88\x9b\xae\x2e\xd8\x39\xb7\x09\x64\xb2\xc9\xd9\x57\xea\xc6\x22\x2a\x49\xb3\x37\x73\x04\x11\x98\x44\x48\xe5\x8c\x02\x73\x71\xbc\xf9\xc2\xc7\xd6\x86\xde\x3b\xda\xe1\x67\x38\xdb\x52\x76\xe0\xf5\x38\xd1\x5b\x35\x41\xc0\xed\x86\xd3\x18\xb4\x23\xd8\xc7\xf1\x85\x96\x02\x10\x8a\x4b\x11\xc2\x77\x29\x41\x39\x6a\x14\xa2\xa8\x8e\xc7\x97\x12\x97\xc1\x86\x33\x02\x09\x98\xee\x02\xb3\x11\x4d\x19\x01\x2a\x09\xa1\x81\xd0\x1f\x11\xcb\x8f\x8c\xb5\xf4\x38\xe8\x2f\xb4\x5e\x76\x78\xbc\x8d\xf9\xa2\x6f\x1a\x34\x95\x43\x9a\x7a\xc1\xf1\xbd\xa6\xfb\x86\xc9\xb3\xed\x6c\xb5\xf7\x88\x63\x49\x46\x34\x8b\x7e\x24\xb0\x89\x4c\x39\xc5\x06\xce\xd2\xda\x65\x7a\x33\x5e\x54\xe8\xf9\x97\x38\x4e\x40\xc5\x6a\x17\xa2\x8a\x9b\xb6\x48\x75\xa1\x59\xca\xda\x5a\x64\x4a\xb3\xbd\x6e\xa7\xbc\x4c\xca\xed\x43\xdd\x09\x55\xf6\xbe\x6e\x45\x9e\x2e\x6a\x7b\xa6\x52\xf1\xe9\xa3\xf8\xa8\x3e\x47\x95",
        );
        test::<Sha224>(
            b"\x89\x53\x0f\x81\x6a\x5e\x2a\xbd\x4b\x42\x2f\xdf\x96\x8f\xfd\x96\x4e\x0c\xcf\x82\xa4\xfc\x6d\x9a\xc5\xa1\xa4\xcb\xf7\xff\xf3\xe1\xe4\xe2\x87\xab\x35\x22\x6a\x5a\x63\x26\xf7\x2b\xca\xa7\x91\x46\x00\xb6\x94\xe5\x64\x01\x8c\xb8\xfa\x52\xa5\x89\x76\x58\x63\x1c\x96\xaa\x93\x59\xb5\x09\x82\xac\x9e\xe5\x6c\xad\x9e\x23\x37\xfc\xdd\x1e\x61\x6f\xed\xec\x38\x70\xa4\xe2\x49\xa0\x27\x5a\x1a\xc1\x48\xb3\x1c\xd2\x12\x9a\xdb\x7b\xa1\x88\x78\xac\x38\x8c\x59\x82\x8d\x4b\x1f\x6a\x67\x45\xd8\x88\x6b\x5a\x76\x5a\x33\x8c\x81\x98",
            n,
            e,
            b"\x27\xc7\x96\xca\xee\xe6\xb4\xbc\xd7\x50\xd8\xdf\x13\xcb\xe5\x16\x4f\xd7\x26\xf9\x1b\xaa\x57\x5f\x70\x2f\xe2\x96\x67\x44\xcf\x2b\xef\x38\xc9\x3e\xfa\x11\x11\xc9\x27\x7d\x77\xf3\xec\xf6\x97\xd0\x20\x30\xf0\x1e\x3d\x96\x4c\x31\x25\x53\x3d\x40\x88\x34\xb7\xce\x65\x28\x24\x30\x3e\xb2\x78\xdc\xa6\x10\x23\xa2\xf9\x28\x03\x52\xf8\x9b\x5d\x03\xd0\x08\xc1\x03\x03\x2b\x2b\x5c\x6b\x8c\xf7\xbe\xfc\x1f\xff\xfa\x9b\x55\x9a\x99\x57\x59\xa8\xd3\x3c\x6f\x49\xae\x57\x4a\x2d\x31\x80\x5a\xb0\x55\xe6\x46\xab\xed\x71\xb3\x0e\xcf\x73\x67\x03\x0b\xf2\x6b\x96\x2d\x41\xa2\xc7\xd7\x73\x5d\xdc\x0e\x5f\x1e\xda\x30\xb1\xae\x6e\xfe\xaa\xe9\xa4\xcf\x50\xb6\x85\x06\xc2\x1b\x12\xe3\xdf\x2b\x99\x3f\xea\xee\x44\x8a\x64\x43\xe6\x13\xcf\x53\x6e\x2a\x71\x1a\xa5\x26\x48\x71\x87\xb4\xfc\xd1\xfa\x68\x4e\x99\x47\x8c\x28\xb8\x4d\x9a\xf0\xeb\x6a\x49\x56\xc0\x37\x7d\x08\xee\x26\xeb\xd2\xd8\xd2\xf4\xce\x7e\x66\x04\x8d\xa3\xc0\x9c\x05\x38\xff\x8e\xfa\x17\x86\x90\xd4\x2f\x03\x41\xb2\x8a\x8f\xcb\x64\x9b\x53\x1a\x07\xaf\x1f\x21\xc4\x24\x32\x42\xe0\x45\xb1\x94\xa0\x4a\xd0\xf9\x2e\xdc\xe4\x82\xf3\x55\xf6\x69\x69\xcd\x90\x25\x4a\xb1\x59\xff\x9d\x9c\x0c\x66\x80\xf7\x8c\x99\x6d\x70\x48\xe2\xc5\xf0\x07\xad\x36\x21\x9d\x67\x2a\x0e\x76\xf1\xbf\x8b\xc8\x90\xfa\xa5\x6e\x49\x3f\x0c\x52\xd0\x9f\xa1\x26\x5c\xe5\x38\xe1\x66\x70\x9a\x00\xa2\xcd\x64\xe4\x5b\x9e\x5a\xca\xe2\xb9\x5d\xcb\x22\xbc\xfe\x96\x30\xe3\x2f\x37\xd0\xbb\x52\x9e\xfc\x8d\x29\x8c\x0b\xa7\xb8\xd6\x5e\x16\xde\xe9\x9a\xd7\x44\x6a\x39\x39\x46\x25\x87\x24\xd0\x8d\x84\x76\xe7\xf1\x6c\xcb\xc0\xe4\x26\x38\x38\x1a\x58",
        );
        test::<Sha224>(
            b"\xe3\x76\x56\xde\xfd\xee\xdf\xb4\x6b\x14\x62\x8d\xff\x3f\x69\x17\xb8\x42\x0e\x5a\x97\xef\x6c\x54\xaf\xda\x55\xe0\x7c\x60\x43\xdd\x75\xe7\x90\x8b\xe4\x66\xe9\x38\xf6\x29\x00\x1d\x0e\xce\x81\x83\x5f\x94\x48\x2a\xba\xd5\xd1\xea\xa4\xd0\xef\x9b\xac\xac\xc1\x33\xfc\xba\xe2\x2e\x2d\xfb\xe1\x33\x60\xe2\xf1\xf4\x8a\x5a\xe1\x56\x0f\x0b\x4e\xd2\x93\xd9\x17\x1a\x0c\xae\x11\x00\x1c\x7a\xfc\x94\x9f\x78\xb6\x8d\x80\xb2\xaf\xeb\xd0\xc7\x9d\xda\x19\xec\x71\xd8\xef\x31\x89\x1a\xc9\x06\x27\x2c\x0f\xfd\x22\xd9\x74\xd1\xdb\x4a",
            n,
            e,
            b"\xa9\x27\xec\x4c\xeb\x2e\xc1\x47\xcc\x45\x7e\x66\xc1\x2a\x64\x6f\xdc\x41\x2d\x9e\xeb\x1d\x51\xf3\xb5\xa3\xe5\xa8\xf4\xb0\xd3\x6d\xeb\xa3\xa7\x19\x14\xcc\x6f\x23\x21\xc3\x9d\x83\x4a\xdd\xb4\x85\x7c\x82\xab\xe9\x28\x0c\x7c\x82\x31\x89\x39\x04\xbd\x27\x47\x4c\xb2\xcc\xe1\x01\x2b\x92\x1f\x0a\x4d\x63\x80\xaa\xed\x61\x43\x56\xd6\x53\x65\x33\x88\xce\x86\xac\x71\xa2\x7c\x97\x67\x47\xc9\x21\x3c\xf2\x97\xe7\x59\xfc\x3e\x2d\x7b\x1a\xd5\xba\x8c\xb3\x10\x6c\x0a\x67\x62\x44\x79\xce\x55\xd0\xcd\x67\xc2\x4b\x5a\x45\xc1\x80\xef\xb5\x83\x0f\xc2\x0d\x87\xad\x3b\x15\x15\xe9\x0b\x77\xaf\x87\xf0\x6c\x6b\x0e\x71\x29\x71\x8a\x2f\x93\xae\xfb\xd1\x02\x8b\x1a\xc6\x3f\x6b\xd7\xec\xa0\xa0\x02\x69\xc0\x47\x3e\xaa\xc5\x57\x97\x51\x19\x50\xb1\x15\x25\xc2\x41\x41\xcb\x5a\xc4\xcf\xe2\xd9\xfd\xbf\xfc\xbd\xdf\x84\x12\xa7\x0e\xb1\xb8\xf4\x56\x48\x55\x3b\x70\x67\x58\x1b\xc8\xee\x2d\x6a\xa0\x89\xb9\x7e\x40\xdf\xe6\x1c\x33\xfa\xf9\xfc\xd5\x65\x0f\x61\x07\x85\x71\xf0\x3c\x6d\xf9\x4e\x01\xdd\x7f\x90\xf1\xdb\xea\xf0\x42\xd9\xbb\xc8\xb3\x63\x5c\x4c\x89\x93\x28\x52\xb3\x11\xf6\x3f\xf6\x19\x55\x0a\xab\xa0\x0f\x06\x14\x18\x88\x62\x24\xf8\x47\x87\x08\xf9\xec\xdb\xd9\x6f\x0f\x25\x15\x35\x31\x92\xad\x93\xd4\x6c\xfa\x8a\x4b\x3a\xc3\xea\xf7\xab\x9d\x1a\x3c\x4d\xfc\x62\x74\x6c\xeb\x08\x9e\xd3\xab\x40\x51\xae\x09\x27\x4f\x54\xf2\xa9\xc3\x79\xff\xe8\xc8\xc0\x10\x94\x87\xb6\x88\x3a\x48\x49\x41\x5c\x6a\x0c\xcc\xc6\x8b\x30\x96\x93\x8d\x6e\x54\x66\x9e\xda\xf7\xb8\x2e\xc9\x01\xc0\x53\x33\xe6\xc3\x10\x55\x41\xf0\x31\xab\x59\x04\x61\xe7\xf1\xf7\x76\xa2\x93\xe5\x93\xd0\x0d",
        );
        test::<Sha224>(
            b"\x99\xea\x30\xdf\xbb\x1e\xff\x6f\x56\xad\x6e\x0b\x05\x59\x89\xa2\xcb\xa1\x1f\xd3\x9e\x38\x6b\x00\x26\xb5\xf3\xa4\xc2\x8c\xb1\xe6\xcc\x3d\x71\x6e\x1e\xcb\x7a\x77\xd4\x70\x70\x25\x54\x8f\x79\x19\x8c\xea\x9f\x44\x7b\x15\x76\xf8\xf3\x2b\xfe\x45\x9d\xbf\xca\x82\x3d\x15\x62\x2a\x37\x92\xe7\xea\x53\x72\xf5\xf7\xbd\xb9\xcd\xa5\x50\x6c\xb4\x36\x13\x00\x96\xef\x04\x13\xef\x72\x15\x5a\xec\x47\x75\xdb\xcd\xbc\x10\x5c\x8d\xef\x59\x1b\xc5\x29\x47\xbf\xce\x6d\x9d\x8f\x25\x51\x6f\xe2\x14\x0d\xe2\xd6\x8f\xd2\x33\x45\x5d\x5d",
            n,
            e,
            b"\x69\x21\x0e\xe2\x7a\x00\xdf\xbf\xcd\x50\xaa\xf2\xeb\x50\x2c\x57\x06\xdd\xff\x6d\x9d\x23\xfb\x38\xd1\x11\x2f\x25\xc0\x47\xea\xac\x57\xdc\x90\xa6\xda\x67\x38\x76\x31\x9d\x5c\x04\x49\x4e\xce\x80\x37\xc2\xfb\x60\x20\x3c\x9f\x23\x32\x2e\x2c\x20\x63\xfa\x7d\x19\x16\x5e\xdd\xd8\x9e\x1b\x91\x93\x5a\x2b\x50\x02\x1e\x62\x68\x25\xbf\x19\xcc\x46\xaa\xeb\xfa\xb0\x9b\x49\x04\xde\xde\xf8\xc4\x63\x2a\xae\xdb\x42\x9f\xeb\x68\x7b\xba\xc2\xb4\x06\xf9\x23\xff\x1e\x84\x49\x41\xb0\xc0\x2b\x08\xdc\x2d\x8b\x42\x65\xfc\xeb\x61\xa8\x2f\xce\xf0\x62\x4f\x28\xee\xf3\xa9\x19\x3b\x86\xf1\x5f\x7a\xc4\x70\xdf\x59\x0a\xe8\x55\xa7\xaa\x75\x40\x49\x9d\xd4\x6a\x67\x85\x5a\x5b\xae\x6e\xc5\xdc\xa8\xb0\xc1\x6b\xcc\x69\xc0\xa1\xf9\x21\x8e\xc7\xcc\xae\x21\x7a\xc9\xb4\x7e\x8f\x7c\xae\xfc\x1e\x10\x2e\x3b\xdb\x42\xa6\x77\xfa\xbe\x18\x27\x4a\x5e\x69\x44\x7b\x33\x41\x4d\xf5\xbb\x29\xcc\xeb\x2a\xbd\x35\xc9\x4d\x36\x9e\xed\x25\x63\x02\xd7\x58\xdf\x99\x48\xbe\xe4\xef\xbd\xcc\x4a\xe3\x56\xe7\x8b\xe7\x35\xf7\x42\x5b\x64\x43\xcb\xff\x7e\x85\xc6\x53\xa6\x66\xde\xd2\xe7\x4e\xc7\xf6\x11\x03\xd6\xe8\xba\xc1\x10\xb1\x57\xae\xbf\x61\xce\x32\xf8\xb6\xf5\x67\xac\xbe\x92\xf6\xe3\xe2\x6e\xfd\xd3\x94\x2a\xf6\xc2\x79\xc2\xc7\xb4\xf1\x83\x98\xcc\x0a\xb4\xe2\x76\x88\x1b\x60\x46\xcc\x55\x25\x94\xcd\x96\x56\xf2\x2c\x3e\xe4\x98\x07\xcc\xe0\xf0\x9f\x2b\xfa\x7a\xbb\x87\x97\x27\xb7\x34\xdc\x19\xc4\x68\xf4\xaf\x4d\x72\x0d\xa8\xff\xd6\x50\xcd\xd6\x93\x82\x49\xb6\xa4\xc8\x47\xa5\x13\x83\x88\x8d\x12\x92\xa6\x16\x32\x22\x12\x6d\x5a\x42\xdc\xa6\xfb\x22\x83\xe7\xbb\xb6\xc2\x0d\x7b\x60\xb1",
        );
        test::<Sha224>(
            b"\x1e\xe4\x3d\xe2\xd8\x79\x7e\x65\xbf\xaf\x52\xc2\x5a\x0b\x00\xb6\xc0\x4e\x0e\x40\x46\x92\x05\x56\x5a\x96\x74\xd6\xaf\x57\x37\xbf\x9b\xe0\xf9\xa1\xbd\x62\xf8\x52\xe2\x87\x08\xe3\x29\x04\xdb\xd2\x66\x6e\x7f\x81\xc5\x4e\x28\xe7\xb0\x77\x30\x86\xf2\x97\x5c\x9d\x1c\x0d\x61\x9e\x27\xfa\xa7\xe2\x5c\x9b\x4c\x9c\x71\xd6\x6c\x0c\xf5\x12\xa0\xf5\xee\x4c\xc4\x50\xc0\x67\xfe\x42\x50\xc4\xfb\x4c\x6a\x13\x7c\xc2\x60\x69\x12\x7e\xf4\x92\x25\xd5\x78\xa8\x3b\xca\x34\xe4\x77\x82\x08\xb5\x60\xf8\x53\x0f\xe5\xf2\x13\x06\x9d\x34",
            n,
            e,
            b"\x3d\xd7\x22\xd2\xf0\x54\x3e\x66\x74\x3f\x8c\xdb\x60\x34\x1d\x61\xfd\x7b\x6e\xf8\xcb\x23\xa9\xe9\xf3\x40\x57\xd7\xa0\xaf\x49\xe3\x08\x26\xaa\x0a\xaf\x1f\xd3\x4e\xfe\xbd\xbf\xc9\x3a\xe5\x21\x27\x11\xa1\x60\xf2\xb8\x78\x6f\x4f\x5b\xec\xc4\x92\x09\xbd\x05\xdd\xf8\xde\x9f\xec\xd0\x0a\xf5\x30\x4d\x66\x15\x27\x2f\x2e\x49\x40\xbc\x8c\x39\xc2\xfb\xc6\x36\xf8\xc1\x05\x56\x5e\xc0\xf1\x57\x00\xcd\xb0\x66\xc5\xca\x1f\xd0\xe3\xe3\xf4\x94\x52\xe4\xf6\x71\x5a\x58\x22\x27\xd5\x9e\xc1\x04\x57\x5c\x17\x4f\x8c\xd1\x3e\xca\xbc\x4d\x58\x99\xe0\x2e\xbd\x3e\x81\xbd\x2c\x00\x32\x42\x73\x8b\x3b\x95\xb0\xe0\xcf\x0e\xf0\x2f\x8e\xe0\x28\x96\xdf\x64\x60\x68\xae\x23\x3f\xfc\x44\x36\xf1\xe9\x7d\x37\xd4\x5d\x49\x7e\x1a\x54\xa0\xd6\xfc\x5a\xaf\x27\x5e\xc5\x0c\xbf\x0b\x40\x20\x52\x20\x0f\x6b\xc3\x53\x73\x82\x8b\xcd\xb4\x8a\x17\x8c\x96\x88\x65\x8a\x23\x63\xa8\x68\x3a\xb9\xea\xfa\x97\x90\xee\xf2\xc7\x9d\xa1\x48\xa9\xd9\x95\x39\x5d\x9f\x6a\x7b\x31\x0f\x6f\x71\x41\xd3\xcb\x0f\x20\x6e\x8b\xaa\x82\xa3\x38\xd5\x19\xee\x88\x1c\xf6\x1d\x5e\x1f\x90\x6d\x42\xc2\xe8\x5f\x25\xcd\x19\xd9\x86\x4a\xb5\x4a\x32\x96\x9c\x8e\xdf\x29\xe5\xac\x52\xf6\x20\x06\xd9\x21\x9c\x21\x14\x00\x07\xb0\x5c\x63\xe3\xba\x4c\x04\xec\xe5\xd8\x80\x50\x26\xdb\xe8\xff\x66\x52\x52\xd5\x37\xd0\x13\xf7\x09\xd8\x49\x99\xf8\x4b\x43\x82\xa8\x94\xc1\xba\x03\x18\x49\x37\x83\xa5\x98\xf6\x37\xbc\x2d\x8d\x56\x78\xcf\x65\xd0\x38\x33\x80\xad\xa0\xdb\x5a\x51\x07\x37\xa8\xb7\x0c\x3b\xae\xee\xe4\x70\x85\x08\x8e\x96\xd9\x94\x38\xba\x5e\x98\x87\x88\xf2\x88\x6a\xa7\xe2\x95\xd8\x57\x8e\xb2\x7f\x1d\x68\x38",
        );
        test::<Sha224>(
            b"\x74\x03\x22\x95\x3b\xfc\x8e\x84\x0c\xec\xd9\x96\x3f\x58\xbe\xa7\x0d\x2b\xd2\x0a\x50\x63\x66\xd7\x8a\x0b\xad\x86\x29\x69\x22\xbd\x09\x78\x24\x67\x3b\x99\xe3\x06\x05\x85\x80\x4a\x29\x86\x70\xe2\x6a\xe7\x22\x92\x4d\xa8\xe1\x86\x1d\x77\xfb\xe6\x31\xdc\x23\xaa\x72\xb4\x14\xb0\x17\xe0\x77\x0b\xb3\x3e\x07\x9f\x72\xd8\xf3\xeb\x9f\x5c\x83\x25\x6a\xcd\xff\x8c\xe9\x77\xcd\xfa\x5c\x28\xd9\xb8\xb5\x9d\x3a\x97\x58\x3b\x12\x3c\x1f\x00\xb5\xbc\xa1\xb8\x0e\x69\xb4\x74\x3f\xeb\x30\x38\x88\x92\xf6\xf4\x6a\xea\x31\xb5\x0c\x90",
            n,
            e,
            b"\x7c\x41\x48\x40\x91\x0c\xa0\x8f\xec\xd2\x3f\xf1\x2c\xee\xbc\xd4\x8b\x7a\xfa\x4e\x6a\x87\xa4\x06\x54\xba\xae\xc6\xc9\x05\x00\x87\xb1\xf0\xb6\xfa\x04\xe3\x6c\xd5\x95\xad\x29\x3d\x08\x27\xe9\xe1\xc9\x4f\xe0\x33\xec\x42\xbb\xd0\x21\xf7\xce\x2e\x75\xda\x6d\xd2\x06\xb9\x91\x51\x76\x8d\x6d\x5a\xe7\xb1\xf0\x44\x16\x80\x4c\x2a\xd7\xc6\x74\x4c\x73\x43\xc8\xf0\x1b\xe2\x59\x36\x1c\x11\x68\x10\xf0\xad\xa1\xc6\x43\x48\x05\x5b\x25\x94\xa0\x6b\xdc\x08\xdb\x39\x0d\x75\x0e\x4a\xee\xa5\x43\x59\x32\xa0\x4d\x0e\x69\xd5\x90\x63\x04\xc8\x4e\x19\xd5\xfb\x86\x88\xca\x25\x98\xb6\xfa\xe6\xd1\x69\x59\x3f\xac\x29\x09\x23\x8c\x55\x3c\x66\x4d\xe9\x2c\xba\x6d\x89\x15\xe0\x1a\x6e\x99\xd8\xd9\x2f\xec\xbc\x6e\xae\xfd\x93\x15\x1c\x61\xfb\xbd\xe2\xea\xcf\x26\x34\xe7\xb6\x11\x6a\xd2\xfe\x88\x59\xb6\x5a\x70\x66\xd7\xb5\xb7\x76\x38\x65\x0b\x60\xa4\x3d\x82\x77\xda\xb0\xac\xa1\x45\x06\x5b\x3c\xf0\x0d\x96\x3b\x7f\x81\x8d\xda\xdd\x7c\x54\xbe\x5b\x4b\xa7\x69\xae\x01\x34\x46\xa5\x74\xdb\xbb\x8f\x7c\x22\xb2\xd1\x54\x3e\x7b\x5e\xc0\x8d\xfd\xe3\x8e\xf9\xad\x84\x3c\x1b\xb6\xd9\x55\x8a\xef\xcd\x45\xd3\xb1\x2c\x82\x06\xb7\x92\xca\x72\xbf\x49\x50\xbe\xfb\xee\xc0\x4f\xc1\xa2\x8c\x37\x20\x58\x85\x13\xa2\x9a\xf9\x69\x1d\x2f\x31\xdd\x7d\x39\xa5\x6b\xcb\x5f\x49\x9f\xb1\x4c\xa4\x7f\xa5\x41\xe2\xea\x67\x84\x33\x99\xe0\xc8\xab\x89\xc8\x1e\x58\x93\x41\x59\x42\xbf\xe4\xe4\x70\xa6\x78\xc0\xe5\x61\xed\x64\x55\x47\x11\xb1\x6b\xe3\x35\x0c\x98\x5b\x61\xf2\x92\x19\xc5\x27\x4d\x87\x93\x08\xdd\x25\xfc\x03\x3f\x81\x9c\x38\x59\x04\x65\x43\x99\xe5\x43\x8f\xd9\xc8\xcf\x1e\xc7\x6e\xcc",
        );
        test::<Sha224>(
            b"\xf7\xe3\x78\x20\xa1\x9d\x5f\x6a\x05\xeb\x47\x79\xc2\x40\xe7\xfb\x58\x6a\xe8\xc3\xdf\x71\x3b\xcd\xf9\xc2\xaf\x7c\x05\x8c\xc3\x27\x95\x6b\xb8\xd4\x22\x44\xeb\x43\xff\x70\x62\x2f\x8c\x1c\xa5\xd0\xac\xef\xcf\xa4\x79\xee\xe4\x6f\x36\x9d\x65\x81\x84\x67\x22\x37\xd9\x40\x50\xc4\x2f\x89\xdb\x31\xf9\x34\xfe\xa3\x5b\x28\x10\xdd\x9a\xe7\xa1\x05\xd2\x6e\xc5\xab\xe7\x5d\xb0\x07\xbd\x57\x83\x82\xac\xac\x66\x79\x2e\x35\xd7\x3d\xdb\x80\x41\x5e\x98\x2d\xd1\x29\x0b\x98\x85\x6f\x52\xb9\x86\x88\xf4\x48\xb7\x98\x17\x24\x8e\x11",
            n,
            e,
            b"\x56\x3e\x22\x61\x7d\xd8\x89\xe7\xbe\x8d\xd2\x6a\x17\x6e\xe9\xf6\x7b\x9b\x3e\xb0\x40\xad\x7a\x7f\xab\xc0\x89\xb2\x7e\xd4\xe7\xa7\x82\xf1\x52\x2b\x44\x6f\x42\xa5\x67\x49\x21\x37\x77\x0c\x61\x2d\xc5\xe4\x28\xec\x28\xa3\xc5\x02\xaa\x25\x08\xfb\x46\xb7\x03\xd7\x9d\x1f\xde\x8e\x1a\x50\x7d\x70\x62\xe2\x64\x40\xb3\xa3\xff\x16\xbc\x82\xfc\xc9\xb3\x01\xf2\xb5\x8f\xa8\x18\x52\xb5\x7f\x95\x1d\x92\x51\x64\xbe\x0c\x70\xbd\x28\x1d\x72\x6c\x9a\x71\xb9\x84\x28\x03\x52\x28\x9f\x8c\x1b\x39\x4a\x85\xdf\x9e\x17\x32\xa4\x53\x9a\x30\xa7\x59\xe8\xf1\x26\x09\x6b\xf7\x3f\x7b\x25\xa5\xed\x34\xc3\x2a\xf3\x45\xbc\x32\xe4\x12\xe0\x8b\x6c\xa9\xb6\x56\xa6\x92\x85\x19\x65\x5e\xc9\x76\x9c\xf1\xda\xe7\xc9\x85\x50\x5a\x81\x2e\xe4\x4b\xb3\xb4\x2e\xcb\xec\x91\x1b\xec\xed\x8f\xe8\x73\x65\xf1\x13\xaa\xc0\x0a\x65\x9c\x0e\xb3\x7b\xfe\x75\x36\xf9\x17\x6a\xfe\x9c\x45\x9a\x08\xae\x23\x60\x0d\x4c\x85\x43\xef\x3c\x3a\xf4\xcd\x10\x11\xe0\x8f\xdc\xf1\x99\xba\x49\x02\x4f\x08\x80\x8c\x47\x59\x86\x87\x05\x61\xd6\xa0\x88\xb7\x9c\x38\xae\x8c\xe0\xe6\xec\x40\x26\x8b\xc9\xfb\x7a\x3b\x61\x85\x87\xf5\x5f\xbc\xd3\x1c\xea\x93\x70\x24\x38\x65\x49\x2e\x5f\x13\xc9\xfd\xad\x61\xf4\x0b\x32\xd3\xa9\x15\x19\x42\x44\x94\x9a\xdd\x15\x02\x6c\x0a\xe1\x9f\x52\xad\x5b\x70\x36\x5e\x77\xf2\xcf\x53\x29\x8c\x9e\x2b\xad\x06\x17\x1b\x09\x08\xdf\x26\xb2\x2e\xf1\xc7\x37\xc3\xb3\x21\x39\x5f\xfc\xdb\x71\xc8\x22\x8f\xe9\xde\x02\x7f\x0d\x31\x06\x86\xb1\x68\x3a\x67\x41\x9e\xa0\x89\x71\xcf\x0b\xf1\xa3\xe5\xa1\x07\x27\x24\x83\x46\x01\xd5\xf9\x44\xfa\x23\xf7\x7d\x8e\x77\xe8\x87\xf8\x8d\xdb\xee\xb1",
        );
        test::<Sha224>(
            b"\x87\x10\xa8\x77\xb7\xa4\xc2\xe5\x78\x79\x3b\xd3\xe4\xd1\x9c\xb5\x6d\xe9\x7f\xcd\x1f\x2a\xf5\xfb\x25\xa3\x26\xd6\x8f\xb7\x37\xfb\x52\x13\x71\xa6\x90\xe4\x9f\x7f\x1a\x46\xb7\xb6\x34\xff\xbd\x51\x98\x6d\xe5\xc5\xbd\xbd\xf8\xc4\x58\x5e\xf8\x57\x24\xb5\x07\x2c\xde\x13\x85\x31\x94\xe4\x79\x62\x20\x29\x32\xde\xf0\x28\x2e\x41\x08\x61\x3a\x2e\x49\xc5\xdb\x2b\xf3\x23\xed\xb2\x69\xe3\x8a\x84\x34\xf6\x2d\x41\x4b\x0d\x17\x36\x91\x09\xf2\x76\xa0\xb3\xb5\x2c\xc5\xae\xc7\x2f\x4b\xaa\x67\xd7\xfd\xd9\x4b\x10\xe6\xa7\x87\xac",
            n,
            e,
            b"\xa7\x83\x58\xef\x28\x30\x3d\xeb\xa1\xbf\x1b\xc3\xca\xe5\x9a\xb0\xff\x66\x14\xc5\x20\xee\xb7\xd8\xc8\xfd\x5c\xed\x34\xda\x74\x54\xad\x14\x0b\x53\x9e\xf7\x5e\x2d\x65\xdd\x89\x1e\xbf\x89\x9a\x88\xad\xa2\x5b\xcc\x35\x72\x60\x53\xda\x68\xe2\xe0\x2b\x6a\xcd\x2e\x7e\x21\xcb\x8b\x37\x35\x5d\x19\xbd\x4c\x3e\x36\xa8\xc1\x64\x7e\x1a\x38\x4c\x8a\xd2\xab\x39\xbd\x22\xf3\xd3\x0f\x0f\x9d\xd6\x85\xfe\x4d\xd7\xf8\x36\xec\x46\xbb\xce\xf0\x80\x5d\x08\xa7\x84\xa6\x96\x4c\xd5\x0f\x58\x07\x1e\xd7\x9f\x88\x24\x91\xa3\x31\xb4\x45\x39\x0b\x43\xf2\xa2\x95\xa1\x3a\x28\xce\x0f\x44\xbb\x6d\x63\xf3\x19\xd8\xde\x90\xe3\x90\x17\xf4\xcb\xc1\x45\x33\xda\x33\x38\x0f\x55\x3f\x09\x7e\x79\x6a\x67\x1b\xa2\x9c\x94\x58\x2c\xd5\x19\xf1\xf6\x4d\xb3\xbe\x89\x4b\x66\x15\xf6\x84\x4f\xf2\xfc\x62\x10\x13\x82\xb0\x44\xf5\x85\x6b\x9b\xb9\x78\x71\xcf\x13\x7c\x4e\x9e\x48\x4e\x84\xa3\xcd\x2d\xae\xa8\xe1\xc6\x35\x8d\x66\xcd\x83\x26\xc1\x92\x5c\xe1\xf7\xd2\xd2\xe9\x04\x57\xad\xaa\x65\xec\x3a\x67\xf4\x86\x5b\xf6\x12\x0e\xff\xa0\x6a\x79\xde\xb6\xb6\xca\x9f\x85\xc9\xdd\x96\x7f\x2f\x31\xa2\x2d\x5d\xb2\x5b\x15\x53\x0a\x9e\x85\x0a\xca\x48\x6b\xc0\xca\xc2\xbe\x6b\x0a\xf6\x6e\xcb\x56\x8c\x09\x55\xa3\x04\x95\xbd\xd5\xd0\x5a\x22\x0c\xd0\x6c\xb0\x6f\x04\xf2\x16\x07\x6a\xaa\xd4\x38\x2a\x94\x04\x0d\xcc\xda\x68\xa1\x9d\x55\xb4\x93\x38\xc9\x31\x5a\xa8\x02\x91\x06\x55\xfe\x93\x94\xaa\x73\x59\x0a\x6b\x2a\x04\x39\xbb\xef\x5e\xc7\xcc\xb5\x20\xf2\xc5\xcb\x71\xd3\x93\xa6\xcc\xe2\x5b\xf7\x7d\x80\x33\x44\x4f\xb3\xda\x8a\xc8\x61\xc6\x3d\xc2\x56\x1f\xfd\xcc\xe8\xc2\x06\x5b\x35\xb5\xc8\x3b",
        );
        test::<Sha256>(
            b"\xbc\xf6\x07\x43\x33\xa7\xed\xe5\x92\xff\xc9\xec\xf1\xc5\x11\x81\x28\x7e\x0a\x69\x36\x3f\x46\x7d\xe4\xbf\x6b\x5a\xa5\xb0\x37\x59\xc1\x50\xc1\xc2\xb2\x3b\x02\x3c\xce\x83\x93\x88\x27\x02\xb8\x6f\xb0\xef\x9e\xf9\xa1\xb0\xe1\xe0\x1c\xef\x51\x44\x10\xf0\xf6\xa0\x5e\x22\x52\xfd\x3a\xf4\xe5\x66\xd4\xe9\xf7\x9b\x38\xef\x91\x0a\x73\xed\xcd\xfa\xf8\x9b\x4f\x0a\x42\x96\x14\xda\xba\xb4\x6b\x08\xda\x94\x40\x5e\x93\x7a\xa0\x49\xec\x5a\x7a\x8d\xed\x33\xa3\x38\xbb\x9f\x1d\xd4\x04\xa7\x99\xe1\x9d\xdb\x3a\x83\x6a\xa3\x9c\x77",
            n,
            e,
            b"\xd1\xd2\x1b\x8d\xfa\x55\xf0\x68\x1e\x8f\xa8\x61\x35\xcf\x29\x2d\x71\xb7\x66\x97\x13\xc2\x91\xd8\xf8\xdc\x24\x64\x64\xde\x3b\xbb\x96\x1b\x59\x6d\xfc\x8f\xda\x6c\x82\x3c\x38\x40\x08\xd0\x5b\xcb\x3d\xcc\xc3\x6a\xcc\xf1\xb2\xbe\xde\x1a\x95\xe5\x22\x58\xd7\xd1\xbd\xf1\xfc\x44\xe1\x80\x72\xab\xd4\x5c\x13\x92\x01\x5e\xe7\x16\x92\x69\x0e\xf8\xcd\xaa\xed\x33\x7d\xd8\x54\x67\x83\xf9\x61\xbb\x96\x20\xeb\x5c\x7b\x8b\x67\x16\xe8\xc6\x00\x35\x1f\xab\x77\x65\xee\x38\xa1\x5d\x32\xd8\xa2\xc0\x94\x98\x25\xc4\x9a\x7f\x25\xee\xdd\x9b\xe7\xb8\x07\xbb\xfd\x51\x79\x13\x78\x66\x20\xd2\x49\x82\x3d\xae\x6f\xe2\xfd\x39\xac\x63\x9d\xd7\x48\x21\xb0\xc1\x20\xb4\x2f\x31\xc2\xc6\x39\xd2\xc6\x1b\x39\x5f\x09\xf8\x68\x51\xbc\x80\x9b\x34\xc4\x98\x1a\xc6\x5c\xf2\x5b\x2e\x8a\xdc\xbc\xe1\x90\xef\x2e\xf6\x7a\x01\x89\x03\x9c\x91\x10\xf2\x67\x01\xc3\xee\xd7\x31\xc8\xd9\xea\xd1\x78\x22\x0f\xfc\xac\x7f\x0f\x67\x8a\xa2\x22\x68\xe1\xd0\x19\x42\xec\x51\xe8\x0e\xef\x06\xe2\x11\x28\x30\x85\x5e\x87\xba\xfe\x8c\xc9\xc2\x2f\xd7\x37\xc7\xab\xbc\xa5\xeb\x7a\x22\x1d\x38\x35\xa8\x66\x10\xd2\x4b\x50\x7b\x5d\xcb\x46\x18\xaa\x42\x1f\x63\xa5\x60\x9e\xf5\xd6\x8f\x57\x60\xfd\xdf\x97\x01\x35\x60\x2e\xfa\xd0\x85\x1b\xbf\xf9\x8f\xe8\x7f\xa5\x8b\xc3\x65\xf3\x8e\xe7\xec\x8e\xf5\xaa\xb1\x7f\xd1\x1d\x89\xd9\x1e\xf4\xc6\x04\xe0\xd1\xf0\x01\xd0\xe0\x88\x69\xdf\x92\x25\xe3\xb4\xce\xf5\x2f\xf8\x68\x15\xe1\x3b\x3e\xfd\xf4\x57\x76\xf9\x35\x37\x69\xa8\xa5\x1f\xe7\xd8\x91\xa7\xef\x70\x35\xee\xcf\xa2\x59\x84\x87\x38\x37\x68\x86\xed\xc9\x1c\xc7\x8f\x6d\xa3\x1c\x2f\x07\xee\x36\x2c\x3d\x82",
        );
        test::<Sha256>(
            b"\x2b\xca\xd6\xe7\x44\xf2\x49\x0b\xa6\xa6\xe0\x72\x28\x32\x41\x7e\xbd\x91\x0f\x91\x46\xeb\x62\xba\xaa\x5c\x74\x95\x29\xf7\x9d\x6c\xed\x0b\x81\xa2\xe2\xa4\x88\x52\xc8\x55\x8e\x33\x87\x35\xdc\xbf\xc2\x28\x57\x94\xae\x60\xf8\x1a\x25\x23\x7c\x66\xf6\xce\x5d\x5e\x80\x1a\x00\x1e\x7f\x9e\x30\x9b\x25\x95\xcb\x86\x6d\xe2\xbb\x74\xac\x51\x28\x3b\x68\x20\xec\x9f\x6e\xbe\x48\x2e\x1f\xd2\xd5\x68\x0b\x7f\xbd\x23\xc1\xe6\x2a\x2e\xe4\xed\xff\x35\x82\x3f\xc7\xe4\xa2\x95\xea\x4f\x1c\x33\x27\x92\xae\xb5\x3e\xb4\x4b\x0b\xed\xd2",
            n,
            e,
            b"\x37\xd9\x60\xfe\x39\x12\x98\xbb\xdc\x22\x3f\xa1\xeb\x1d\x3c\xd9\xa4\x6b\xa8\xc6\x2e\x1d\xa8\xc5\x63\xc8\x9a\x8f\x0e\x67\xb8\x64\xfc\x89\x83\x7f\xfc\x08\xaa\xb7\x12\x2b\x84\xc4\x35\xc7\xf9\x40\x6e\x16\x5a\x10\x29\x85\x7c\x1e\x4d\xea\x65\x35\x69\x27\x72\x73\xb1\xd9\xb0\xa9\xf5\xb0\xdc\x24\xaf\xdd\x21\x44\x76\xd4\x72\x08\xad\x52\x21\xa7\xd7\x93\xca\xb8\x06\x71\xfb\x49\x87\xc8\x6b\xd6\x14\x48\x80\xc5\x9d\x24\x87\x14\x00\xf6\x4b\xdc\x6d\x49\x6d\xbd\x49\x7f\x3d\xbf\x64\x28\x64\xfe\x49\xaf\x3e\x21\x51\x5e\x62\xd6\x0f\x00\x71\xdb\x48\x84\xf4\x96\x70\xea\xa9\xe4\xe4\x98\x2f\x26\x9a\xbe\x72\x42\x44\x28\x88\x59\xc2\xad\xf6\x0a\x09\xfa\xaa\xbb\x07\x99\x0e\x09\xe5\x6d\xe2\x54\xba\xbb\xee\x14\xbe\x7e\xb6\xed\xa0\xcd\xb2\x2f\x3d\x0d\xe8\x72\x48\x04\x67\x3f\xb9\x9f\x86\xef\xb4\x26\x3d\xcc\x50\x17\xab\xc9\x1b\xd9\xcd\x83\x36\x79\x47\x5b\xfa\xc5\x0a\x2b\xe8\xdb\x86\x29\x6b\xbf\x80\x17\x88\x93\x57\x37\x13\x14\x60\x4e\x83\xd6\x8b\x6e\xfe\xcd\x4b\x79\xf0\xa8\xaf\xa0\xdf\xfa\x44\x8f\xb7\xfc\xe6\xd3\x44\x70\x9a\x67\x0e\x0c\xff\x43\x2c\x3e\x18\x7b\xcf\xf7\xfd\xc4\xf4\xe9\xab\xe1\x09\x5c\x46\xb0\x1d\x88\xb6\x04\x4b\xb9\x50\xe9\x28\x59\x01\x0d\x9a\x0e\x3b\x2d\x1f\x27\xa0\x96\xea\xca\xa2\x42\x63\xa2\xa0\x52\x3d\x6e\x0d\xa1\xfb\xa8\xaf\x76\x81\x96\xf7\xa5\x1f\x92\xfd\xf1\x52\xbe\xf0\x62\xdd\x1f\x83\x27\xce\xe1\xd3\x44\xc2\x00\xc2\x11\x5a\xc6\xec\x1d\xd8\x51\x4c\xef\x9e\x36\xd0\xce\x8c\x32\xe5\x87\x83\xc4\xfc\xba\x90\x1a\xa7\x0c\x2b\x42\x96\x64\x88\x00\x2f\xf1\x71\xd3\x64\x14\xa1\x44\xbf\x46\x77\x51\x83\xa8\x81\x5d\xe9\xee\x3e\x81\xf3\x1b",
        );
        test::<Sha256>(
            b"\xc3\x97\x8b\xd0\x50\xd4\x6d\xa4\xa7\x92\x27\xd8\x27\x0a\x22\x02\x95\x34\x82\x87\x59\x30\xfb\x1a\xea\xe4\xe6\x7f\x87\xe7\x94\x95\x28\x9d\xe2\x93\xb4\xa4\x0d\x92\x74\x6f\xc8\x4c\xc8\x31\x8c\x23\x18\xfd\x30\x65\x0e\x2b\xb9\xce\x02\xfd\x73\x4e\xb6\x83\x41\x0d\x44\xbb\x31\xad\x54\xfd\x53\xcf\x92\x96\xcc\xd8\x60\xb4\x26\xf5\xc7\x82\xea\x5c\xb4\x93\x71\xd5\x61\x84\xf7\x79\x11\xdd\xf1\xba\x00\x39\xa0\xa4\x9a\xa7\xe7\x63\xeb\x4f\x5a\x04\x57\x59\x97\x80\x8b\x0a\xd9\xf6\xb3\x30\xca\x38\xed\xc1\x99\x89\xfe\xbf\x4d\xa5",
            n,
            e,
            b"\x9a\xed\x20\xa8\xbd\xaf\x26\xf1\xf1\x19\x02\x0d\x8f\x3e\xa6\xce\x91\x51\x38\xd4\xc8\x7d\xce\x02\x5e\x7f\x4e\x49\x53\x6c\x8e\xc0\x79\xed\xc6\xca\xf0\xd6\x03\xbf\x42\xbd\x6a\x45\x4a\x6d\x52\xd0\xd9\x9f\xd0\xf5\x9f\xfb\x3b\x22\xe9\xe6\x7b\x3d\x0b\xb2\xd2\x75\xd9\xae\xdc\x6d\xa9\x6a\x72\xcb\xff\x35\xc4\x3e\x7f\x39\xa9\x96\xfa\x8a\x6d\x33\x8a\x07\x25\xf7\x85\x25\x4f\xe9\x1a\x20\x83\x4b\xa5\x57\xfe\xdf\xe7\x15\x2b\x99\x56\xfe\xdd\xfd\x94\x17\x41\xef\xf9\x17\x7c\x2f\xbb\x55\xe2\x00\xbb\xe4\x21\x62\xb3\x2a\x94\x0c\xc3\x00\xab\x37\x55\x57\xdf\xfd\x48\xdf\xa5\x39\xf5\x0e\xdd\x52\xdf\x15\x8d\x90\x72\xd1\x49\x82\xe9\x63\x03\xbc\x61\x2c\x2c\x25\x06\xdb\xca\x3a\x93\x9d\x62\x6d\x2e\x7f\xb4\x44\xc6\xad\x7d\x8d\x9f\x3b\xba\x82\x10\xb2\xac\x2f\x69\x67\x83\xc3\x49\xfc\x52\x80\xc1\x05\x40\x2a\x4b\x3d\x86\xbe\xf5\x02\x6c\x3d\xd9\x99\xe3\xb2\x23\x80\xf9\xdc\xce\x40\xe3\xa9\xcc\x9f\x1d\x7b\xc3\x8e\xf3\xdd\x7e\x94\x13\xbb\x57\x98\x00\xc0\xe6\xc3\xe9\xab\x91\x2d\xa8\xfe\xc1\xa4\xab\x21\x39\x8e\x96\x80\xba\x0d\x04\xf3\xb4\xc8\xd5\x3c\x02\xf0\x5c\x7a\xe4\x9b\x70\xa5\x61\x1c\xf8\x2e\x38\xde\x84\xaa\x8c\x24\x26\xf0\xb6\x3e\xa0\x1b\x28\x9f\x20\x1d\x3a\xf4\x0d\xad\x5d\x6e\x5b\xcc\xc7\x5b\x99\x59\xe5\xc9\x75\x8e\x79\x10\x5a\xf7\xa9\xaf\xb1\x2a\xee\x57\x7c\xb3\x99\x18\x79\xdb\x0f\xd8\x66\x2c\x5b\xc4\x90\x22\x75\x24\x98\xa3\x01\xd9\x5f\x4b\x1d\x08\xc0\x1e\xbc\x31\x3f\x89\xc0\x0b\x1e\xc2\x73\x5a\x07\x98\x3f\xd5\x28\xe6\x38\x82\x45\x03\x6f\x0e\xd4\xa2\xdb\xb6\x5d\xd3\x3a\xb7\xf1\x24\xc0\x14\xec\x16\x79\xf1\xc2\xf1\x1e\xdf\xfb\x93\xfa\x2d\x1d\x73",
        );
        test::<Sha256>(
            b"\x0c\x11\x95\x02\xc2\xa0\x19\x20\xa0\x90\xe4\x33\x57\xe7\xb2\x8e\x33\xc7\xee\x85\x8b\x43\x30\xe0\x5c\x71\x04\x89\x31\xc0\xed\x88\x46\x8c\xa9\x31\xec\xf0\xb7\x9c\x2f\xdc\x17\x56\xb7\x67\x51\x56\xec\x66\xb8\x33\x5e\x3d\xf0\x94\x63\xf5\xae\xe7\x02\x8f\xbf\x56\x0f\x98\x4c\xf6\x98\xfe\x5c\x42\x80\x22\x9a\xc9\x6a\x2e\x59\x23\xd8\xa9\xd5\x29\x94\x49\xbb\x66\x50\x08\xec\xc8\x89\x79\x7e\x9b\xb1\x5d\x04\xb8\x8c\x72\x10\xfa\xdb\x8b\xf6\xf2\x38\xe5\xd2\xdc\x41\xb9\xcc\xd1\xf8\x0e\x9a\x3e\x6a\xd1\x47\x94\x8f\x27\x33\x41",
            n,
            e,
            b"\x8a\xbf\x2a\x30\x77\x4e\x6e\x73\x38\xec\xa0\x9c\xcc\xac\xa3\x68\x43\x99\x94\x04\x92\xfb\x94\xb2\x3b\x5a\xd6\x2c\xe3\xe1\x1d\x2d\xbe\xf8\x96\x6b\xa5\x26\x99\x79\xeb\x96\x53\xba\xad\x71\x95\x16\xd3\xe8\x39\x90\x79\xa2\xf6\x70\x27\x5a\x2e\xd4\x2c\x82\x0a\x9a\x31\xfc\xd7\x03\xa7\x66\x37\xe0\xd7\x13\xf3\x2d\x79\x2b\x9a\xe3\x6d\x72\x88\xf6\x0c\x2d\x1a\xe5\x26\x83\xbb\x15\x94\x1b\x1c\xd8\x90\xd2\xcd\x64\x99\x8b\x77\x25\x85\xe7\x60\x32\xa1\x70\x2e\x06\x52\xcb\xf2\x59\xa1\xce\xae\x69\x5d\x40\xcf\x2f\x4f\x6d\x81\x34\x1c\x8b\xc9\x08\x2c\xb9\x6c\x75\x2c\x35\x5d\xfb\xe2\x96\xdd\x21\xd6\x98\x46\xfa\x37\x61\x3e\x73\x81\x7b\x2a\x07\x04\x66\x58\xc9\xe3\xfc\x6d\x09\x1e\x17\x59\x1b\xb1\xa4\xfb\x6e\x2a\xc0\x0a\x31\x94\xc1\x48\x8e\x16\xa9\xd2\x90\x37\x86\xdb\x86\xae\x90\xe9\x6a\xcb\x4d\xe9\x90\x1a\xaf\x1b\x06\x51\xfb\x76\xa5\x8d\xcb\x3d\xb4\x73\xef\xbf\xb8\x31\xef\x8e\x30\xf8\x99\x67\xdd\xd3\xa6\xc2\xf1\x89\x79\xa0\x45\x06\x57\xcd\xae\xef\x6e\x59\x37\x7c\x6d\xb1\xec\x46\x06\x5f\x61\x40\x24\xa6\x9c\x51\x8a\x55\x99\x42\x59\x4a\x46\x26\x6e\x0d\x3c\xa1\x33\x42\x96\xb9\x68\xa2\x3a\x4b\x11\xc6\x3a\x97\xe2\x9e\xb1\x6b\x24\xc0\x2d\x54\x5d\x5b\x42\x7e\x6a\xa5\x85\x33\x33\x18\xe6\x3a\x20\x45\x24\xe0\xe4\x2a\xc1\xed\xb7\x0d\x34\x56\x78\x0d\xbe\xad\x31\xf7\x85\xf0\xb2\xa7\x7f\xfe\xb0\xd3\x73\x84\xcb\x5f\x65\xb4\xe3\x6c\xa2\x41\xf3\xb2\xb0\x59\x10\x5f\xaa\xa3\x22\x2d\x6c\x13\x5e\xa5\xa3\x66\x51\xae\xa3\x96\xd2\x2f\xc4\xea\x1b\x40\x4d\x7e\x83\x4b\x6d\xf1\xfb\x83\x8b\xb5\xba\x0d\x78\x4a\x96\xe2\xae\x28\x43\xdb\x3e\xee\xa4\x96\xc7\xad\x2b\x42\x41",
        );
        test::<Sha256>(
            b"\xdd\xbd\x84\x68\xbd\xb0\x36\xf4\x79\x9f\x42\x8b\xc8\xb4\x37\x4e\xd9\xb7\xcd\xe5\x41\x33\x7a\xc4\x39\xd4\x41\xac\x06\x14\xcb\x75\xb8\x16\xb8\x0c\x17\xd2\x37\xb8\xdb\x73\xd4\xa1\x1b\xfd\x92\x92\x08\x33\x3a\xfe\xdb\xb8\xf2\x41\x0c\x74\x11\x29\xc5\x39\x32\xb5\x96\xa7\x88\x1c\x6a\x4d\x71\x11\xba\x10\x4d\x46\x00\xd1\x90\x2f\x6f\x4a\x16\x08\xe1\x39\xb7\x19\x11\xc1\x1c\x39\x0a\x0d\xd0\x91\xdf\x36\x9a\xa2\x9d\x67\x0b\x8a\x7e\x3f\x53\x82\x5f\x76\x59\xac\x74\xc4\x0a\x0c\x3b\xfe\xf0\xd3\xae\x83\x07\xe4\xbd\xd6\xcd\x91",
            n,
            e,
            b"\x4e\x37\x7e\x24\x59\x81\x5d\x5b\x33\x91\x5f\xa6\x3c\xd4\x77\xb5\xbe\x7c\x6b\x7f\x78\x14\xd1\x35\x00\x34\xce\x71\x0b\xe6\x7e\xd6\x91\x39\xdb\x62\x2e\xf6\x0e\xc6\xb7\x63\x8e\x94\xb2\x02\x36\x8b\xac\x63\x1e\x05\x77\x02\xb0\xe6\x48\x7b\x32\x4a\x6b\x98\xed\x7e\x03\xd1\xf3\xf2\x0a\x98\x14\xb0\x0e\x21\x7a\x46\x48\xe4\xbb\xc4\x49\xa2\xaf\x40\x5c\xa4\xb5\x9f\x84\x38\xdd\xfd\x75\xd3\x4d\x10\x64\xe5\x8b\xfb\x32\x5c\x55\xbd\x54\xea\x6c\xdf\x77\x12\xba\x80\x7c\x3e\x4c\x66\x5d\x62\x0c\xd5\x95\x13\xd7\xbc\x08\x55\x24\x7e\xb6\x70\xec\xc2\x92\x50\x96\x61\x81\x27\x02\x70\x32\x75\xd9\xb2\xf8\x7e\xf2\x79\xd7\x70\x0e\x69\xd9\x95\xdb\x98\x14\x4a\x14\xc8\x17\x74\xa4\xcd\x89\x0e\xc0\x3d\x13\xf8\x58\xf3\x76\x9e\x50\x48\xed\x55\xca\xa8\x12\x01\xe8\x78\x5d\x37\x71\xce\x6d\xa5\x11\x75\xd0\x17\xd2\x11\xfa\x70\x37\x94\x41\x6f\x46\x9b\x11\x29\xd7\x31\xab\xde\x74\x4d\xa5\xb2\xfa\xcd\x7a\x9b\x09\x3d\x6c\x97\x43\x50\x9b\x01\x03\xba\xb9\xc8\x1c\x6e\x5f\x38\xbc\x97\x18\xe3\xe4\xfa\xa8\x64\x75\xd1\x37\x25\xa8\x29\xac\x61\xdf\x8d\x15\xf0\xb2\x7c\xb4\x0d\x0e\xba\x0b\x24\x6b\x9c\x36\x0b\x56\x9b\x81\xb3\xab\xf3\x80\xee\xc2\x74\x92\x31\x6b\xc2\x92\xe5\x15\x0e\xe0\x60\x72\x19\xa2\xbd\x80\xba\x98\x4c\x7e\x3f\x19\x89\xbc\x51\xe4\xc5\xda\x3a\xe5\x07\x06\x76\xe0\xc1\x50\xd0\x37\xa8\x6a\x0f\x91\xbf\xc0\x7c\xde\x64\xc1\x9f\x9c\x7a\x7a\xf4\x4d\x69\x29\x97\x00\x41\x44\x8d\x3b\x17\xc2\x49\xd5\xe0\xb5\x86\x2e\x9a\x25\x20\x9e\x8f\x97\xd7\xa0\xf0\x30\x18\x15\x04\xfe\xad\x22\x66\xc8\x73\xfd\x23\x59\x83\xdf\x3d\x06\x57\xb9\x20\x96\xe2\xb4\x90\xdf\x33\xca\x11\x57\x33",
        );
        test::<Sha256>(
            b"\xf9\x96\xf3\xad\xc2\xab\xa5\x05\xad\x4a\xe5\x2b\xc5\xa4\x33\x71\xa3\x3d\x0f\x28\xe1\x95\x0b\x66\xd2\x08\x24\x06\x70\xf3\x52\xef\x96\x18\x5e\x9a\x70\x44\xf4\xce\x2f\x2f\xf9\xae\x01\xa3\x1e\xf6\x40\xe0\xb6\x82\xe9\x40\xc5\x10\x51\x17\x59\x46\x13\xdd\x1d\xf7\x4d\x8f\x2b\xa2\x0c\x52\x22\x3b\x04\x5a\x78\x2e\x85\x0a\x12\xa2\xaa\x5c\x12\xfa\xd4\x84\xf1\xa2\x56\xd0\xcd\x08\x72\xd3\x04\xe8\x85\xc2\x01\xcd\x7e\x1e\x56\xd5\x94\x93\x0b\xb4\x39\x21\x36\xfb\x49\x79\xcc\x9b\x88\xaa\xb7\xa4\x4b\xfc\x29\x53\x75\x1c\x2f\x4c",
            n,
            e,
            b"\x30\xb3\x48\x62\x4f\xaa\x99\x85\xfc\xd9\x5f\x9c\x7e\xad\x3a\xfe\x64\x56\xba\xdf\x8c\x0f\xed\xbd\xad\xb3\xa9\x00\x3a\x67\x02\x97\x3a\xcd\xb4\xe8\x66\x52\x36\x7d\xb2\x3e\x0a\x81\x41\x88\x0d\x66\x31\x83\x4f\x9f\x17\x1c\x94\xa8\xfe\x9c\x31\x5b\xcb\x86\x80\xec\xfb\x5a\x4f\x59\xb4\x5d\x4e\x4c\x3c\x05\x82\x8b\x7f\xaa\xa8\xe4\x23\x4a\xad\xa4\xe7\x66\x64\x6c\xc5\x10\xd0\x7b\x42\xbd\x38\x83\xa8\x3b\x5b\xcb\x92\xd9\xe7\xcc\x1d\xdf\x59\x0a\x69\x01\x11\xbf\xc6\x2a\x51\xaf\x7e\x55\x54\x3e\xa5\x18\x8c\x92\x45\x3d\x41\xd3\xe8\xfd\xab\xee\x3e\x1d\xef\xa9\xd0\xaf\xdb\x85\xc8\x15\x3a\x50\x19\xae\x45\x56\x3e\xa3\x08\x0a\x30\x22\x66\x81\x68\xf0\xc2\x73\xa6\xdb\x1a\xfa\xdc\xd5\xed\xbc\xa5\x02\x1c\x2e\x53\xf4\xd9\x51\xc6\x04\x20\x6a\xe1\x0f\x28\x7f\x45\x18\x67\x27\x1d\x37\x04\x82\x79\x1c\xdf\xdc\xb6\xa4\x01\x0f\x6b\x3d\x9b\x92\x85\x63\xd1\x68\xda\x19\xf1\xc1\xe5\x70\xf8\xc1\x58\xf3\xd4\x90\xb2\x9a\xa2\x3a\xbd\x1f\xfd\xf2\x08\x66\xc3\x4c\x6e\x63\xb9\xe8\xa9\xa0\x2d\x7a\x1b\x19\x6d\x05\x5f\x4c\x53\xce\x82\xb4\x00\xe4\xab\x9e\x1b\x9d\x70\xd0\x04\x9d\x6d\x57\xcf\x0a\x49\x49\xcf\xc6\x8d\x63\x38\x82\x88\x2d\xcf\xdf\xc5\x0c\xf4\x49\xdf\x10\xac\xf2\x03\x05\xc2\xaa\x43\xbd\xa1\x0f\xd8\xa1\x0b\x4e\xca\xa2\x31\x00\xaa\x47\xe9\x29\x36\xdc\xe1\xbf\xb8\xd6\x59\x52\x35\xbb\xfe\x2c\x85\x85\xcb\x16\x47\xb2\xbe\xac\xb1\xe1\xd4\xb6\xce\xf7\x58\x81\x1a\x68\x33\x0f\xa9\xc3\xa8\x25\x73\xc0\x8f\xa2\xcd\xa5\xa0\x3f\x34\x25\x55\x4e\x45\xd9\x8c\x16\x45\xc5\xbd\x27\xd1\x2e\x6c\x20\xb2\xc4\x62\xa7\x46\xe8\x82\xa3\x42\x1a\x7b\x1b\x1e\x25\xb4\xc3\x6c\x8b\x16\xa1",
        );
        test::<Sha256>(
            b"\x6a\xce\x05\x2d\x7e\x99\xcd\x97\x3b\xb5\xc9\xf6\x67\x9b\x1c\x30\x5e\x07\x20\x89\x65\xfe\x58\xc6\x3b\x10\xa6\x92\xf1\xdb\xbe\x22\xfc\xd0\xdb\x15\x89\x3a\xb1\x9e\x10\x7b\xa2\xe4\x2c\x99\x34\xa9\xaa\xfa\xc3\x2a\xdf\x6c\x73\x47\x3f\x69\x69\xe4\x2c\x98\x3b\x8f\x0c\x96\xa4\x63\x9e\xf7\x7d\x2c\x8e\x88\xe8\xcc\x47\xd7\xcf\xdd\x08\xf6\x8d\x97\x3a\x7b\xea\xf4\x01\xcb\x4d\x13\x11\x99\x2d\xda\xc3\xa9\xc9\xe0\x67\xda\x19\x8a\xdc\x63\x04\x74\x5f\x5d\xd3\x12\xa1\x82\xe6\x97\x1c\x34\xa5\x15\xa6\xc1\xba\xe6\x47\xe5\x7e\x4c",
            n,
            e,
            b"\x5f\x0e\x74\xf4\x54\x75\x4a\x30\x74\xfa\xaf\xc6\x05\xf3\xc9\xaf\x47\x60\x4a\x89\x83\x65\x0a\x9b\x62\x11\xfb\x19\x1d\x9a\xfa\x53\x15\xdf\x4d\xb4\x50\x1f\xd4\xf0\x4c\x74\x1d\x76\x46\x56\xd4\xa5\xd0\x06\x38\x8a\xd8\xfd\xb2\x19\xec\x6b\x75\x69\x08\xe2\x3b\x30\xcb\x63\x9f\xfa\x7b\xbf\x28\x74\x71\x3b\xfd\x5a\x10\x62\xc1\x9d\x04\xe0\xe4\xa7\x4b\x14\x44\x6a\x7f\xdf\x5c\xb8\x12\xe9\xac\x7b\x60\x12\xd9\xae\x99\x1c\x47\x65\x6d\x2a\xde\xd2\x40\x74\xbb\x8a\x38\xb1\xa8\x8b\x1c\x2b\x13\x1e\x5b\x09\xc9\x37\x57\xfd\xb2\xd6\xb6\x9a\xa8\x26\x5a\x43\x5f\xba\x00\xae\xb3\x6a\x1f\x62\x9b\xc3\x4b\x87\x60\x89\xd2\x8a\x94\x8d\xd6\xab\x4c\x89\x94\x30\xda\x60\xa2\x6f\x6c\x13\x60\x3f\xc8\x89\xc7\xb2\x93\x6c\xa3\xc5\x15\x6b\xd7\xfa\x6e\x34\xea\xc9\xe0\x48\x00\x83\x3e\xf0\xcb\x9b\x6e\xef\x78\x8c\x0e\xf0\x02\x1a\x45\x36\xfb\x83\x71\xfa\x3e\x2c\x8b\xb8\xbe\xfa\xc1\x6e\x80\x92\xd6\x9c\x57\x1c\x1e\x15\xfd\x25\x5e\xc0\xa0\x7a\xcf\x9a\xe9\x95\x38\x31\xef\xd3\xdc\xbe\xf4\x4e\x0f\xcc\xeb\xb1\xaf\x95\x9d\x71\xf5\x01\x30\xe8\xac\xb4\xfa\x23\x19\x26\x1f\xba\x12\xf2\x71\x5d\xef\x82\xbf\xaf\xbf\x40\xe3\x45\xec\x5d\xcd\xab\x5c\x1b\xf5\xf6\x6b\x1d\x0e\x9f\x7a\x9c\x62\xc9\x37\x57\x46\xe1\xae\x0c\x8f\x14\xa4\x89\x18\x43\x83\xe8\x1d\xce\x20\x70\xad\x4b\x52\x5d\xf7\x6b\x44\x6b\x1f\x22\x92\x1d\x42\x4d\x9b\xa3\xce\x21\x57\x75\x01\xdf\x62\x80\xfd\xc6\x9f\x02\x39\xae\x11\x27\xb6\x99\x50\x75\x9d\x5f\x0b\x69\x3f\x54\xe8\x7e\x07\x63\x62\x3b\xf5\xd3\xff\x69\x43\x00\x81\xb9\xc9\xe2\x44\x5a\x05\xe1\x15\x67\x5e\x09\x0b\xca\xb2\xaa\x1d\x75\xce\xee\x2a\xd6\x19\xec\x8b\x80",
        );
        test::<Sha256>(
            b"\x0e\x49\x74\x0f\xdc\xca\x6b\xfc\xe2\x94\xc1\x1f\x45\x40\x78\x05\xb3\xda\x41\x2b\x01\xef\x3f\xb5\x13\xe7\x0e\x62\xfd\x95\x04\xc0\x67\x0d\xb6\x9c\x36\xb6\xbe\xbd\x69\xa0\xbc\xd2\x40\x17\x9b\xa8\xa4\x78\x16\xa0\xc3\x43\x7a\x61\xfb\x72\xad\xca\xf9\x09\x6f\x2a\x22\xef\xe0\xb4\x31\xfc\x42\x2d\x22\x53\x01\xe8\x50\xf2\xf0\xf4\xda\x87\xd6\x94\x4a\x85\x29\xef\x79\x78\x19\x09\xad\x96\xd1\xf2\x05\x96\xf9\x3e\x17\xc5\x7f\xb4\xd7\x56\x97\x4b\xbb\xf9\x00\x52\x1c\xb0\x89\xee\xe0\xde\xd5\xc9\x56\xa1\x5b\x09\x61\x62\xb0\x7f",
            n,
            e,
            b"\x7b\xbb\x3d\xdd\x17\xa4\x2b\xe7\xcc\x4e\x7e\xaf\x45\x65\x09\xa4\xba\x58\xd4\x0c\x49\xa3\xd9\x95\x73\xb7\x33\xe1\x94\x2f\x9f\xca\x20\xba\x8b\x91\x07\x08\xd6\xe7\x50\x36\x7e\x84\x73\x02\xfc\x60\x3b\x80\x63\xc1\x9a\xf8\x83\xe7\x50\x7f\xb0\xd9\xcc\x2b\xe3\x74\x79\xa3\x7c\xca\x25\xb8\xc7\xc4\x6f\x6b\xf6\x61\xdc\x6a\x32\x32\xf8\x8b\x48\x3f\x1b\x8f\x41\xb4\x6d\x49\xba\x3f\x17\x95\xd6\x8e\xaa\xd4\xa2\x55\x6f\xb5\xd7\x87\x3b\xbb\x65\x01\xec\xf0\x6a\xc5\x58\x23\x5e\xd1\x39\x90\xb0\xe1\x6f\x67\x96\x5b\x09\x36\x6b\xcb\x36\x2c\xfc\x6f\xb9\x78\xf4\xf6\x8d\x81\x46\xdc\x8b\x81\x98\x04\xdf\x42\x4e\x8c\xa5\xb6\x3c\xf1\xfc\xf9\x7b\xbf\x30\x0d\x0b\x99\x88\x60\x79\x8a\x63\x42\x43\x83\xfc\xd8\x1d\x37\x77\x3d\x59\xbb\x13\xb4\xfa\x5d\x46\x8c\xd1\x28\xbb\xab\x18\xa8\xce\x51\x73\xbe\x5d\x9d\x54\xd3\x17\x7f\x02\x45\x78\x84\x09\x97\x3d\xf4\xa9\x01\x6b\x94\x4b\xae\xfb\xf3\xbf\x11\x46\xa9\x39\x3d\x22\xe3\x5e\xc2\xbe\x0a\xe6\xf4\xc3\x1d\xc4\x98\x1f\x40\xfc\x1b\xaf\x38\x26\x00\x69\x9e\xaf\xce\xa9\x2c\xbe\x24\xe2\x6e\xe8\x46\xfa\x23\xbc\x19\x3b\x6e\x72\x14\x01\xb7\xac\x3f\x5f\x4e\xbe\xb6\x33\x97\x9f\x8e\xf3\x5f\x4a\xb1\x11\x7a\x86\x9d\x5b\x9d\xbb\x74\x82\xf0\xd5\xa5\x9e\x41\x63\x54\x8d\x25\x12\xae\x06\x72\x05\xb5\x7d\x03\x0c\x48\x3f\x72\x0d\x2c\x44\x35\x04\x28\xf5\x26\x89\x43\xfc\x5f\x6e\xa1\xc8\x8e\x2e\xc1\x3a\xb3\xdc\x14\x56\xe9\x6a\x3b\x8e\x7c\x12\x1a\xf4\xd6\xa5\xfe\x4e\xe5\x5e\x99\xfb\xc3\x59\x2a\x48\x7c\x19\x4b\xc2\xf2\xbf\x6e\x79\xfb\x79\xc2\x87\x6c\xf3\x36\x5e\x07\x5b\xee\xac\xc7\xdb\x4d\xb7\xee\x69\xe7\xf1\xfe\x12\xa3\x27\xe6\xcb\x0f",
        );
        test::<Sha256>(
            b"\x0e\x67\x5d\xac\x9a\xec\x91\x01\x06\xa6\xab\x21\x9b\x4c\xce\xb5\x2d\xed\x25\x49\xe8\x99\xc9\xa2\x4d\x5e\xe5\x51\x77\x76\x18\x88\xa3\xbe\x1a\x2d\xef\x6a\xa3\x2d\x62\xf7\x88\x13\x2d\x62\x27\xd9\x30\x98\x06\xfd\xc0\x2d\xb7\xd8\xa8\x50\xff\x2c\x6d\xff\x37\xfc\xd7\x77\xf1\xa0\xac\xef\xdf\x18\xbf\x85\xf1\xa1\x29\x79\xbe\x86\xd7\x99\x25\x39\x45\xfc\x34\xa2\x88\xf3\x48\xb7\x92\x3d\x76\x4d\xb2\x7a\x2a\x2d\x5a\xe2\x0e\x6b\x25\x37\x2e\xf3\x18\xf8\x59\x65\x29\xd8\xca\x23\xfd\x6f\x08\xa8\xf6\x2e\x0a\x1b\x6d\x98\x9f\x23",
            n,
            e,
            b"\x80\x52\xd9\x5f\x12\xce\x0e\x6e\x53\xa5\xa3\x56\xa0\xeb\x35\x3b\xdc\xc1\xa6\x65\x14\xd6\xcf\xb3\xa3\xd9\x61\x55\x31\x0b\xdd\xa0\xa0\xd1\x79\x5f\x97\x64\x3f\x3a\x44\x96\x63\x4f\x2d\xd9\xb9\x5a\x21\x38\xee\x39\x0e\x1e\x74\xbe\x31\x34\xf3\xf4\x7a\x91\x9e\xe7\xb5\x9f\x8e\xcd\x27\x2a\xb8\x8c\x82\xcb\xce\x7c\x21\x7e\x5f\x92\xd0\x57\xa5\xb0\x0f\xbf\x05\x75\xcd\xae\xcd\x7d\xc2\x85\xa4\x21\x8c\x8a\x95\x52\x16\x59\x8f\x07\x42\x67\x1e\x01\x8e\x8e\x4e\x76\x83\x9a\x57\x5f\x50\xb2\x10\x2a\x8b\x77\xd1\xb8\x4f\x6d\xce\x98\xd7\x8e\x57\x58\xe0\xa6\xf9\x2b\xf3\x5d\x6a\x2f\x18\xad\x40\x09\x25\xd7\x88\x0f\x9e\xfc\x77\x4a\x8c\x7e\xbf\x64\x88\x5c\xd2\xf6\xf6\x29\xb5\x4a\x7c\x12\xec\x91\xd3\x9b\x3c\x25\x18\x24\x1f\xdc\x32\x2d\x9b\x23\x5a\x8e\xa4\x4f\x77\xe8\x2f\x3d\xc4\xf7\x28\xf6\x20\xc0\x7d\x1e\x7f\xf4\x09\x4f\x29\xc6\x74\xab\x0f\x08\x02\xef\xa1\xc9\xe6\x48\x1e\xbb\x84\xe0\xbf\x13\xef\x46\x8d\x8c\xca\x11\x45\x70\xb9\xed\xcd\xdf\x98\xac\x4a\x83\x4f\xe7\xa0\xd5\xc6\xfa\xe8\xa6\x0a\x48\x39\x9f\x3c\x8a\xf4\x2f\xf4\x02\x6e\x42\xa8\x1a\xac\x36\x11\x4f\xfc\x05\x3f\x3f\x72\x9b\x7c\xf9\xa9\x7a\x56\x84\x8e\xbe\xa0\x11\x5a\xa8\x29\x83\x41\xaa\x22\x69\x63\xeb\xdf\x57\xab\x2d\x8e\x4b\x90\x00\xdd\x05\x1a\x6c\x5d\x69\xf6\x0e\x1d\xc1\xb3\x3f\x20\x94\xfd\xbf\x8e\x5b\x62\x7b\xc0\x76\x4d\xb9\x52\x2c\xbb\xc0\x81\xdb\xf3\x8c\x21\xb1\x3f\x98\x08\x13\xbd\x2b\x00\xc7\x57\xeb\xb8\xc0\xb2\x12\x13\x15\x2e\x69\x40\x39\xf3\x06\xf7\x34\x28\x57\x65\x1f\x72\x2b\xdd\xa0\x12\x12\xa8\x55\x27\x99\xbd\xa6\xef\x07\xc5\x20\x7d\xc7\x44\xef\x79\x69\xaf\xd5\xaf\x2e\x6f\x12",
        );
        test::<Sha256>(
            b"\xf6\xa7\xa6\xe5\x26\x59\x12\x5f\xbb\xc8\x72\x74\x17\x28\x3b\x9a\x64\x44\x1f\x87\x12\x1e\x27\xf3\x86\xd5\x01\x9f\x10\xcc\x9b\x96\x1e\x09\xf1\xb3\xb0\xdb\x23\x63\x0c\xc0\xca\xac\xb3\x85\x8c\x6f\x93\xaf\xee\xea\x7e\x1a\x6a\x80\xdb\xe0\xc2\xbd\x9c\x7c\x93\x95\x70\x30\x2d\xec\x39\xa4\xa2\x5c\xc0\xcf\x1d\x32\xa7\x1a\x75\xb9\xa0\xc3\x02\xbc\xdd\x80\xb0\x46\xc8\x66\x51\xac\xf3\x08\x38\xcd\x52\xe3\x03\x99\xa8\xfa\xb8\xd0\x3f\xbd\x14\x0c\xdc\x2f\x1f\x02\xf2\x48\x04\x05\x16\x98\x20\xcc\xb3\x2e\x59\x74\xff\xb8\xb1\xc8",
            n,
            e,
            b"\x84\x60\x3a\xcb\xfe\x1f\x2f\x76\x9f\x1a\x62\xb0\xf2\x87\xf3\x06\x94\x0b\x22\x54\x76\x71\x4a\x4b\x68\x27\xc0\x2d\x7b\xd0\x52\xf3\x03\xf3\x0a\x5f\xa6\xda\x83\xe6\x06\x15\x30\x56\x69\xca\x9e\xc1\x77\xc5\xb3\x2b\x14\x15\xee\xbe\xf7\x86\x20\x29\x6e\xba\xd6\xdb\xbd\x52\x08\x39\xd3\xaa\xcc\x97\x81\xac\x86\x02\xdd\xce\x07\x36\xdc\xfa\x72\x90\xb4\x5f\x15\x5b\x8e\x92\x4d\x0a\xfd\xf7\xdf\xc8\xd1\x99\xbf\x09\x50\x9d\x01\x76\xa6\x8b\x14\x57\x56\xee\xf5\x3d\xe4\x56\xe1\x70\x78\x85\x98\x49\xa3\x52\xa5\xbb\x65\x42\x39\xd8\xeb\xaf\x88\x00\xca\x82\x63\xd3\x4a\x86\x8d\x52\xbf\x8f\x22\x64\x4d\xd9\xf3\xc0\x5b\xd8\x91\xcd\x92\xf2\x63\x53\x0c\x58\x96\x02\x3c\x6b\x21\x3d\xdb\x64\xed\xe1\x77\x0f\xf1\x68\x6c\x34\x03\x6e\x28\x1e\x91\x1d\x9d\xc9\x60\x35\x4f\xd8\x44\xcb\x7b\x22\xdc\x0c\xd8\x1a\x96\x20\x3b\xa8\x18\x40\x1c\xcc\x22\x5f\x85\x7e\x59\xa5\xcb\x7b\xa6\xdf\xc7\xf5\x13\x5e\xa3\x27\x81\xe6\x3d\xaa\x14\xfb\xda\x1b\xac\xc1\x8e\xbc\x50\x82\x4d\x40\x28\xb8\xfd\xec\xda\x49\xe8\x10\xba\xe5\xac\xc8\xad\xc0\xdc\xa2\xe2\x36\xfc\x83\x2a\x97\x33\x0a\x12\x14\xfa\x0a\xed\x15\xcd\x10\xc0\x49\xef\xb6\x5c\xe8\x55\xc0\x60\xf0\x5b\xef\xb3\x17\xb8\x06\x58\x43\xc4\xeb\x5a\x03\x71\xfc\x6f\x20\x9f\x6f\xfb\x94\x8c\x88\x1f\x2f\x20\x91\xca\xf0\xf5\x9f\x60\xb7\x2c\x5f\x67\x27\x1b\xae\x96\xb9\x13\xfd\x21\xfa\x1d\xfa\x97\x5d\x5e\xcd\x62\xb0\xd5\x08\x73\xb6\x86\xd2\x9c\x88\x0d\x36\xed\xca\xd3\x3e\xc3\xe2\x21\x6c\x9c\xfc\xfb\x4f\x98\x4c\x23\xfd\xe8\x15\xe2\x80\xa8\x02\x42\x86\x08\xbe\xd3\x73\x9a\xf9\x20\x0d\xe1\xf8\x5e\xde\xe2\x83\x4c\x04\x94\x2c\x06\x8a\xac\xd2",
        );
        test::<Sha384>(
            b"\xbb\x29\x4b\x95\xd9\x13\x00\x5b\x11\x09\x87\xcd\xe4\x58\x87\x48\x4a\xe6\xdf\x79\x48\x73\xdf\xc5\xc4\x1f\xb7\xe8\x99\x2c\x2f\xdc\xe7\x06\x99\xfc\xac\x80\x04\x69\x99\x61\xb3\xad\x1e\x1f\xce\x9e\xc8\xea\x56\x85\xcc\xec\x5e\x80\xe4\xd0\x79\x25\x59\x81\x6f\x68\x61\x34\x34\xbf\xac\xa8\x1a\x84\x3a\xac\x45\x9a\x6f\xe3\x5f\x53\x69\xc4\x8e\x91\x91\xe4\xa3\x2c\x70\x78\x95\x94\xc5\x15\x2d\xb8\xd4\xbb\x02\x26\x00\x12\xa8\x73\x9c\xf3\x25\xdd\xff\x2a\xa4\x2f\xd6\x7b\x6e\xe5\xbf\xe3\x15\x91\x13\x1f\xf2\x7d\x02\x73\xd2\x92",
            n,
            e,
            b"\x32\x63\x7c\x60\x79\x8b\x45\x0b\xff\x10\x0b\xff\x12\x83\x83\x57\xde\xff\x28\x1d\x5b\x31\xe4\xf4\xc2\xcf\xc9\x6e\xb7\x79\xce\x6d\x31\xb1\xce\x8b\xd7\xaa\x7f\xa8\x8d\xdc\x42\x79\xc8\xc3\x28\x06\x04\xb0\x18\xcc\xf4\x52\x00\x4a\x14\x88\xed\x47\x50\x18\x1c\x50\x25\x63\x65\x11\xac\x67\x24\xfe\x51\x76\x1c\x27\xd7\xcf\x9a\x0c\x87\x82\xea\x22\x31\x26\x88\x53\xc4\xb1\xf7\xac\xb0\x00\x5e\x56\x87\xc8\xf3\xdf\x16\xc9\x62\xf0\x2c\xe5\x6b\x23\xd3\x87\xa2\xba\xad\xc8\xbe\xc9\x42\x29\xc3\x55\x75\x26\xe6\x17\x07\xa8\xb5\x92\x93\xa9\x76\xe3\x2c\x7f\xa1\x33\x28\x50\x88\xf3\xce\x3e\x67\x77\x88\xaa\xa9\x47\xe7\x62\x2c\x75\x7e\x84\x4b\x11\x75\x92\xbe\x99\xfe\x45\x37\x6f\x8b\x30\x13\xe8\x77\x2e\xc9\x2c\x5b\xb0\xb9\xfa\x30\x1b\x95\x54\x45\x99\x69\x0a\xd9\x36\x68\xd8\x3b\x2d\xaa\x7d\xf0\x5c\x66\x21\x4e\x27\x50\x14\x78\x0a\x91\x2d\x8b\x19\x32\xd7\xa6\x55\x05\x8e\x74\x3f\x50\xb0\x74\xb1\xd9\x69\x1c\xa2\x3a\x2f\x95\xf6\xaf\xfb\xd5\x16\xd6\x4c\xcb\x2a\xa4\x3c\x23\x6e\xb9\x5d\x36\xd2\x72\x54\x5e\x3b\xeb\x8f\xf5\xaa\xcd\x95\xb3\x0f\x7f\x1d\x64\x18\xaf\x04\x2c\xd9\xa0\xcf\x01\x89\x84\x62\x62\x32\x2a\x18\x87\x5a\xe4\xc3\xe6\x8e\x4e\x8f\xfa\xa0\x27\x6c\xdd\x99\xa0\x04\x7c\x86\xc0\xf7\x1d\x2d\xee\xfd\x50\x64\x2d\x29\xc1\x95\xe6\xd1\x4f\xb4\x6f\xba\xc3\x3a\x50\x8c\x1f\x03\xa2\x32\xde\x08\xaa\xe0\x9f\xaf\x1d\xa8\xed\x2b\xa2\xae\x84\xbc\xca\x88\xb7\x8d\xcc\xbd\xe9\xaf\xde\x08\xa3\xbe\xb3\x22\xdc\x79\x35\x6b\x29\xc8\x48\x41\x69\x89\x14\xb0\x50\xbe\xb7\x5a\x7b\x2f\x67\x01\xaa\x81\x01\xa5\xa4\x95\x5e\xe2\x7b\xaf\xe8\x1b\x21\xd0\x3b\x43\xe3\xc7\x73\x98",
        );
        test::<Sha384>(
            b"\xf9\x46\xc6\xbd\x5e\x1d\x6b\x89\x09\x2f\x3c\x48\x7c\x05\x68\xfa\x07\xc3\x56\xfa\xe9\xb8\xe8\x31\xb8\x32\x02\x89\x03\x97\x46\xa4\x35\xb1\x22\xcf\xbc\x4a\x0d\x31\x6b\xf9\x0d\x48\x1d\x3b\x7d\x97\x9c\xc5\x0d\x98\xc1\x19\x0a\xf8\xdc\x58\xe0\x03\x55\x57\xdd\x5e\x94\xf4\x37\xf4\x1f\xab\x51\x32\x02\x64\x3a\x77\x74\x8f\x76\xc6\xb7\x73\x02\xbf\x40\xc3\x92\xcd\x18\x73\x1d\xa0\x82\xc9\x9b\xde\xde\xb7\x0e\x15\xcd\x68\xbf\xf5\x96\x19\xca\xbc\xc9\x2a\xdc\xf1\x22\x75\x3c\x55\xaf\xde\x08\x17\x35\x2b\xc2\x47\xd1\x17\x0b\x8d",
            n,
            e,
            b"\x50\x70\x6b\xa4\x9d\x9a\x31\x66\x88\xa3\xee\x80\xa0\xbd\x98\x67\x57\xd4\x3e\xc8\x32\x85\xaf\x9e\x78\x19\x6b\xd5\x2c\x90\x0d\x40\xb2\x80\xfa\x0d\xe5\x4e\x35\xac\xe7\xd6\x66\x00\x12\xf1\xa6\x62\x04\x09\x2f\x0e\x63\x4b\x97\xe0\xe5\x16\x65\xb4\x07\x5e\x36\xf1\x42\x22\x66\xc7\xca\xd7\xb2\xd9\x98\x1b\x91\x3d\xf3\xfa\x3e\x6a\x5a\x1c\xad\xfc\x63\x78\xa8\x54\x0e\x0f\xaa\x26\xf1\xcc\x6f\xb2\xfb\x49\x2a\x80\xd0\xa6\x94\x5b\xce\x5b\xbc\x23\xdd\xb3\xb1\x07\x01\xf0\x24\x9b\x27\x40\x7a\x67\x00\x80\x2e\x88\x42\xef\x3c\xc7\x61\xc4\x82\x3a\xcb\x5d\x14\x53\x50\x8d\xcd\xbb\x97\x9e\x7b\xd8\xd0\x01\x28\xe6\x0a\x9b\x37\x89\x16\x7c\x91\x41\x7d\x93\xf0\xe9\xfb\xb0\x0c\x9a\xf1\x49\x8e\x09\xeb\x64\x85\xeb\x94\xce\xa4\x88\x3f\x6a\x25\x6e\xab\x2c\xaa\x82\x6d\xe4\xfd\xac\x01\xba\xca\x3a\x21\x6e\x3d\x20\x4a\x3d\x83\x7f\xfd\x4d\x0b\xe2\xb2\xce\xf7\x11\x90\x90\x54\xc4\xda\x1d\x5b\x93\xa8\xf9\x84\x51\xc7\x00\x2a\xe8\x4a\x5e\x70\x80\xd9\x86\x71\xc5\x0e\x3c\x91\xc4\x08\x7d\x04\x77\xb1\x04\xf9\x16\x01\x0e\x74\x2f\x2d\x20\x7f\xb4\x0d\x12\x2d\x8f\x21\x1a\xf6\xd7\xc5\xec\xa4\x95\x42\xd9\xac\xb0\xf1\x66\xe3\x6a\xbc\x37\x15\x50\x70\xc1\x2e\x9f\x28\xb9\x07\xd6\x7a\x2c\xa7\x0b\xfc\xe5\x54\xe1\xc4\x4c\x91\x52\x0e\x98\xfc\x9a\xd0\xc0\xee\x47\x7f\x75\x05\x16\x47\x6a\x94\x16\x80\x66\xce\x47\x00\x00\x30\xa9\x9c\x23\xe2\xc3\x87\x55\xde\x94\x6d\x5e\xdf\x0d\x6a\xa9\x42\x12\xf9\x92\x31\x5b\x24\x8c\x1f\x82\x72\x3b\x29\xc4\x22\x16\xc7\x8c\xdc\xb6\x68\xf1\x12\x78\x26\x1c\xee\x92\x52\xc8\xfd\x0e\xd3\x7d\x0a\x85\x80\xca\x9b\x9f\xde\x75\x05\x61\x59\x43\x71\x2d\xa1\x9a",
        );
        test::<Sha384>(
            b"\x9a\x33\x7d\x4c\x0b\xb9\xa0\x05\xb4\x7f\x47\x65\xd6\x96\xd1\x9d\xec\x58\xbc\x84\x82\xf2\x17\x3a\x4a\x20\x3a\x0b\x6d\x38\xb4\x96\x1f\x6a\x85\x2e\x76\x46\x8e\x80\x7c\x7e\x45\x76\x83\xee\xad\x5c\xb8\xd9\x86\x42\xfb\x76\xc0\xa1\xee\xab\x36\x41\x4c\x18\x99\x59\x7d\x57\xaa\xf9\x67\x82\xad\xa5\x86\xf6\x1a\x42\x3f\x57\x95\x37\x71\xd5\x20\xcc\x4e\xad\x90\xd5\x69\xf2\x3d\x95\x0f\x8d\xfe\xdd\xdb\x83\x55\x74\x85\x76\xe6\xbb\xfb\x6f\x2e\x91\xb3\xda\x71\x75\x3f\xd2\xf4\xea\x22\x9f\x6d\x20\xe2\x7d\xb8\xd0\x5e\x9f\xcb\x68",
            n,
            e,
            b"\xcf\xf7\xaa\x7f\x87\x56\x42\xfb\x93\x43\xe0\x7e\xf5\xe7\x30\x3b\xbf\x5f\x06\x9b\x44\xc1\x9f\xbf\x83\xe5\x9d\x42\x2e\x25\x26\x7e\xf9\x30\x74\x14\xb6\xb1\xef\x61\x71\x1e\xd0\x01\x32\x76\xd1\xa2\xad\x98\x39\x04\x74\x02\x7a\x0a\x70\x3b\xfe\x8a\x6e\x87\x70\x60\x59\xd8\x9c\x06\x09\x80\xc9\xc9\xe6\x0d\xc7\xe1\xfb\x9f\x77\x7a\x41\x78\x5a\xb4\xd2\xb6\x63\xba\x0e\x3c\x19\x21\x54\x5c\x47\x9c\x2a\x38\x3a\x50\xda\x8e\x48\x9c\xb2\x2b\x71\x10\x1d\x0e\xc1\x48\xac\x70\x92\x87\x32\xa7\x72\x19\x5a\x14\x0d\x08\x01\x52\x76\x2a\x9c\x40\x80\x3a\x39\xfa\x2a\x69\x78\xc2\xa7\x5a\xc4\xd8\xbd\x1b\xcc\xaa\x1f\x42\x04\xba\x65\xed\xdd\xf3\x2f\xed\xf2\xd9\xd0\xa3\xae\xd9\xb0\x6c\x47\xe7\x17\x73\x3c\x57\x78\x12\xd7\x23\xdb\xa7\x4a\x85\x2b\x29\x05\x23\x5c\x81\x2d\xc5\xf1\xd0\xdf\x0f\x0d\xe7\x3d\xfb\x86\x22\x1c\x6f\xfd\xd1\xed\xa1\x19\xbb\xe9\x8d\x14\x8a\xdd\x36\xa4\xfe\x50\x48\x9b\x06\xaa\xee\xfc\xb5\xc2\x06\x6d\x90\xfa\x79\x73\x87\x06\xcd\x18\xe4\x74\xd6\x96\x09\xff\x12\x10\xc7\x7d\xe7\xcd\x23\xba\x2a\x77\x5a\x43\x29\xcb\x27\x1a\x82\x6d\x60\x2c\x40\x1a\x71\x43\x90\x19\xce\xc1\x0c\xd9\xf1\x84\xc4\xd0\x45\x84\x21\x18\x27\xb1\x9e\xad\xac\x32\x58\xd8\xa0\xf2\x63\x16\x13\xf0\x51\xaa\xe0\xc6\x13\x05\x0c\xb2\x44\x42\xf1\x5e\xd4\xfe\x0d\xbd\x29\x0e\x42\x62\x91\x41\xbd\x2c\xd5\x6d\x20\x58\x4a\x1d\x10\xe1\xf2\xc2\xa9\xec\x73\x14\x33\xd5\xbc\xd1\xd3\x18\xbe\xd5\x24\x3b\x4b\x7d\x0f\x9a\x79\x82\x06\x1c\x55\xdf\xaa\x86\xb2\xc0\x18\x45\xc0\x21\xfd\xd2\xa9\x78\xd4\x20\x34\x21\x2f\x43\xb3\x35\x1b\x6a\xde\xb0\x3b\xdd\x6c\xaf\x7d\xe0\x59\x50\x2f\x16\xd7\x73\x48",
        );
        test::<Sha384>(
            b"\x32\xfd\x45\xe7\x3f\x6f\x69\x49\xf2\x0c\xab\x78\xc0\xcc\x31\xd8\x14\xba\xea\x63\x89\x54\x6a\x36\x5d\x35\xf5\x4f\x23\xf1\xd9\x95\xb7\x41\x01\x18\x77\x60\xc8\x9b\xb0\xb4\x0b\x50\x57\xb1\x82\xe2\xfa\xfb\x50\xb8\xf5\xca\xd8\x79\xe9\x93\xd3\xcb\x6a\xe5\x9f\x61\xf8\x91\xda\x34\x31\x0d\x30\x10\x44\x1a\x71\x53\xa9\xa5\xe7\xf2\x10\xeb\xe6\xbc\x97\xe1\xa4\xe3\x3f\xd3\x4b\xb8\xa1\x4b\x4d\xb6\xdd\x34\xf8\xc2\xd4\x3f\x4a\xb1\x97\x86\x06\x0b\x1e\x70\x07\x0e\x3e\xd4\xd5\xf6\xd5\x61\x76\x7c\x48\x3d\x87\x9d\x2f\xec\x8b\x9c",
            n,
            e,
            b"\xc3\x89\x61\x37\x17\xec\x74\x76\xec\xda\x21\x44\xd0\xe8\xc8\xf9\xd6\x6f\xb4\x69\xc1\x67\xc4\x20\x9e\xc0\xbd\xee\xbf\xb4\x71\x66\x5d\x33\xda\xd4\x7b\x8f\x3c\x31\x9a\x76\xfe\x8a\x8a\x9f\x66\x2b\x6c\x69\x0b\x74\x90\x3d\x17\xf6\x1e\x23\x14\xe5\xea\x8d\x26\x67\x0e\xe4\xdb\x4d\xad\x29\x5b\x27\x7c\xa0\x8a\xde\x88\x0d\xe2\xe4\x2d\x12\xb9\x29\x52\x76\x4c\x1d\xc8\x08\xc2\x66\xdb\xbe\xdb\x67\x01\x58\xee\xf3\x6e\x89\x6f\x55\xa2\x03\xfb\x99\x55\x6d\xed\x05\x97\x41\x0b\xa3\x74\x86\xb1\xd8\x41\xf3\xd6\xd5\xc0\xb3\x9f\x2f\x49\xf0\xc5\x79\x48\x24\xfb\xa9\x4a\x8e\xc7\xc2\xb2\xc9\x1e\xad\xd5\xc8\xcb\xe4\x48\x95\xfe\x3b\xe3\xbc\x17\x27\xd6\xfc\x0e\x53\x64\xf5\x35\x78\x63\x9d\x3b\x3a\xf6\x96\xb7\x50\xa0\x78\x53\x69\x4f\xfe\x14\x5a\x28\xc0\x36\x20\xc7\x8d\xd7\x37\x7d\x09\x4d\x92\xc3\xe0\x95\x46\x88\x3d\x47\x03\xe6\x2a\x98\xdd\xf8\x1f\xd0\x1f\xcd\xf3\xc4\xb2\x15\x22\x4f\xe2\xb1\xb4\x99\x2a\xbf\x31\xf2\x0d\x12\xaf\xa8\x68\x20\x23\x90\xde\x33\x4a\x84\x6b\x2d\x58\xb2\x53\xea\x8a\xb3\xc5\x26\x5d\x84\x77\x3a\x65\x9e\x8b\xac\x7a\xf4\x41\x23\xd9\xea\x15\x06\x2e\x65\xd4\xd4\x19\xcf\x2d\x97\x07\x7d\x06\x24\xf8\xe5\xc3\x6f\x2c\x7b\x35\xcc\xf9\x54\x35\xd5\xc3\x68\x86\xff\x91\x05\xa6\xc1\xea\x22\x5e\x15\xea\x8c\xbc\x7b\x6b\xf6\x85\x61\x51\xcd\x76\xfb\xb7\x5b\x5b\x98\xf0\xe3\xdb\x51\x6a\x8e\x21\x81\x89\xfc\xb1\xcd\x5d\xe3\xca\xfe\xaa\x33\xef\x13\x5c\x5d\x8b\x8a\xa5\xf8\x81\xaf\xaa\xca\xf4\xc0\x8b\xd7\x28\x12\x55\xbc\x2a\x33\xb7\x6d\x4a\x36\xe0\xb1\x70\xc4\x55\x88\x23\x9e\x5b\x38\xc6\x79\xb0\x8c\xf8\x02\xaf\x73\xb6\xd7\x9b\x39\x35\x94\x94\x61\xe7",
        );
        test::<Sha384>(
            b"\xab\x66\xcc\x48\x7e\xc9\x51\xf2\x11\x9d\x6e\x0f\xa1\x7a\x6d\x8f\xeb\x7d\x07\x14\x9b\xec\x7d\xb2\x07\x18\xe4\xf3\x1d\x88\xc0\x1f\x9a\x53\xd5\xba\x7e\xce\x3a\x4d\xbc\x67\xaf\x6a\x35\xd1\x30\xea\xe7\x62\xcb\x79\x62\xb9\xae\x55\x7c\xa3\x84\x52\x46\x40\x02\x22\x3f\x61\xbc\xd3\xc7\x35\x3e\x99\xd6\x25\x58\xce\xed\xfc\xb9\x37\x4d\x4b\xbf\x89\x68\x0c\x8e\x2b\x95\x85\x60\x3e\x07\x6f\x1c\xdb\x00\x58\x29\x9b\x42\x46\x84\x5d\xc7\x9d\x10\x43\xb1\x42\x2e\xfe\x84\x01\x8e\x4c\x93\x2c\x45\xbe\xb8\x85\x1f\xbf\x48\x5e\x36\xd2",
            n,
            e,
            b"\xb5\x13\x31\x55\x2b\x08\xbe\x35\xa1\x69\x8a\xa6\x20\x3d\x84\xdb\xff\xf9\x00\x1e\xd5\xdd\x77\x6f\x2b\xe4\xdd\xfc\x07\xdd\x46\x20\xe9\x65\x4e\x82\xa3\x34\x65\xbd\x20\xf1\x18\x63\xc0\xed\x02\xa0\xae\xa2\x7a\x44\xd4\x14\xc3\x28\xa9\x38\xbf\x87\x7e\x15\x83\x8a\xb9\x9d\x67\x0d\x01\x41\x42\x62\xe8\x86\x5d\xc1\xd9\xfc\x30\xfd\x08\x12\x69\x9f\xa6\x90\xc3\x4f\x30\x2f\x63\x7e\xc8\x02\xcd\x40\xac\x85\x91\xe9\x76\xc0\xb8\xbc\xcb\x1b\x01\x37\xaf\x64\xa2\x87\x02\x10\xe8\xfa\x3d\xc4\x31\xfe\x09\x56\xb8\xad\xdf\xf1\xe4\xb1\x8c\xf0\x7e\x07\x8a\xa9\x3a\xf8\x1b\xb3\x02\x3c\x9e\x59\x4e\x66\x59\x5f\xd9\x2b\x10\x22\x6e\xa1\x26\x00\x5f\x47\x24\x42\x73\x52\xc3\x8e\x9e\x85\xfc\x2e\x07\x23\xf8\x0a\xf1\xf6\x15\x99\x55\x0b\x5e\xf5\x4c\x5b\x38\xca\x40\x57\x38\x01\x7b\x89\xcb\x94\x68\xd9\x74\x1c\xd6\xbd\xf7\x11\x21\x62\x25\x1b\xa1\xd0\x83\xcc\x37\x0a\x4a\x82\x61\xc3\x9b\x6b\x94\xbf\x21\xa5\x3b\x75\x64\x53\x1a\xe9\xeb\xc4\xcc\xea\x7e\xbb\x8b\xd3\x14\xb2\xe1\x3b\x58\xed\x10\x18\xae\x5b\x41\x5e\x0f\x9e\x3e\x19\xa5\xea\xd3\xa4\x46\x03\xf9\x06\x74\xa1\x90\xfe\xbd\xe2\x5f\x8a\xd8\x77\x8a\xee\xad\x4d\x0f\x64\xfb\xae\x37\x16\x6a\x54\xe3\xa7\x63\xe3\x55\x59\xbf\x8c\x3f\x17\x3f\x19\xff\x7b\xab\x98\xf3\xef\x80\x3d\xd5\x6c\x07\x62\x83\x99\xaf\xf8\x74\x85\xee\x73\xdb\xc3\xdb\x34\xec\xc7\xbf\xf3\xa5\x32\x26\xcf\x87\xbc\x81\xd2\x56\xe8\x0c\x09\x52\x0c\x8f\x38\xe9\xbc\xda\x09\x5e\x36\x35\x12\x8e\x1b\xed\xd9\x97\x06\x00\x54\x6a\x75\x1e\xb1\x1d\xab\x42\xe2\x89\xd6\xfd\xfe\xa0\x4b\xd5\x8d\x45\x71\xa7\x9d\x24\xbc\xe4\x50\x8c\x54\xe1\xec\x4c\xf7\x5b\x98\x5f\xd3",
        );
        test::<Sha384>(
            b"\xfe\xf7\xfe\x89\xb9\xa5\x99\x02\xa7\x0a\x1d\x9c\xaa\xd0\x9c\xed\x8b\xee\x41\x45\xed\xcb\xe3\xef\x7f\xa6\xda\xb3\x76\x35\x12\x9f\x3b\x8c\x5e\x08\x60\x41\x0e\xcb\xd9\xce\xc3\xd8\x69\x36\x82\xf2\x5a\xec\x08\xb0\x71\xf0\x5d\xc8\x21\x3b\xac\x8c\xff\x5d\x52\xb5\x76\x65\x35\x60\xbc\x01\x57\x56\x04\xe6\xab\x90\xf6\x72\x27\xfb\x5c\x90\x1a\x78\x1e\xdd\xc0\x27\x70\x09\x13\xe5\x4a\x7f\xe5\x13\x18\x48\x2c\x9a\xb4\x2c\x9d\x2b\x91\x1b\x7c\xcc\x39\xcc\xb2\x90\xf9\xa4\x20\xa5\xda\xd9\x33\x94\xd4\xd7\xb8\xc5\x3f\xe3\xf2\x42",
            n,
            e,
            b"\x45\x06\x8c\xa6\xd8\x2f\x2c\x12\x39\x25\xcd\xe1\x19\x71\x21\x5d\x8f\xa4\xa4\xdf\x68\x48\xbb\x76\x54\x86\x87\x00\x97\x87\x64\x85\x46\x38\x92\x1b\xea\x58\x69\x28\x0d\xc6\xad\x95\x81\xab\x43\xff\x70\x12\x96\x99\x48\xa5\x67\x7f\xa0\xa6\x61\x36\xa3\x16\xa4\xbf\xec\xb8\x9a\xdf\x41\x31\xb5\xbe\xdf\x3d\x46\x93\xb7\x80\xd1\x33\xaf\x9b\xf9\xc1\x33\x30\x5b\xe7\x83\x74\xaf\xda\x3b\xa3\x85\x42\x03\x32\x44\x81\xa9\xd1\x0b\x9c\xa9\xb9\x2d\xc7\xd7\x4d\xf5\x31\x87\x2d\xdf\xc7\x6c\xaa\x82\xde\x02\x0e\x2c\x41\x56\x43\xcb\xcc\x42\x80\xe6\xd2\xf4\x37\x1f\xda\x7d\x92\x49\x31\x4a\x8f\x43\x76\x48\x99\x1a\x9b\x03\xd7\x1b\x58\x39\xad\x38\xa1\x55\x5a\xd3\x45\x26\x99\x4b\xa5\x68\x70\xb6\xea\x18\x01\x12\x95\xf2\xca\x2b\x07\x13\xb2\xe9\x2a\xd7\x76\x80\xc0\xdc\x5b\xed\x8d\x3b\x9b\x31\xac\x14\xdf\x76\x99\x49\xc4\xa4\x3e\xa6\x7f\x6d\xee\xb3\xdc\x9e\xd5\x89\xea\x4e\x8a\x2c\xf6\x69\x5d\xf4\x6f\x94\x6f\x14\x67\xb2\x8e\x87\x54\x77\xae\x4e\x64\x50\x80\xfa\xfd\xa6\xdd\x55\x1d\x2c\x02\xfd\x6b\x2b\x19\x4f\xc0\xbd\xb0\x50\xe0\x6d\x4c\x78\x41\x05\xf5\xa3\x3b\x53\xe7\x30\x98\x05\x59\x63\x07\x1e\xfc\x1b\xf3\x97\xfd\x32\x5f\x3a\x6f\x4e\x10\xd7\x6f\x04\x11\xa0\x01\xe6\x2e\xc7\x37\x29\x01\x83\x16\xf5\x63\x10\xf8\x93\xa5\x93\x63\xd1\xf6\xfe\x5c\x17\x44\x4b\x6c\x72\x8a\x49\x33\xb7\x52\x12\xfd\xfa\x25\x8e\x40\x18\xb7\x76\x39\x51\xab\x4e\x50\x96\x41\x1d\xf9\xe5\xbc\x16\xdf\x38\x96\xe4\x6c\x97\x3d\x32\xac\x92\x76\xa4\xe2\xb5\xb8\x0e\x3d\x8d\x79\x8d\xc0\x47\x0b\x45\x09\x6b\x4d\x73\x86\x69\xce\x05\x2e\xd8\x18\xe5\x60\xaf\x1e\x92\xc9\x15\x18\x7d\x66\xcc\x30\x8b\x70",
        );
        test::<Sha384>(
            b"\x82\xb3\x84\x0e\xeb\x95\xc9\xc5\x77\x24\xc7\x0f\x11\x2b\x6c\x2d\xc6\x17\xc3\x17\x85\xac\xd0\xc8\x23\xf8\xbc\xdd\xa2\x85\x32\x5e\xb3\xd3\x08\xdc\x79\x05\x22\xbc\x90\xdb\x93\xd2\x4e\xe0\x06\x32\x49\xe5\x5d\x42\x19\xad\x97\x14\x5f\xea\xf7\xf3\x06\x68\x62\x3c\xc8\x89\x0a\x70\xf4\xf1\x49\x86\x6f\x82\xcf\x86\xf9\x8b\x00\x53\xb2\x3c\x98\xc8\xdd\x5e\x91\x07\xe3\x41\x46\x0e\x9b\xf5\xd8\x8c\xc8\xbc\xd1\xf2\xe4\xc0\x07\xcc\x1c\x02\xc4\x52\x9b\x93\x23\x3a\x0b\x06\xbd\xd1\x59\x25\x85\x4a\xb9\xe3\xf1\x56\xeb\x92\x5b\xf5",
            n,
            e,
            b"\x05\x93\xb9\xfd\x44\x21\x45\x23\x76\xd2\x7b\xc7\xa2\x80\x10\x1c\xfd\x6e\x88\xa6\x72\x7d\x7d\x77\xcf\x65\xce\xb7\x23\xec\xd2\x57\xf3\x2f\xe1\x02\x77\xe8\x57\x98\xe0\xda\x75\x91\x77\x36\xda\x1a\x3b\xfc\x22\xad\xc7\x65\x8f\xbb\x84\xda\x6e\xbe\xa0\xb0\x7d\x1c\xc4\x05\x73\x2f\xb0\x40\xb5\x85\xc1\xb6\x3c\x80\x34\x06\x9b\xff\xb8\x22\x06\x56\xf1\xac\x54\xce\x69\x37\x20\xd6\xfb\x1b\x5a\xec\x67\xb0\x3c\x88\x7c\x80\x77\xda\x14\x8d\x10\xf4\x8a\xf7\xc0\x28\xf9\x92\xb1\x8f\x13\xc0\xe5\x75\x30\xc0\x86\xd7\x75\x48\x3d\xa5\xf6\x6f\x3a\x6a\x19\x18\x78\x68\x34\x0a\xc6\x3c\x62\x12\xbc\xbd\x6c\xbb\x7b\xed\xa8\x62\x0a\xfd\x9b\x66\xde\x47\x47\x3e\xf2\x4d\x1b\x6a\x36\xf4\xec\xe9\xad\xd4\x95\x14\xfd\xf1\xd8\x4c\x7a\x78\x5b\x7f\x0e\x00\xf3\x82\x23\x58\x99\x79\x0f\x47\x2d\x13\xf4\x85\x58\xa4\x31\x47\x42\xf3\x76\x80\x8d\xec\x96\xed\xd2\xe2\x29\xe9\x43\xf7\xb9\x83\xbe\xa5\xec\x6e\xdf\xa5\xe9\xbb\x37\xf5\x88\xe5\x5e\xf6\x2e\xbc\x92\x14\xbe\xaf\x9d\xa5\x02\x43\x4e\x10\x88\xdf\x27\x2c\x6c\x77\xc1\xe1\xd8\x97\xc4\x7b\xea\xb7\x7e\x3b\xbe\x31\x7f\x8d\x43\xd2\x1f\xd7\xe9\x43\x37\xc7\xe2\x63\xe2\x86\x7b\xf5\x80\xa2\xa8\xec\xb9\xe3\x6a\xb7\xd3\xe1\xd5\xcf\x9a\x23\x23\x09\x53\xd5\x9d\xf0\xd7\xe2\x35\x58\xfb\x61\x2b\x79\x18\xab\xba\x31\xb1\x64\xce\x17\x88\x18\xa1\xa9\xe6\xb6\x68\x7f\x4d\xe6\x85\xd7\x0e\x16\xbe\xf6\xe1\x92\xfa\xed\xfe\x0b\x2b\x95\x47\x7d\x37\xb0\xa3\xa2\xd0\x02\xf3\x3e\xf4\x32\x1c\xb9\x05\x04\x0c\xe0\x6f\xda\x1c\x98\xa0\x08\x76\x7f\xbc\x78\x1a\x1e\xaf\x33\x75\xda\xb8\x66\x4b\x59\x03\x36\xb9\x9e\x15\x7b\x86\x87\xa6\x60\x2f\xef\x6a\x3b",
        );
        test::<Sha384>(
            b"\xe1\x53\xcc\xa4\x43\x1e\xd9\x71\x3f\x47\x44\xba\x05\x4f\x5f\x19\x1c\xb3\x7b\x28\x01\x08\xae\x3a\x11\x4a\xd3\x49\xa8\x72\xd1\x30\x8b\x46\x21\x1a\x83\x75\x8a\x3b\x4b\xe3\x2f\xbe\xac\x42\xcc\xfe\xe7\xe2\x3d\xf8\x53\xca\x40\x01\x47\x07\x7b\xb4\x3a\x44\xc1\x2f\x29\x9b\x91\x7f\x3a\xab\xdf\x58\x9e\xeb\x17\x09\xbb\x3d\x60\xb0\x8b\xc7\x1e\xaa\x3f\xfe\xba\x4e\x29\x03\xa5\xdb\xd8\x33\x9a\xae\x85\xfa\x24\xb9\xae\xe7\x61\x30\x00\x06\x05\x85\x7a\x6a\xa1\x97\xd0\x09\x26\x27\x0d\xcd\xa5\x8b\x7d\xe7\x58\xa6\xca\x67\xe6\x17",
            n,
            e,
            b"\xa8\x35\xcd\x41\x46\xbe\xf4\x65\x64\x2d\x49\x49\x36\x26\x8a\x31\x1a\x54\x90\xd2\xc9\xf9\x16\x6c\x6c\xe9\x82\x16\xa9\xa2\x3a\x64\x35\x97\x30\x0a\x00\x50\xe6\x44\x5a\xbd\x5a\x9b\xfc\x7a\x2d\x9b\x70\x72\x6c\x82\x4c\x38\x3b\xf5\xac\xad\xdd\xdc\x34\xd4\x34\xa3\x1e\x53\x14\xd2\x5f\xb5\x8e\x25\x8f\x51\x88\x66\xc1\x36\xe5\x28\x55\xc1\x6f\xe6\x4f\xf8\xf1\xc4\xd6\x6c\x4e\x9e\x39\xb8\xcb\x11\x96\xd8\x09\x44\xd0\x74\x6c\x0a\x3e\x17\x69\xcd\x41\x67\xdf\x72\xab\x5e\x4c\x9d\xba\xe9\xcb\x35\xf4\x82\x8e\x12\x09\x9f\x9b\x36\xa5\xa7\x0c\x48\xd4\xae\xc9\x87\x2d\x7b\x19\xe1\x29\x1b\x33\xcb\xdf\x08\xa2\x26\x3d\x50\x0c\x0a\x83\xb5\x23\x7e\xf6\xce\x92\xde\x34\x4b\x3b\x41\xd0\xd0\x74\x04\xfc\xd5\x46\x7b\x04\x6b\x52\xb8\xf8\x5f\xc6\xb5\xd7\xaf\xc4\x37\xf1\xee\x9e\x78\x39\x0c\xa9\xbb\x6c\xec\x61\x88\x85\xec\xe2\x97\x58\xf2\xfd\x6f\x4e\x5f\x4f\x89\x69\x35\xde\x5f\x67\xcc\x04\x05\x5a\x4c\x4c\x0f\xba\x5d\xef\x8d\x2c\xaa\x17\x93\x31\xa8\x55\x01\xed\x25\x82\x2a\xe7\x9d\xa9\xbc\x81\x5c\xc3\x9c\x6a\x97\x92\x11\x08\x3e\x86\x83\x13\x6c\x94\x2e\x1e\x17\xe9\xeb\x8f\x84\xaa\xcf\x09\x1a\xa1\xe5\x16\x65\xfa\xe4\x46\xbc\x48\xc3\x04\xaf\x65\x39\x1f\x27\x9a\xfb\x98\xb9\x2e\x04\xc2\xb7\x3d\x9d\x94\xe9\x91\x19\x8f\xe7\x78\x1f\x0f\x96\x96\xfc\xba\x2c\x03\x48\x5f\x76\xe6\xde\x30\xb9\x53\x5c\xf3\x90\x3d\xb2\xf3\xaf\xa8\x51\xa4\x7b\xcd\xe7\x2d\x4e\xd2\xe8\xfa\xbf\x9b\xb7\xd4\x69\x6c\xb4\xab\x8c\x28\x9b\x0c\x21\xe1\xf9\x79\xeb\xc5\x32\xe2\x80\xcd\x90\x10\xdf\x4e\xe7\x2f\x84\xbb\x9e\x82\x75\x28\x28\xf1\x67\x03\x0c\x0f\xe3\x48\xeb\xc3\x1e\xc1\x7b\x8f\x07\xd9\x4b",
        );
        test::<Sha384>(
            b"\x9c\x63\x89\x9d\xfc\x7b\xdc\x0d\xb3\x84\x72\x72\x44\xca\xf7\x1e\xcf\xb9\xb8\x79\x2b\x9f\x57\xe9\x36\xb3\xc2\xf5\x69\x55\x65\xa9\xb0\x97\x9f\x3c\x78\xfd\x73\xf0\x09\x81\x81\x3a\x16\xda\x34\x23\x92\xfe\x3c\xee\xc6\xe6\x3f\xfb\xa1\x91\xcb\xeb\x4f\x4b\x90\x05\x0d\x2f\xcc\xd8\x3b\xeb\x06\x22\xb2\xc3\xff\xf1\x59\xd9\xe6\x08\xf3\xab\xcb\x84\x3b\xdd\x56\xc0\x33\x39\xb9\x75\xb9\xf4\xe3\x26\x5b\x32\xf6\xbb\x6c\xcd\xfc\x6c\x57\x52\xd6\xe0\x34\x4d\x74\x96\x99\xc7\x4c\x85\xb3\x0c\x04\xff\x95\xb2\x72\xdb\xcf\xd6\xc7\xd3",
            n,
            e,
            b"\x4d\x38\xa2\x97\x30\x2a\xd0\x77\x0d\x97\x29\xce\x5b\x72\x12\xee\xf2\x87\xce\x02\x50\xf4\x03\xe3\x2b\x4a\xcc\x36\x17\xdc\x0d\x2e\xdc\xcc\xc2\xd5\x80\xdd\xbd\xbc\xa5\x72\x2b\x70\x70\x40\x58\xa3\xb8\x07\xf5\x92\xe4\x00\xbd\x56\x3f\xca\xa8\xb0\x66\xa6\x14\xb4\x90\x6f\x14\x33\x96\x8e\xd2\xf5\x20\xa2\xf6\xb0\x34\xd4\xb2\xd6\x89\x0a\x24\x1a\xfd\x1a\xdb\x86\x39\xa6\xca\xd9\xdb\xfd\x2e\x27\x8d\xfe\xbf\x79\x74\x0d\x75\xf2\x95\x75\x9d\x29\x13\x0b\x19\xab\x19\x98\x3d\xd6\x8f\x77\x9d\xe4\x1f\xfe\xfd\x4e\x82\xb5\xe6\x2f\x72\xf9\x0e\xfb\x73\x43\x7f\x08\xa2\x50\x3d\xd9\x81\x9d\xae\x20\xba\x97\x06\xc1\x99\xde\x9c\xf8\x84\x43\x3e\xeb\x75\x62\x86\xa8\x5e\xae\x14\xbf\x9f\x6d\xbe\xb7\x05\x46\x1d\x91\x82\x22\x82\xf1\x8e\xfb\xb1\x05\x89\xa5\x78\xf2\xc9\xc3\x45\xb0\x79\xa7\xe9\xdd\x07\xfd\x4b\x34\x05\x1b\x27\x11\x97\x29\x90\x6c\x77\xdf\xb7\xd2\xf8\xfa\x6b\xdd\x5f\xaa\x1e\x13\x2b\xfb\xa9\xd3\x91\xe6\x63\x95\xe6\x7f\x01\x35\x3f\xa2\x75\xea\xce\x8b\x53\xaa\x91\xcb\x6f\xb6\x93\xe1\x91\x91\xd4\x2a\x4c\x1a\x85\xa0\xc5\x04\xb1\xc8\x5f\x49\xa4\xd6\x09\x36\xde\xe4\x64\x6a\xca\x62\xa9\x4a\xa4\xbc\x78\x28\xc1\xff\xaf\xde\x8b\xe6\x56\x31\x7d\x50\x6a\xbe\xc1\x79\xcc\x90\x19\x1d\x12\x35\x6f\xf5\x06\x44\xd3\xe0\x1a\xa5\xbc\xfd\xd7\x1d\x3c\x82\x8d\xc3\x53\x9d\xc0\xcf\x3f\xe8\xb9\xb9\x1e\x0c\x25\x24\xf6\xa3\x71\x03\x79\xc9\x0a\xff\xd0\xd0\xa5\x0d\x74\x38\x7f\x9c\xa8\x8b\x46\x46\x3e\xf1\xbd\xba\x58\xcc\x9a\x36\xe5\xc2\xc4\x35\xa2\x0d\x96\x83\x50\xd1\x5d\x94\x1c\x32\x12\xcd\xce\x81\x55\x92\xb3\x10\xd2\x59\x86\x0d\xe1\xdc\x1a\x3d\x70\xac\x22\x30\x2a\x51",
        );
        test::<Sha384>(
            b"\x04\x84\x6c\x2e\x67\x6a\xc7\x31\x60\xbf\x4e\x45\x65\x2b\xdc\x6c\xc4\xd4\xc9\x28\x45\x77\xb4\x32\x0a\xb7\x7f\x6e\xbb\xb5\x9a\x1f\xe0\xe0\x85\x58\x8e\x0f\x90\xb3\x46\xcd\xe6\x44\x1a\xf3\xc9\xd0\x11\x7d\x1f\x3b\xcd\x96\x2e\x40\x6b\xf5\xa4\x65\xab\x6c\xda\x2d\x51\xbe\x59\x8f\xcb\xb2\x9e\xa7\x13\x65\x1a\xac\xd7\xe4\x7d\x22\xd8\xfa\x34\x50\x90\x47\x30\xf5\x17\x92\xea\x37\x47\x61\xa4\xdc\x1f\xc6\xf1\xbc\x65\x7b\x77\x76\x8f\x31\xf4\x63\xe4\x26\x7f\xc8\xdf\xf6\x11\x50\xd4\xb3\x43\xb9\xd5\x37\x59\xcd\xd7\xb9\x80\x94",
            n,
            e,
            b"\x10\x3b\xee\x57\xe2\x5b\xe8\xc3\xa2\xf7\x74\xe7\x39\xb4\x7f\x93\x43\x5e\x41\x49\x32\xc0\x49\x4b\x6b\x6a\xa2\x47\x5b\xf7\xc9\x30\x5c\x73\x74\x7e\x0a\xdf\x82\xc2\x03\x20\x07\xb3\xf7\x5a\x69\xc9\x31\x12\x61\x7a\x62\x56\x6c\x5a\x2d\xea\xa2\x5f\xb9\x52\x09\xda\x49\xfe\x9c\x16\x1c\xb2\xff\xa4\x0f\xd9\xd7\x7f\x1f\xf6\x60\xc8\xb6\xcd\x3b\x54\xe3\xe7\x9a\x75\x9c\x57\xc5\x71\x98\x02\xc9\x31\x1d\xb7\x04\xba\x3c\x67\xb4\xa3\x11\x37\x54\xa4\x1b\x8d\xa5\x9c\x64\x5b\xe3\x90\x9e\x7d\xb7\xe7\xcf\x72\x94\xda\xb4\x4f\x74\x24\x0f\x81\xa2\x81\xee\xcd\x6e\xf3\x1c\x7c\xf1\x8b\x1a\x19\xc7\xd0\x2a\x31\x2b\x91\xd6\xed\xfa\xa9\x54\x46\x2d\x34\x74\x0a\xf5\xab\x70\x8d\xb5\xa1\x0b\x00\xc5\x42\xbe\x82\xfa\x2b\x20\x26\xb0\x9e\xf3\x8a\x40\x01\x45\x7e\x27\xa6\x02\x37\x70\xe4\xb4\xd5\x00\x32\x67\xc8\x5c\x9e\xea\x1d\x5f\x8d\x77\x0b\xd4\x0b\x55\x4d\x5b\x4d\xaf\x14\x6d\xcc\xab\xac\x3e\xa8\xa1\x3a\x05\xc3\xbd\xdf\xc9\x71\xc5\x15\x8f\xac\x02\x7c\xa1\x9b\x72\x32\x62\x1e\x9d\x2e\x37\xb6\xa6\x55\xaf\x54\x5e\x44\xa2\x98\xbe\x78\xcd\x47\x5c\x22\xa4\x8b\xff\x7c\x34\x94\xa5\xf8\xa6\xab\xdf\x1a\x46\xf9\xde\x08\x2e\x37\x4f\xd5\x98\x86\x7d\x61\xe4\xd5\x1d\xae\xd8\x41\x52\xe4\x3c\xc6\xa2\xaf\xfa\xe2\x05\xed\xc5\x26\x13\x48\x0d\x41\x1a\xba\x84\xfc\xc9\xb6\x9d\x1c\x28\xf1\x6f\x76\x83\x69\x01\xa7\xc5\xb3\xeb\x2f\x2c\x94\x0d\x0a\x3f\xad\x38\xa8\xef\xab\x96\x8a\x0c\x85\xeb\x22\xe1\x1d\x3d\x08\x61\x13\x6c\xed\x5f\x06\x73\x4f\xdf\x8d\x4f\x15\x1d\x23\x86\x1b\x1c\xba\x9b\x9c\x58\x0d\x33\x50\xc7\x6d\x4d\xc8\x08\x46\x1d\x5f\x87\x2e\xc5\x48\xb2\xb4\x27\xdf\xf7\x4b\x1d\x1a",
        );
        test::<Sha512>(
            b"\xdb\x6c\x9d\x4b\xad\xb1\xd9\xb7\x4d\x68\x34\x64\x48\xb4\xd5\x34\x06\x31\x78\x3b\x5a\x35\xac\x24\x58\x56\x3e\xd0\x67\x2c\xf5\x41\x97\x58\x7f\xb7\x34\xc4\xac\x18\x9b\x2d\xda\x95\x4c\xdf\xb1\x8b\x41\xc0\x10\xa7\x7e\x90\x46\x4e\xea\x6f\x86\x3c\x5d\xa0\x95\x6b\xfa\x8c\xc6\x36\xbf\x0a\x28\xbe\x5a\xdd\xfe\x8d\x3e\x7e\x6f\x79\xf7\x1d\x7f\xcb\xba\xe2\x3e\xa1\x41\x78\x3f\x91\xd6\xcc\x4c\x8f\xad\x12\x58\x11\x76\x0a\xb5\x71\x33\x81\x88\x92\x47\x1a\x79\xc6\xd0\x4e\xaf\xef\x37\xb2\xfb\xe5\x06\x78\x53\x18\xf9\x39\x83\x77",
            n,
            e,
            b"\xd4\x80\xd5\xa9\x79\xad\x1a\x0c\x4c\xa3\x29\xeb\xd8\x8a\x4a\xa6\x94\x8a\x8c\xf6\x6a\x3c\x0b\xfe\xe2\x25\x44\x09\xc5\x30\x54\xd6\xff\xf5\x9f\x72\xa4\x6f\x02\xc6\x68\x14\x6a\x14\x4f\x8f\x2b\xa7\xc4\xe6\xb4\xde\x31\x40\x0e\xba\x00\xae\x3e\xe8\x75\x89\xdc\xb6\xea\x13\x9e\x70\xf7\x70\x4f\x69\x1b\xc3\x7d\x72\x2f\x62\xbb\x3b\x2c\xd3\x03\xa3\x4d\x92\xfd\xe4\xde\xb5\x4a\x64\xdd\x39\x18\x43\x82\xd5\x9c\xca\xf0\xc0\x7a\x7e\xa4\x10\x7d\x08\x08\x26\x0e\xd8\xd4\x21\xcb\x8b\x14\x07\xcd\xf9\xe9\x15\x15\x92\x82\xb9\xf7\xbf\xfd\xbf\x40\xd8\x77\x88\x5d\xa7\x39\x9e\xde\xbd\x30\x0a\x7e\x77\xa9\x08\xf7\x56\x65\x9a\x18\x24\xf9\x5c\x8a\x81\x2a\xa5\x40\xeb\xaa\x64\xab\x54\xa2\x33\x72\x3d\xb5\x5c\xaa\x8b\x44\x66\xea\x9a\xe6\x61\x4a\xd1\xbb\x86\x9e\x9d\x8e\x0d\x03\x2f\x39\x01\x67\x1e\x94\xc0\xb6\x73\xbe\x65\x37\xcd\x54\x27\x8e\xd3\xda\x2e\x1e\xdb\xc0\x4e\xe3\xa9\xe8\x07\x0d\x73\xba\x0f\xfb\x93\xe6\x0f\x30\xb8\x7f\xf3\x86\x2e\x9c\x53\x90\x8f\x2c\x8e\x99\x91\x56\x68\xc1\xf4\x66\x35\xe0\x5b\xf7\x16\x30\x51\xff\x9d\x92\xbc\x71\xa6\x26\x55\x3c\x69\xdf\xdd\x06\xa4\x9f\x7f\xf1\xed\x51\xe9\x18\xf3\xed\x80\x1d\xae\x62\xca\x27\x6d\x70\x63\xd7\x2a\x6e\xbc\x13\x6b\xa0\x6c\xfe\xdf\x5a\xa2\x32\x77\xe8\x10\x08\xc6\x3b\x2e\x00\x83\xd0\xfd\x68\x14\xf6\xd4\xb4\xb4\x0a\x42\xe8\xc0\x20\x6f\x3c\x35\x6a\x5e\xc7\x09\xb7\xc8\xa4\xb7\x4b\x7b\x48\xd5\x3c\x9d\x86\x94\xd2\x73\x59\xc2\xc7\x70\x19\x38\xd2\xf0\x16\x17\x21\xa5\x73\x13\xbb\x1a\x2e\x11\xda\x21\x58\x72\x49\x81\x82\x49\x3d\x85\x17\x04\x3b\x4c\x03\xf9\x34\x46\xaa\xc9\x38\x30\x27\x65\x42\x02\x6c\xe8\x30\x55",
        );
        test::<Sha512>(
            b"\xd5\xdd\x3b\x6c\xe9\x77\x2d\x9a\x97\xfe\x21\x64\x84\x97\x78\x3b\xac\x5b\xb5\x25\x4a\xad\x82\xb6\xf7\xcb\xf4\x3b\x15\xa4\x0f\x38\x6e\xea\x8d\x15\x19\x67\xdb\x14\x9e\x94\x65\x86\x59\x68\x13\x3f\x24\x6e\x13\x47\x30\x1a\xda\xd2\x34\x5d\x65\x72\xca\x77\xc5\x8c\x15\x0d\xda\x09\xa8\x7b\x5f\x4d\xa3\x6b\x26\x6d\x1f\xa7\xa5\x9c\xcd\x2b\xb2\xe7\xd9\x7f\x8b\x23\x15\x43\x19\x23\x53\x0b\x76\x2e\x12\x6e\xac\xaf\x5e\x5a\xc0\x2f\xf1\xaa\xef\x81\x9e\xfb\x37\x3c\xf0\xbb\x19\x6f\x0e\x82\x9e\x8f\xe1\xa6\x98\xb4\x79\x0a\x2a\x05",
            n,
            e,
            b"\xbf\x9e\x8b\x4f\x2a\xe5\x13\xf7\x3d\x78\x89\x58\x00\x37\x33\xdb\xe2\x09\x57\xb1\x47\xb1\x7c\x3f\x4f\xd6\xd0\x24\xe8\xe8\x3f\x07\xb6\x5d\x9f\x3d\xbc\x3b\x1f\xe8\x4d\xa0\x21\xce\xab\xfc\xcd\x8c\x57\xa0\x14\xfb\xe5\xa2\xbc\xe3\xe4\x05\x1b\x7d\x03\xe0\x9f\xc0\x35\x0b\x6a\x21\xfa\xd2\x14\xae\x7a\x07\x32\x77\xc7\x7a\x40\xdc\x44\xa5\xae\xea\x51\x94\xa7\x56\xb6\x9c\x93\x97\x7b\x69\xee\x92\x94\x36\x0e\xaa\x73\xa5\x74\x54\x8f\xa6\xa9\x74\xa7\xcd\x5a\x6a\xdc\xf0\x9e\x80\x63\x11\x56\xaf\x85\xa8\xe5\xc5\x31\x7e\x18\x9e\xea\xd4\x7e\x2e\xad\x65\xc3\x81\x39\x6b\x5c\xac\xde\x26\x0e\x93\x72\x84\xa8\xe9\x0e\xff\x2c\xbc\xb9\xde\xe2\x29\x25\xf2\xf7\x25\x6f\x74\xc6\x7c\xf3\xff\xc7\xb8\xce\x65\x7e\x8d\x13\x5f\x0f\x37\x6d\x9d\x93\x6a\x79\x79\x2c\x98\x16\x14\xd9\x8e\x3f\x7d\x66\x2a\x4f\xd4\x6d\xcd\xa9\x69\x16\xb3\x2f\x36\x6e\xd2\x7d\xab\x18\x8f\x18\x4b\x98\x4d\xf0\xb5\x59\x71\x0d\x8f\xf2\x04\x0b\xe4\x62\xf9\x19\x43\x50\x1b\xda\x48\x40\xfd\xd5\xc8\xec\x15\xd1\x89\x06\x4d\xef\x75\x6e\x54\x5d\xb3\x19\xe0\x07\xc4\x33\xf0\x46\x8a\x67\x23\x35\x7b\xa4\x7d\x15\x6a\xb7\x65\x2b\x06\xae\x2b\x18\x87\x4f\x07\x71\xc6\x26\x46\x6d\xbd\x64\x23\xe6\xcb\xc5\x18\xb5\xe4\xae\x7b\x8f\x15\xe0\xf2\xd0\x47\x1a\x95\x16\xdf\xa9\x59\x16\x97\xf7\x42\x86\x23\x24\xd8\xd1\x03\xfb\x63\x1d\x6c\x20\x73\xd4\x06\xb6\x5c\xde\xe7\xbd\xa5\x43\xe2\xe9\xeb\xff\x99\x06\x98\x5d\x1c\xb3\x65\x17\x2e\xa6\x23\xed\x7a\xa4\xc7\xa3\x22\xf0\x98\x46\x80\xe3\x4e\x99\xbc\x62\x31\xb0\x2e\x3d\x14\x58\x16\x08\xbc\x55\xbc\xa7\xfb\xe2\x2d\x7f\x03\xe9\x04\xda\x45\x52\xe0\x09\xe5\x60\x7f\x04\x18",
        );
        test::<Sha512>(
            b"\x59\x16\x52\xb6\xeb\x1b\x52\xc9\xbe\xbd\x58\x32\x56\xc2\x22\x86\x80\x11\x0b\x87\x89\x17\xde\xa5\xad\x69\xe8\xc5\xd2\xab\x51\x42\x77\xb0\xac\x31\xe7\xe2\xcc\xea\xb2\xe5\xd9\xc4\x5d\x77\xa4\x1f\x59\x9b\x38\xa8\x32\xf6\xb2\xd8\x09\x79\x52\xbe\x44\x40\xd1\xff\x84\xba\xf5\x1b\xd7\x0b\x64\xf1\x30\xae\xb6\x86\x14\x5f\xcd\x02\x95\x38\x69\xfb\x84\x1a\xf7\xf6\xe3\x4e\xaa\x2b\x99\x6c\xcd\x89\x69\x7c\x58\xfa\x25\x5c\xc1\xe8\x1f\x62\x14\x00\xe1\x41\x46\x36\x1e\x31\xc7\x09\xe8\x4a\x56\x08\x22\x31\x19\x95\x39\xf7\xed\xe9",
            n,
            e,
            b"\x1d\xe7\x9d\x72\x16\xdd\xe1\x25\xde\xb7\x7c\x34\xd9\x0a\xb3\x21\xa4\xde\x5f\xb1\x1c\x29\x66\x56\xad\x9b\xf9\xa2\x46\x53\x59\x11\x17\xac\xe4\x15\xe1\x8e\xad\xce\x92\x82\x3f\x31\xaf\xe5\x6f\xc8\xe2\x94\x94\xe3\x7c\xf2\xba\x85\xab\xc3\xba\xc6\x6e\x01\x95\x84\x79\x9a\xee\x23\x4a\xd5\x55\x9e\x21\xc7\xfd\x4f\xfd\x24\xd8\x26\x49\xf6\x79\xb4\xc0\x5d\x8c\x15\xd3\xd4\x57\x4a\x2e\x76\xb1\xf3\xee\x9f\x8d\xec\x0a\xf6\x0b\x0c\xed\x1b\xe8\xa1\x9c\x2f\xa7\x1b\xcb\xc1\xfb\x19\x08\x99\xec\x85\x56\x95\x8e\x07\x82\xac\xe7\x19\x6b\x36\x65\x86\x56\xcf\x36\x4d\x37\x73\xde\x86\x26\x0f\xd8\x98\x76\x04\xef\x35\xea\xe8\xf3\x8e\xc2\xcb\x0d\xa8\x64\xcc\xa7\x19\x21\x9c\x2a\xd7\x1c\x08\x50\x6c\x41\x2e\xc7\x79\x95\xf3\x74\x39\xc8\x56\x97\x7b\x71\xdf\xb9\x64\x79\x90\xef\x70\xfa\xf4\x32\x73\xae\x60\x83\x9c\xd0\x67\x9e\xc9\xaa\x42\xbf\x91\x4e\x42\x1b\x79\x7c\xba\x21\x8a\x40\x0f\xf9\xdb\xaa\x20\x6c\xb9\xc2\xb0\x59\x6c\x70\x9a\x32\x2b\x73\xcb\x82\x72\x1d\x79\xf9\xdb\x24\x21\x1b\xf0\x75\xa1\xce\xf7\x4e\x8f\x6d\x2b\xa0\x7f\xe0\xdc\x8a\x60\xf4\x8a\xf5\x11\xad\x46\x9d\xcd\x06\xe0\x7a\x4c\xe6\x80\x72\x13\x9c\x46\xd8\xbe\x5e\x72\x12\x53\xc3\xb1\x8b\x3c\x94\x48\x5c\xe5\x5c\x0e\x7c\x1c\xbc\x39\xb7\x7b\xc6\xbb\x7e\x5e\x9f\x42\xb1\x53\x9e\x44\x2d\xa8\x57\x65\x8c\x9e\x77\x1c\xcb\x86\xbe\x73\x97\x64\x7e\xfb\xc0\xcc\xb2\xc3\xad\x31\xac\x4e\x32\xbf\x24\x8c\xc0\xce\xd3\xa4\xf0\x94\x52\x6b\x25\x63\x1c\xb5\x02\x47\x09\x61\x29\xb0\x8a\x9c\x2c\xdf\xb7\x75\x97\x8b\x0f\xee\xe2\x65\xa6\xc4\x19\x91\xc1\xdc\x44\x52\x61\x5b\x78\xc9\x06\xc7\xed\x1b\xd2\x07\x96\x9d\x98\xd0",
        );
        test::<Sha512>(
            b"\x8d\xff\xaa\x91\x51\x27\x1a\xd2\x26\x22\xf2\x28\xc8\x92\xe1\xd9\x74\x8b\x3c\x39\x43\x97\xf2\xcb\xb6\xfe\xbe\xaa\x92\x44\xa0\x27\xee\xf2\x8d\xb4\x8a\x9a\x66\x01\x62\x15\x27\x64\x83\x0f\x61\x7e\x1e\xc6\xea\x1c\xdb\x0e\xd2\x5b\x6f\x99\x9a\x10\x71\x75\xa1\x66\x69\xd6\xdf\xc9\x2b\x16\xd5\x03\x63\xfa\xc4\xa5\x70\x37\x1e\xa9\x76\x34\x3a\x55\xae\x12\x4b\x63\x01\xea\x93\x5e\xd6\x55\xd4\x4f\x28\x32\x08\x99\xdb\xa3\x51\x22\x50\x59\x33\xb3\x37\x12\x01\xa2\xa4\x5f\x95\xae\x65\xab\x44\x2a\x94\x79\x12\x5e\x68\xed\x21\x2a",
            n,
            e,
            b"\xb3\x29\xae\xf8\x3a\x56\xdd\xc5\x7c\xd9\xa0\xe1\x5e\xb0\xb0\xb7\xae\xa7\xd7\x8d\x5e\x8c\xa3\x98\x2b\xd3\x1c\xc8\x25\xa0\xcd\x1c\x44\x4d\x9f\x7b\xea\x9e\x7a\x27\xf3\xbb\xb3\x76\x10\x60\xff\x95\xfe\xe1\xa3\xe8\x64\xd2\x10\x8f\xc4\x0b\x64\x78\x6a\x96\xa6\xd6\x2d\x20\x12\x17\xe0\x3a\x8b\xa2\xc0\x7e\xe9\x4c\x26\x71\x49\xd1\xe7\x2c\xc5\x77\x9b\x73\x7e\x85\x47\xac\xd6\xaa\x4b\xba\x3f\xf3\x8b\xf9\x68\x7e\x9e\x82\xf5\x11\xb5\x97\xad\x7e\xc1\xd7\x95\xc3\x6a\x98\xbf\x83\xa9\x0f\xc8\x6b\x0c\xad\x41\x95\x33\x60\x73\x89\x21\x93\x6a\x45\x86\x74\xb2\xe9\xa7\x01\x2a\xc3\x02\x9f\xdb\x0a\x9d\x12\x31\x82\x02\xd2\x54\x4a\x0d\x97\x6e\xe5\x36\xe0\x3b\x7e\x8d\x89\x4b\x3b\x9c\x76\x2d\xab\x01\x10\x84\x9c\xc1\xea\xad\x74\x7e\x3d\x88\xd7\xdc\xf4\x9f\x82\x4d\xf0\x27\xe6\x45\xc0\xb9\x29\x4e\x65\x5d\x9f\xc9\xe1\xef\x95\xeb\x53\xaa\xff\x57\x75\xc3\x49\x48\x6d\x4b\x5d\x67\xdb\xa2\x9b\x62\x17\xf8\xb9\x97\x66\x12\xb5\x7e\x16\xfc\x1f\x99\x98\x3f\x2a\xf0\x45\x79\x93\x86\x06\x87\x9b\x7c\x72\x53\xe8\x70\x71\x4b\x4f\x0f\x24\xe2\x6d\xc8\xc7\xa6\xfc\xef\xfb\x5f\x98\xe3\xb2\xfb\x5d\xb9\x49\xd2\xf9\x8c\xd1\xae\x1a\xa5\x52\x69\x6b\x48\xc3\x9f\x67\x8e\x15\x43\x51\xcc\x75\x6d\x3e\x9a\x97\xf7\x92\x79\x85\x3e\xbd\x0d\xb9\xae\x68\x59\xfb\x2d\x57\x21\x38\x5d\x06\xf5\x56\x5a\x3a\x8f\xf0\x99\x2d\x51\x7a\xcd\xa1\xaf\x69\xa9\x28\x54\xa1\xb3\x2a\x79\xcb\x9e\x44\x2a\x90\xb0\x55\xbb\x2e\xc3\xaf\x8d\x99\x26\xa0\xd8\x57\xe3\xcb\x1e\x7e\x4a\x73\x00\xd1\xac\xcb\x94\x92\xec\x78\x32\xaf\x45\x35\x29\xff\x0f\x4a\x6a\xd3\x25\x97\x57\xf7\x07\xf7\x13\xaa\xa5\xdf\x23\x1f\x74\x87",
        );
        test::<Sha512>(
            b"\x71\xd4\x16\x3e\x70\x8c\x12\x1e\x93\x1b\xb9\x69\x2b\x21\x7d\xdd\xd3\x5c\x73\x46\xf6\x1c\xfc\x95\x91\xf7\xa4\x31\x3a\xbd\x4a\x92\x62\xaf\x82\x0b\xd7\xeb\x37\xe7\x8c\x2b\x95\xb8\x9d\xaf\x25\xec\x8e\x78\x3a\xa1\xd4\xb7\x8d\xbb\x96\x85\x24\x33\xb4\xd4\x78\xb1\x09\xa6\xd6\x5e\xed\x7d\x06\xf3\xfe\x12\x2b\x17\x21\x49\xea\xe7\xc3\x65\xce\xd6\x65\x78\xeb\xb7\x57\x1e\xc2\x18\xc3\x6b\x65\xd2\xee\x22\xdc\xde\xbb\x28\xc6\x6a\x71\x38\x43\x2c\xbd\xd7\x12\xf7\xfb\x8b\xf7\x8c\xb1\x48\x60\xb2\x5c\x2b\x47\x89\x70\x6b\x5a\x1b",
            n,
            e,
            b"\x25\x22\xee\x3b\xda\x30\xc0\x43\x4e\x54\xb1\x99\xda\x8c\x97\x33\x96\x4f\xd4\x02\xb7\x07\xf5\xb3\x30\xf4\xf7\x54\xa0\x50\x2c\x7a\x71\x3c\x78\x14\xf0\xe8\x51\xa4\xa4\xdb\x72\x69\x0d\xb9\x6e\xa8\xb8\x81\x3b\xd8\x62\x9a\x94\x8b\xb3\x0c\x1b\x82\x72\xa8\x16\xb3\x0a\x75\x5f\xc6\xfb\x17\x54\x16\x7c\x3e\xb1\xf1\x94\x39\x59\x07\xa5\x6c\xf5\xa7\x3b\x41\x54\x38\x3a\x05\xb7\x8b\x73\x1f\xed\xd9\x07\x7f\x3c\x22\x67\xa5\xcf\x92\x66\x97\x87\x1f\xe0\xa4\xbe\xd9\xc2\x19\x55\x2d\xd1\xc8\x7a\xff\x50\x61\x30\x94\xbc\xaa\x2d\xec\x42\xa3\x53\x80\xa6\xba\xc6\x73\xda\x25\x94\xf8\x24\xa8\xf3\x2f\x21\xd7\x59\x3a\x3e\x49\xc7\x8e\xe2\x80\x19\x3a\x47\x86\x21\xd3\xb0\x95\xc1\x6d\xce\x72\x93\x53\x14\xd4\xa2\x32\x3e\xeb\xe7\x85\x5c\xa4\x73\x8a\x19\xb5\xa3\x1a\x5f\x95\xab\x91\xfb\xe1\x28\x9c\x02\xfe\xa7\xa6\x5b\x91\x32\x7b\x7b\x97\x90\x55\x62\x89\xe1\xb9\x88\xe4\x5d\x50\xeb\x8c\xea\x15\x81\xde\x5d\x5d\xfd\x21\x00\x1c\x73\xb4\x39\x21\xd8\xb2\x1b\x96\x44\xb0\xf2\xb9\x6e\xe6\xb0\x9d\x73\x70\x9c\x33\x33\x81\x43\xd6\xa2\xfe\xc5\x59\xa4\x36\xc5\xec\x86\x5d\x3a\xcc\xa5\xfe\xe6\x54\xf1\x32\x5a\xe5\x72\x55\xdf\xd4\x21\x88\xc8\x4d\xcb\x1f\x7c\x1e\x86\x02\x8a\x74\xe3\x1d\x73\x60\x78\x74\x1e\xe9\x7c\x39\xa5\x6e\x4d\xe0\x0f\xc1\x2b\x80\x51\x83\x5b\xbd\x0d\x8f\xca\xe7\x37\x32\x20\x99\xad\xc1\x01\x71\x07\x02\x2d\xd1\x5c\x11\x4d\xa5\x7e\x78\xb9\x56\x81\xba\x99\x45\x61\x5b\x59\xda\x90\xf5\xa2\xa9\x9a\x25\x2e\xb4\x2b\x20\x06\xee\xdd\x6e\x78\x47\x6c\x29\x05\x47\x3e\xe6\xb4\xf2\x3c\x1c\x5c\xf0\xb8\x04\x51\xc5\x42\x6e\xa0\x09\x14\x1c\xb3\xfc\xb0\xdf\x2d\xed\x92\xbe",
        );
        test::<Sha512>(
            b"\xd0\x0e\x15\x29\x22\x8c\x79\xa2\x0a\x1c\x36\x68\xff\xa4\xa5\x41\x40\xbb\x17\x0b\xc5\xc6\x69\xfd\x75\x60\xd9\x30\x99\x00\x17\x5e\x91\xd5\xa0\xe9\xc5\xf5\x47\x1f\xdf\xb7\x14\xbc\x38\x5d\x52\xb0\x8f\xf7\xe4\x23\x01\x84\xd8\xb7\x35\x59\x3f\x0d\xd8\xc7\x3b\x8a\x49\xf8\x59\x5b\x95\x1a\x21\xb6\xa5\xbf\xec\x63\xb6\x84\xf6\x7c\x0a\xf1\xb4\x71\xdd\xa1\x68\x4e\x9b\xa3\xf2\x41\x50\x1f\xe9\x57\x60\x3d\xea\x86\x78\x42\x30\xf0\xc4\xfd\x65\x66\x63\x61\xb8\x2b\x18\x73\x30\xfb\x42\x67\x40\x4c\x0e\x05\x9b\xd4\xeb\x52\x49\x4b",
            n,
            e,
            b"\x18\x35\xdd\x97\xe5\x09\x3a\x33\xce\x1e\x62\xd6\x83\x86\x3f\x6b\x35\x07\xf3\x58\xa6\x2f\xc8\x79\xb5\x24\x35\x0f\xbc\x73\x30\x68\x1c\xb0\xc6\x82\xee\xf4\x33\x04\x19\xca\xf8\x54\x3b\xd9\x26\x9b\x6d\x91\xd8\xe1\x07\xec\x38\xb6\xe9\xc6\xea\xab\xf9\x06\x45\x72\x05\xd5\x2a\x90\x0e\x05\x57\x9a\xa1\x1f\xc5\x81\x37\x52\x64\xe6\x9a\x92\x57\x98\xe5\xa3\x48\xe5\xa1\x6f\x15\x67\xd5\xd0\xe4\x08\x53\x38\x0b\x34\xde\xac\x93\xad\x73\x77\xaa\xe8\xa2\x7b\x09\x0d\x0d\x3a\x92\xbf\x7a\x82\x4d\x92\x6e\x2e\x35\xa0\xc3\xbd\x0e\x99\x0b\x59\x11\x20\xd7\x4d\xd9\xb0\x52\xa7\x35\x68\xe3\xc3\xf2\x9c\x5a\x77\xfb\x1c\x92\x1b\xce\x9c\x1e\x7f\x76\x4a\xa6\x7b\xac\x11\x9f\x58\x39\xa5\x30\x38\x60\xed\xeb\x63\x48\x14\xc2\x38\x6c\x83\x1f\xee\x62\x00\xcf\x55\xb6\xbf\xea\x05\x8b\x79\x5a\x0f\xcf\x26\xeb\x72\x16\xae\x1b\x75\x87\xc8\x2e\x56\x85\xe5\x84\x17\x0c\xbd\xdc\x89\xa7\x7e\x09\x89\xd4\xce\x5c\x3c\x7f\xdb\x66\x4a\xae\xaa\xdb\xce\x1f\x23\x1e\x64\x79\x8f\x6f\x9a\x85\x45\x6b\x5a\x93\xa5\x02\x12\x6a\x80\xe2\xd2\x1f\x46\x92\x1c\xc3\x60\x1f\x5e\xcd\xbd\x56\x99\x8a\x63\xb8\x65\xfc\xe7\xeb\x29\x9f\x76\xaf\x40\xe9\x12\x81\xbf\xc0\x19\xf4\x0e\x0d\x46\x81\x1e\x38\x36\x91\xe4\x02\x4c\x94\x56\x6f\x18\x02\x4f\xf2\xb2\x2a\xa7\xe1\x27\x02\x33\xff\x16\xe9\x2f\x89\xc6\x85\x09\xea\x0b\xe2\xd3\x45\x11\x58\x1d\x47\x22\x07\xd1\xb6\x5f\x7e\xde\x45\x13\x3d\xe8\x7a\x5f\xfb\x92\x62\xc1\xff\x84\x08\x8f\xf0\x4c\x01\x83\xf4\x84\x67\x99\x6a\x94\xd8\x2b\xa7\x51\x0c\xb0\xb3\x6c\xf2\x54\x82\x09\xa5\x06\x03\x37\x5c\xb8\x2e\x67\x8f\x51\x49\x33\x45\xca\x33\xf9\x34\x5f\xfd\xf5\x4b\xe9",
        );
        test::<Sha512>(
            b"\xa3\x59\x26\x68\x55\x61\xf0\x9f\x30\x92\x5e\x94\xd7\x4e\x56\x61\x89\x2a\x2d\xdd\x52\x4f\x75\x1f\x83\x21\x16\x3d\x61\x1e\xa1\x59\x1a\x08\xe0\xdf\xfd\x46\xb2\x08\xe9\x88\x15\xa3\x06\xaa\x85\x14\xb4\xdb\x85\x9d\xc1\xfe\x7b\xdc\xdf\x50\xc0\x95\x55\x4b\xf8\xb2\xf4\xcb\x9f\x88\x4d\x70\xe5\x5c\x21\x43\xbc\x26\x19\x9c\x2f\x94\xb7\x43\xf5\x52\x8d\xd5\x46\x89\xad\x69\xed\xa6\x60\x74\x9f\x5c\x1b\xea\x8b\xec\xae\xa6\x32\xa4\xbf\x0c\x79\xa5\x77\xed\xfc\xea\x7b\xaa\xa6\x86\x1e\x9d\x7f\x2d\xd5\xb4\xc4\xf6\xeb\x5f\x3d\x5f",
            n,
            e,
            b"\xb1\xa9\xc4\x5a\x26\x4d\x2c\x9a\xf4\x41\xa7\xb2\xd3\x30\xdd\x78\x80\x89\xcc\xef\x20\x5d\x5d\x66\x6b\xfe\x86\x43\x67\xbe\x97\x38\x12\x4e\x9d\x74\x64\x8a\xd9\x91\x60\xbd\x3a\xf8\x1a\x81\x85\x8b\xab\xe6\x67\xa5\xd9\x5c\x98\x0f\xe2\xf6\xac\x34\x86\x1e\xb2\xec\x9b\x4b\x4e\x8b\x64\x2e\xf3\x82\x0f\x56\xca\x38\x8a\x55\x65\x30\xd4\x27\x54\xc4\x72\x12\xe9\xb2\xf2\x52\x38\xa1\xef\x5a\xfe\x29\xbe\x63\x40\x8c\xf3\x8c\xaa\x2d\x23\xa7\x88\x24\xae\x0b\x92\x59\x75\xd3\xe9\x83\x55\x8d\xf6\xd2\xe9\xb1\xd3\x4a\x18\xb1\xd9\x73\xff\xac\xcc\x74\x5e\x52\x7c\xe7\x6c\x66\x3e\x90\x37\x19\x35\x5e\x45\xcd\x6d\x11\x8e\xd0\xb8\x5b\x70\xcb\xb8\xe4\x96\x41\x13\x53\xf8\x4f\x88\x66\xa0\x1f\xad\xc8\x19\xca\x0f\xf9\x5b\xbe\x2c\xc6\x8c\x8c\xf7\x8d\xa5\x58\x1b\xec\xc9\x62\x47\xb9\x11\xd1\x85\xed\x1f\xae\x36\xc4\xca\xd2\x62\x08\xeb\x80\x88\x3f\x42\xa0\x81\x23\xda\xc6\x8d\x88\xf2\xf9\x89\x3c\xde\x02\xef\x5a\x57\x66\x1d\xb2\xb3\xe1\xe9\x26\x9c\xbb\x0e\x15\xc4\x07\xbc\xf5\x5d\x92\xe6\x79\x38\x3c\x90\x80\x2c\xd0\xbf\xfd\x46\x96\x46\xdc\xb6\x0c\xa0\x1a\x1d\xea\xd4\x32\x28\x93\x40\x18\x39\x1d\xd8\x1f\x8b\x7e\x79\x7e\x52\x7f\xbe\x18\x15\xb9\x1b\xf3\xcd\x6a\x1f\x2f\xfb\xf5\xdd\x16\x6a\xcd\x55\x26\x76\x1c\xa8\xba\xb5\xd4\x63\xfb\x9f\xb8\x20\x65\x9f\x5c\xd5\x0f\x81\x50\xf1\x2f\x7e\x8d\x52\xe7\x77\x73\xc1\xe6\x48\x0c\x2c\xc1\x84\xd4\x11\xd6\x41\xf7\x1a\x9d\xed\xc2\xc5\xfc\x2e\xc3\x7a\x27\x70\xa9\x38\x3b\xfb\xf6\xa4\x89\xcf\x32\xb5\x6a\x12\xcf\x99\x37\x8e\x39\xb5\x0b\xda\xdb\x9f\x05\x91\xb2\x06\x5f\x9d\x44\xe5\x11\xc9\xdf\xb6\x15\x8f\xdd\xdd\xd1\xbc\x2c\xec\xe6",
        );
        test::<Sha512>(
            b"\x12\x71\xa0\xdd\xb9\x9a\x0e\x1e\x9a\x50\x1c\xa3\x3c\x13\x1b\x0a\x1c\x78\x20\xa3\x97\x79\x08\x69\x09\x0f\xba\x37\x37\x03\xac\x38\xea\x00\xa9\xa0\xdd\xee\xd1\x99\xd9\x7b\xe1\x80\x1f\xfa\xb4\x52\x06\x71\x0a\x61\xe5\xed\x89\x4c\x33\x19\x01\x2d\xed\x0f\xf4\x14\x38\x6e\x56\xb5\x48\xad\x91\x5d\x80\xaf\xcc\x2b\xdb\x97\x6d\x7c\x8a\xdd\xdc\xa7\xdf\xa2\x8a\xeb\x69\x40\x33\xa5\x61\x26\x60\xc6\x44\xe3\x2f\x85\xc2\x80\x56\x51\xd7\x13\x66\x0a\x38\x91\x4d\x70\xf0\xe4\x1f\xdc\x4b\x3d\x16\x2e\xf3\xac\xd7\x06\x59\xee\xf6\x37",
            n,
            e,
            b"\xbf\xfd\x01\x0b\x2e\xc4\xe4\xa3\x27\x77\xb7\x76\x19\xb8\x76\x22\xf8\x92\x1d\xab\x56\xe1\x02\xc8\xd8\x24\xfe\x52\xb5\xdf\x7a\x20\x3f\xe7\x17\x99\xee\xaf\xdc\xc0\xc8\x87\x2d\xba\x6a\x37\x44\x07\xb5\x63\x9a\xeb\x5a\x30\xa9\x04\x71\x2f\x15\x09\x7d\xba\x0f\x2d\x62\xe8\x45\x41\x23\x95\xcf\x09\x54\x0a\xbd\x6e\x10\xc1\xa2\xe2\x3d\xbf\x2f\xe1\xdf\xd2\xb0\x2a\xf4\xee\xa4\x75\x15\x95\x7f\xa3\x73\x8b\x06\x41\x1a\x55\x1f\x8f\x8d\xc4\xb8\x5e\xa7\xf5\xa3\xa1\xe2\x6c\xcc\x44\x98\xbd\x64\xaf\x80\x38\xc1\xda\x5c\xbd\x8e\x80\xb3\xcb\xac\xde\xf1\xa4\x1e\xc5\xaf\x20\x55\x66\xc8\xdd\x80\xb2\xea\xda\xf9\x7d\xd0\xaa\x98\x33\xba\x3f\xd0\xe4\xb6\x73\xe2\xf8\x96\x0b\x04\xed\xa7\x61\x61\x64\x39\x14\x24\x2b\x96\x1e\x74\xde\xae\x49\x7c\xaf\x00\x5b\x00\x51\x5d\x78\x49\x2e\xc2\xc2\xde\xb6\x0a\x57\xb9\xdc\xe3\x6e\x68\xdd\x82\x00\x7d\x94\x2a\xe7\xc0\x23\xe1\x21\x0f\x0b\xe8\xa3\xeb\x3f\x00\x48\x24\x07\x4b\x8f\x72\x5e\xaf\x8a\xc7\x73\xe6\x0f\xbb\xb7\xcb\xa9\x63\x0e\x88\xb6\x9c\x8b\xcb\x2d\x74\xdb\xdb\x29\xbf\xff\x8b\x22\x54\x5b\x80\xbb\x63\x4e\x4c\x05\xf7\x3e\x00\x2a\x92\x8e\xfd\x5a\x6a\xa4\x56\x21\xce\x1b\x03\x2a\x22\x44\xde\x48\xf4\xdf\x43\x58\x15\x66\x78\xcb\xe0\x39\xc9\xeb\xe4\xce\xe9\x45\xa2\x5b\x90\x38\x46\x9f\xe0\x0c\x30\x92\x93\x6a\x8c\xff\x93\x69\x04\x5f\x90\x67\x33\xa9\xd2\xab\x36\x60\x18\x20\x69\xb1\x57\xca\x8f\x9b\x99\xa7\x1f\xc1\x53\xc6\x83\x01\xe9\x7a\x38\xfc\x3a\x87\xae\x2b\x6f\x03\x75\x4e\x6d\xa8\x2d\x0b\x07\x26\xe0\x70\x39\x79\xc9\x32\x02\x89\xfe\xef\xbc\xdd\xcd\x9d\x70\x6b\x71\xb5\x1e\x9a\x1b\x9d\xc1\x41\x2e\x6e\xd4\xb5\x66\x76",
        );
        test::<Sha512>(
            b"\xf3\x0c\x78\x3b\x4e\xae\xb4\x65\x76\x7f\xa1\xb9\x6d\x0a\xf5\x24\x35\xd8\x5f\xab\x91\x2b\x6a\xba\x10\xef\xa5\xb9\x46\xed\x01\xe1\x5d\x42\x7a\x4e\xcd\x0f\xf9\x55\x67\x73\x79\x17\x98\xb6\x69\x56\xec\xc7\x52\x88\xd1\xe9\xba\x2a\x9e\xa9\x48\x57\xd3\x13\x29\x99\xa2\x25\xb1\xff\xaf\x84\x46\x70\x15\x6e\x7a\x3e\xa9\xf0\x77\xfe\x82\x59\xa0\x98\xb9\xee\x75\x9a\x6d\xdf\xb7\xd2\x0a\x7a\xcd\x1b\xcb\x9f\x67\x77\x7e\x74\x61\x5e\x88\x59\xea\x56\x28\x1f\xe5\xc4\x00\x74\x8f\x02\xd1\xa2\x63\xb1\x86\x7a\x3b\x51\x74\x8a\xb7\x0f",
            n,
            e,
            b"\x34\x5e\x2f\x60\xf7\xc8\x2c\x89\xef\x7d\xfd\x7d\xff\x2b\xc2\x34\x8b\xab\x02\x04\x79\x33\x08\x99\xd4\x41\x02\x13\xb3\x5e\x98\xd9\xba\xc9\x2f\xd8\xae\x80\x6b\x5b\xce\x8a\x6c\x4b\xd8\x27\x5b\x0f\xac\xb4\xdd\x13\xf9\xd6\x8b\xa6\x71\x41\xfa\x50\x85\x26\x4d\xa6\xdd\x68\x5a\x6d\x21\x21\x70\xa2\xc9\xcb\xf2\xcf\x59\x30\x18\x0e\xff\xc2\x50\x86\x8c\x98\x4b\xf5\x0f\xf6\x9d\x60\x69\xea\x28\xf5\xbc\x1b\x63\x70\x5d\x07\x32\x41\x6f\xd8\x29\xa5\xf5\xd6\x21\x74\x62\xc2\x2a\x33\xfd\x46\x52\xf7\xc1\xd1\x98\x79\x46\x46\xc0\x84\x06\x02\x4e\x81\x63\xa7\xeb\xe3\x9c\xfb\x51\x4c\x54\x43\x89\x7b\x58\x94\xdd\x19\xa2\x13\xe0\x37\xf2\x7e\x0f\xfb\xd6\xc5\x44\x7a\x80\x5a\x54\xdf\xdf\x4f\x65\x81\x9d\x4e\x0f\xbe\xe2\x5e\x3d\xac\x47\xfb\x6b\x63\x6e\x8d\xe6\x19\x0a\xdc\xcb\xce\xe9\x37\xd0\x97\x7b\x35\xb9\x73\x60\x6b\x0c\xa3\x48\x75\x8b\x50\xcd\xbb\xa0\x28\xb7\x3d\x0e\xf0\x1c\x56\x01\x4c\x03\x1c\x59\x8f\xe8\xdb\x87\xd2\xca\x46\x44\x77\x0a\xaa\x04\x51\xc3\x76\xde\xd8\x2f\xf5\xc6\xb8\xe7\xd2\xed\x9d\x1c\x8a\x17\xc3\x12\x2c\x12\x82\x73\xc6\x0f\xd1\xb0\x08\x8d\xfb\xc9\xc9\x27\xf1\x62\xe4\x38\x79\x40\x59\x64\xcb\x11\xef\x78\x99\x12\x3f\xeb\x8f\x88\xdd\x27\x34\xdf\x98\xaa\x69\x6d\x93\x6a\x8d\xf0\x70\x00\xe8\x4a\xf9\x01\x01\xf7\x00\x6a\x9b\xd2\x54\x9f\xdd\x0a\xd3\xf9\xde\x09\x30\x12\xd3\x2d\x2a\xfa\xa8\x28\x01\x7e\xe9\xc6\x07\xcb\xf5\xb5\x4f\x22\x36\x66\xd4\xb5\xf3\xe2\x6e\x0d\xfe\xc0\x03\x96\x1b\x83\xd8\x3d\xe3\x9f\xf6\xa0\xe8\x1e\x18\x83\xc1\xdb\x4a\xaa\xf0\x82\xfe\xc5\xaa\x30\xa7\xe5\x78\x55\x3d\x89\x77\x4c\x67\x90\x77\x90\xc9\x6d\xc4\xf5\xbe\x4c\x8c",
        );
        test::<Sha512>(
            b"\x13\x2c\xf5\x0c\x66\xac\x4c\xc5\x43\x39\x75\x1a\x0e\xbb\x86\x5e\x1d\x3d\x32\x05\x62\xfc\x90\x5c\x4a\xbd\x1e\x78\xe4\x64\x06\x6c\x46\xc3\xa0\xc0\x2d\xb0\x37\x1e\xe3\x5a\x10\x4d\x66\xdd\xa8\x64\xc6\x13\x3e\x37\xcf\xad\x91\x16\xe8\x83\xeb\xb7\x3b\x29\x5e\x70\x16\xc3\x4e\xa9\x91\x1a\x30\x92\x72\xef\x90\x11\x4d\x8f\x59\xff\xf0\xa7\x51\x93\xfe\x5a\xe3\x1e\xd9\x91\x21\xf9\xc5\x92\x09\xbc\x4b\xd5\x07\xb1\xdc\x12\xbc\x89\xb7\x9f\xfe\x4d\x0d\xf9\x20\x97\x62\xa1\x73\x01\x36\x29\x0c\xde\xe5\x8e\xc8\x28\xcc\xc8\x8e\xba",
            n,
            e,
            b"\xb1\x25\x03\xb7\xb2\xf7\x83\x61\x88\x84\x17\x4b\xcb\x9b\xe1\x08\x77\x96\x04\x31\xed\x63\x63\xc8\x07\xe1\x2d\xb7\x1b\x8b\x6b\xd9\xd6\x40\x1d\x06\x4e\x25\x37\x40\x15\x8e\x8b\x90\x01\x52\xd3\x7f\xaf\x20\x33\x3a\x7d\x80\xb3\xd4\x7c\x7c\x7a\x3f\xa1\x20\x91\xce\x31\xcd\x8a\xae\x27\x2a\x4d\xa1\x5f\xe2\xcb\x5c\xfd\xea\x54\x11\x95\xa4\x69\xc9\x6b\xcf\x69\x5e\x0b\x52\x6d\xfa\x48\xa5\x90\x03\xc6\x76\x3a\xf8\x13\x63\x92\xc4\xb8\xd2\x4d\xb3\x14\x74\x6f\x42\xac\xa5\x50\xac\xc6\x5e\x07\x49\x13\xab\x82\x23\x2e\xb8\x59\x35\x09\x15\x8a\x8b\xa3\x4b\xc0\xf0\xe3\x12\x5a\x83\x4a\x3e\xd2\xd6\xa8\xcb\x1d\x08\x5f\x23\x4a\xe8\x68\xb8\x6a\xea\x8d\x6f\x82\xe1\x3a\x08\x84\x24\x85\x06\x6e\x48\xaa\xe4\x83\x78\x73\x15\x0f\x44\x47\x5e\x12\x60\x2b\x55\x2d\xcb\x34\xd1\xf9\xfd\xaa\xdb\xc6\xbf\xf5\x13\x4c\x6f\xc7\x62\x63\x88\x8b\xe6\x7e\xfe\x63\xee\x18\x40\xfa\x08\xc4\x99\x38\x85\x8a\x9d\x48\xb1\x05\x8d\x18\x97\x6b\xf2\xe3\xbf\xc6\x25\x55\x2f\x75\xb3\xea\x44\xeb\x91\xdd\x36\x68\x65\xf2\x40\xa0\xc3\x36\xa0\x11\x0e\x0f\xa0\x9d\x09\xcd\x94\xc7\x0c\xbc\x88\x95\xae\x3d\x44\xae\x3d\xff\x54\x5f\x0e\x8c\x8c\xc6\x62\xec\xd4\x0f\x90\x99\xa9\x52\x49\x43\x96\xc6\xb4\x23\xeb\xb4\x63\x40\x99\x69\x28\x1c\xdd\x54\xad\x87\xa3\x08\xe4\x87\xce\x19\x74\x5b\x30\xd5\xda\x76\xb9\x8d\x2a\xa9\xa0\x07\xa5\x57\x83\xb3\x03\x7e\x5b\x86\x62\x32\x28\x10\xbd\xd1\x1d\x86\xdc\x3f\x61\x45\x11\x49\x39\x1f\xb2\xf1\x4e\xd9\xc1\x7c\x75\x16\x23\xa4\x04\x2c\xe7\xed\xb8\x75\xee\x27\xbc\xd1\xf1\x9d\x6d\xc9\x28\x3a\xd0\x6d\x15\xe0\x97\xe2\xb0\xb1\x5a\x7e\xb7\x12\x8a\xdb\xca\x0a\xa6\xad\xcc",
        );
    }
}
