// Copyright (C) 2009 by Thomas Moulard, AIST, CNRS, INRIA.
//
// This file is part of the roboptim.
//
// roboptim is free software: you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// roboptim is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public License
// along with roboptim.  If not, see <http://www.gnu.org/licenses/>.

#ifndef ROBOPTIM_CORE_DIFFERENTIABLE_FUNCTION_HH
# define ROBOPTIM_CORE_DIFFERENTIABLE_FUNCTION_HH
# include <cstring>
# include <limits>
# include <utility>

# include <log4cxx/logger.h>

# include <roboptim/core/fwd.hh>

# include <roboptim/core/function.hh>
# include <roboptim/core/portability.hh>

# define ROBOPTIM_DIFFERENTIABLE_FUNCTION_FWD_TYPEDEFS(PARENT)	\
  ROBOPTIM_FUNCTION_FWD_TYPEDEFS (PARENT);			\
  typedef parent_t::gradient_t gradient_t;			\
  typedef parent_t::jacobian_t jacobian_t;			\
  struct e_n_d__w_i_t_h__s_e_m_i_c_o_l_o_n

# define ROBOPTIM_DIFFERENTIABLE_FUNCTION_FWD_TYPEDEFS_(PARENT)	\
  ROBOPTIM_FUNCTION_FWD_TYPEDEFS_ (PARENT);			\
  typedef typename parent_t::gradient_t gradient_t;		\
  typedef typename parent_t::jacobian_t jacobian_t;		\
  struct e_n_d__w_i_t_h__s_e_m_i_c_o_l_o_n

namespace roboptim
{
  /// \addtogroup roboptim_meta_function
  /// @{

  /// \brief Define an abstract derivable function (\f$C^1\f$).
  ///
  /// A derivable function which provides a way to compute its
  /// gradient/jacobian.
  ///
  /// \f[ f : x \rightarrow f(x) \f]
  /// \f$x \in \mathbb{R}^n\f$, \f$f(x) \in \mathbb{R}^m\f$ where
  /// \f$n\f$ is the input size and \f$m\f$ is the output size.
  ///
  /// Gradient computation is done through the #impl_gradient method
  /// that has to implemented by the concrete class inheriting this
  /// class.
  ///
  /// Jacobian computation is automatically done by concatenating
  /// gradients together, however this naive implementation can be
  /// overridden by the concrete class.
  ///
  ///
  /// The gradient of a \f$\mathbb{R}^n \rightarrow \mathbb{R}^m\f$
  /// function where \f$n > 1\f$ and \f$m > 1\f$ is a matrix.
  /// As this representation is costly, RobOptim considers
  /// these functions as \f$m\f$ \f$\mathbb{R}^n \rightarrow \mathbb{R}\f$
  /// functions. Through that mechanism, gradients are always vectors
  /// and jacobian are always matrices.
  /// When the gradient or the jacobian has to be computed, one has to
  /// precise which of the \f$m\f$ functions should be considered.
  ///
  /// If \f$m = 1\f$, then the function id must always be 0 and can be safely
  /// ignored in the gradient/jacobian computation.
  /// The class provides a default value for the function id so that
  /// these functions do not have to explicitly set the function id.
  template <typename T>
  class GenericDifferentiableFunction : public GenericFunction<T>
  {
  public:
    ROBOPTIM_FUNCTION_FWD_TYPEDEFS_ (GenericFunction<T>);

    /// \brief Gradient type.
    typedef typename GenericFunctionTraits<T>::gradient_t gradient_t;
    /// \brief Jacobian type.
    typedef typename GenericFunctionTraits<T>::jacobian_t jacobian_t;

    /// \brief Jacobian size type (pair of values).
    typedef std::pair<size_type, size_type> jacobianSize_t;


    /// \brief Return the gradient size.
    ///
    /// Gradient size is equals to the input size.
    size_type gradientSize () const throw ()
    {
      return this->inputSize ();
    }

    /// \brief Return the jacobian size as a pair.
    ///
    /// Gradient size is equals to (output size, input size).
    jacobianSize_t jacobianSize () const throw ()
    {
      return std::make_pair (this->outputSize (), this->inputSize ());
    }

    /// \brief Check if the gradient is valid (check size).
    /// \param gradient checked gradient
    /// \return true if valid, false if not
    bool isValidGradient (const gradient_t& gradient) const throw ()
    {
      return gradient.size () == gradientSize ();
    }

    /// \brief Check if the jacobian is valid (check sizes).
    ///
    /// \param jacobian checked jacobian
    /// \return true if valid, false if not
    bool isValidJacobian (const jacobian_t& jacobian) const throw ()
    {
      return jacobian.rows () == jacobianSize ().first
	&& jacobian.cols () == jacobianSize ().second;
    }

    /// \brief Computes the jacobian.
    ///
    /// \param argument point at which the jacobian will be computed
    /// \return jacobian matrix
    jacobian_t jacobian (const argument_t& argument) const throw ()
    {
      jacobian_t jacobian (jacobianSize ().first, jacobianSize ().second);
      jacobian.setZero ();
      this->jacobian (jacobian, argument);
      return jacobian;
    }

    /// \brief Computes the jacobian.
    ///
    /// Program will abort if the jacobian size is wrong before
    /// or after the jacobian computation.
    /// \param jacobian jacobian will be stored in this argument
    /// \param argument point at which the jacobian will be computed
    void jacobian (jacobian_t& jacobian, const argument_t& argument)
      const throw ()
    {
      LOG4CXX_TRACE (this->logger,
		     "Evaluating jacobian at point: " << argument);
      assert (argument.size () == this->inputSize ());
      assert (isValidJacobian (jacobian));
#ifndef ROBOPTIM_DO_NOT_CHECK_ALLOCATION
      Eigen::internal::set_is_malloc_allowed (false);
#endif //! ROBOPTIM_DO_NOT_CHECK_ALLOCATION
      this->impl_jacobian (jacobian, argument);
#ifndef ROBOPTIM_DO_NOT_CHECK_ALLOCATION
      Eigen::internal::set_is_malloc_allowed (true);
#endif //! ROBOPTIM_DO_NOT_CHECK_ALLOCATION
      assert (isValidJacobian (jacobian));
    }

    /// \brief Computes the gradient.
    ///
    /// \param argument point at which the gradient will be computed
    /// \param functionId function id in split representation
    /// \return gradient vector
    gradient_t gradient (const argument_t& argument,
			 size_type functionId = 0) const throw ()
    {
      gradient_t gradient (gradientSize ());
      gradient.setZero ();
      this->gradient (gradient, argument, functionId);
      return gradient;
    }

    /// \brief Computes the gradient.
    ///
    /// Program will abort if the gradient size is wrong before
    /// or after the gradient computation.
    /// \param gradient gradient will be stored in this argument
    /// \param argument point at which the gradient will be computed
    /// \param functionId function id in split representation
    /// \return gradient vector
    void gradient (gradient_t& gradient,
		   const argument_t& argument,
		   size_type functionId = 0) const throw ()
    {
      LOG4CXX_TRACE (this->logger,
		     "Evaluating gradient at point: "
		     << argument
		     << " (function id: " << functionId << ")");
      assert (argument.size () == this->inputSize ());
      assert (isValidGradient (gradient));
#ifndef ROBOPTIM_DO_NOT_CHECK_ALLOCATION
      Eigen::internal::set_is_malloc_allowed (false);
#endif //! ROBOPTIM_DO_NOT_CHECK_ALLOCATION
      this->impl_gradient (gradient, argument, functionId);
#ifndef ROBOPTIM_DO_NOT_CHECK_ALLOCATION
      Eigen::internal::set_is_malloc_allowed (true);
#endif //! ROBOPTIM_DO_NOT_CHECK_ALLOCATION
      assert (isValidGradient (gradient));
    }

    /// \brief Display the function on the specified output stream.
    ///
    /// \param o output stream used for display
    /// \return output stream
    virtual std::ostream& print (std::ostream& o) const throw ();

  protected:
    /// \brief Concrete class constructor should call this constructor.
    ///
    /// \param inputSize input size (argument size)
    /// \param outputSize output size (result size)
    /// \param name function's name
    GenericDifferentiableFunction (size_type inputSize,
				   size_type outputSize = 1,
				   std::string name = std::string ()) throw ();

    /// \brief Jacobian evaluation.
    ///
    /// Computes the jacobian, can be overridden by concrete classes.
    /// The default behavior is to compute the jacobian from the gradient.
    /// \warning Do not call this function directly, call #jacobian instead.
    /// \param jacobian jacobian will be store in this argument
    /// \param arg point where the jacobian will be computed
    virtual void impl_jacobian (jacobian_t& jacobian, const argument_t& arg)
      const throw ();

    /// \brief Gradient evaluation.
    ///
    /// Compute the gradient, has to be implemented in concrete classes.
    /// The gradient is computed for a specific sub-function which id
    /// is passed through the functionId argument.
    /// \warning Do not call this function directly, call #gradient instead.
    /// \param gradient gradient will be store in this argument
    /// \param argument point where the gradient will be computed
    /// \param functionId evaluated function id in the split representation
    virtual void impl_gradient (gradient_t& gradient,
				const argument_t& argument,
				size_type functionId = 0)
      const throw () = 0;
  };

  /// @}

} // end of namespace roboptim

# include <roboptim/core/differentiable-function.hxx>
#endif //! ROBOPTIM_CORE_DIFFERENTIABLE_FUNCTION_HH
