#ifndef _RHEOLEF_INDEX_SET_H
#define _RHEOLEF_INDEX_SET_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
// wrapper for std::set<size_t>: union as a+b and a += b and others goodies
//
// motivation: this class is usefull for disarray<index_set,M> to send/receive variable-sized
// lists via MPI correctly
// disarray<index_set,M> is used by the geo class to manage connectivity
// and compute neighbours
//
// author: Pierre.Saramito@imag.fr
//
// date: 19 mai 2012
//
#include "rheolef/distributed.h"
#include "rheolef/pretty_name.h"
#include "rheolef/container_traits.h"

#ifdef _RHEOLEF_HAVE_MPI
#include <boost/serialization/set.hpp>
#include <boost/serialization/base_object.hpp>
#endif // _RHEOLEF_HAVE_MPI

namespace rheolef {

/*Class:index_set
NAME:  index_set - a set of indexes (@PACKAGE@-@VERSION@)
SYNOPSYS:       
    A class for: l = @{1,3,...9@} i.e. a wrapper for STL @code{set<size_t>} with 
    some assignment operators, such as l1 += l2.
    This class is suitable for use with the @code{disarray<T>} class, as @code{disarray<index_set>}
    (@pxref{disarray class}).
AUTHOR: Pierre.Saramito@imag.fr
DATE: date: 23 march 2011
End:
*/
//<verbatim:
class index_set : public std::set<std::size_t> {
public:

// typedefs:

  typedef std::set<std::size_t>    base;
  typedef std::size_t              value_type;
  typedef std::size_t              size_type;

// allocators:

  index_set ();
  index_set (const index_set& x);
  index_set& operator= (const index_set& x);
  template <int N>
  index_set& operator= (size_type x[N]);
  void clear ();

// basic algebra:

  void       insert     (size_type dis_i);    // a := a union {dis_i}
  index_set& operator+= (size_type dis_i);    // idem
  index_set& operator+= (const index_set& b); // a := a union b

  // a := a union b
  void inplace_union        (const index_set& b);
  void inplace_intersection (const index_set& b);

  // c := a union b
  friend void set_union        (const index_set& a, const index_set& b, index_set& c);
  friend void set_intersection (const index_set& a, const index_set& b, index_set& c);

// io:

  friend std::istream& operator>> (std::istream& is, index_set& x);
  friend std::ostream& operator<< (std::ostream& os, const index_set& x);

// boost mpi:

#ifdef _RHEOLEF_HAVE_MPI
  template <class Archive>
  void serialize (Archive& ar, const unsigned int version);
#endif // _RHEOLEF_HAVE_MPI
};
//>verbatim:

// operator += for disarray::assembly
template <class T>
struct index_set_add_op : std::binary_function<T,T,T> {
    T& operator()(T& x, const T&                      y) const { return x += y; }
    T& operator()(T& x, const typename T::value_type& y) const { return x += y; }
};
// for boost mpi and disarray<index_set>:
template <> struct default_set_op<index_set> {
  typedef index_set_add_op<index_set> type;
};
template <> struct is_container<index_set> : std::true_type {
  typedef std::true_type type;
};
#ifdef _RHEOLEF_HAVE_MPI
template <> struct is_container_of_mpi_datatype<index_set> : std::true_type {
  typedef std::true_type type;
};
#endif // _RHEOLEF_HAVE_MPI
// -------------------------------------------------------------------
// inlined
// -------------------------------------------------------------------
inline
index_set::index_set()
 : base()
{
}
template<int N>
index_set&
index_set::operator= (size_t x[N])
{
  base::clear();
  for (size_t* iter = &(x[0]), *last = &(x[0])+N; iter != last; iter++) {
    base::insert (*iter);
  }
  return *this;
}
inline
void
index_set::clear ()
{
  base::clear();
}
inline
void
index_set::insert (size_type dis_i)
{
  base::insert (dis_i);
}
inline
index_set&
index_set::operator+= (size_type dis_i)
{
  base::insert (dis_i);
  return *this;
}
#ifdef _RHEOLEF_HAVE_MPI
template <class Archive>
void
index_set::serialize (Archive& ar, const unsigned int version)
{
  ar & boost::serialization::base_object<base>(*this);
}
#endif // _RHEOLEF_HAVE_MPI

} // namespace rheolef
#endif // _RHEOLEF_INDEX_SET_H
