///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// gnuplot geo visualisation
//
// author: Pierre.Saramito@imag.fr
//
// date: 16 sept 2011
//
# include "rheolef/field.h"
# include "rheolef/field_indirect.h"
# include "rheolef/field_expr.h"
# include "rheolef/field_expr_ops.h"
# include "rheolef/field_nonlinear_expr_ops.h"
# include "rheolef/interpolate.h"
# include "rheolef/field_evaluate.h"
# include "rheolef/piola.h"
# include "rheolef/rounder.h"
# include "rheolef/rheostream.h"

# include "field_seq_visu_gnuplot_internal.h"

namespace rheolef {

template <class T>
odiststream& visu_gnuplot (odiststream& ops, const geo_basic<T,sequential>& omega);

// ----------------------------------------------------------------------------
// puts for one element
// ----------------------------------------------------------------------------
template<class T>
static
void
put_edge (std::ostream& gdat, const geo_element& E, const geo_basic<T,sequential>& omega,
	const field_basic<T,sequential>& uh, 
	const basis_on_pointset<T>& geo_on_pointset,
	const basis_on_pointset<T>& field_on_pointset,
        size_t my_order,
	bound_type<T>& bbox)
{
  using namespace std;
  typedef typename geo_basic<T,sequential>::size_type size_type;
  typedef point_basic<size_type>                      ilat;
  size_type dim = omega.dimension();
  std::vector<size_type> dis_idof1;
  uh.get_space().dis_idof (E, dis_idof1);
  std::vector<size_type> dis_inod;
  omega.dis_inod (E, dis_inod);
  std::vector<point_basic<T> > hat_xi;
  uh.get_space().get_numbering().get_basis().hat_node (E.variant(), hat_xi);
  for (size_type i = 0; i <= my_order; i++) {
    size_type iloc = reference_element_e::ilat2loc_inod (my_order, ilat(i));
    point_basic<T> xi = piola_transformation (omega, geo_on_pointset,   E.variant(), dis_inod, iloc);
    T              ui = field_evaluate       (uh,    field_on_pointset, E.variant(), dis_idof1, iloc);
    xi.put (gdat, dim); gdat << " " << ui << endl;
    bbox.update (xi, ui);
  }
  gdat << endl << endl;
}
template<class T>
static
void
put_triangle (std::ostream& gdat, const geo_element& F, const geo_basic<T,sequential>& omega,
	const field_basic<T,sequential>& uh,
	const basis_on_pointset<T>& geo_on_pointset,
	const basis_on_pointset<T>& field_on_pointset,
        size_t my_order,
	bound_type<T>& bbox)
{
  using namespace std;
  typedef typename geo_basic<T,sequential>::size_type size_type;
  typedef point_basic<size_type>                      ilat;
  size_type dim = omega.dimension();
  std::vector<size_type> dis_idof1;
  std::vector<size_type> dis_inod;
  omega.dis_inod (F, dis_inod);
  uh.get_space().dis_idof (F, dis_idof1);
  size_type nloc = geo_on_pointset.size(F);
  std::vector<point_basic<T> > xdof (nloc);
  std::vector<T>               udof (nloc);
  for (size_type iloc = 0; iloc < nloc; iloc++) {
    xdof[iloc] = piola_transformation (omega, geo_on_pointset,   F.variant(), dis_inod, iloc);
    udof[iloc] = field_evaluate       (uh,    field_on_pointset, F.variant(), dis_idof1, iloc);
    bbox.update (xdof[iloc], udof[iloc]);
  }
  for (size_type j = 0; j <= my_order; j++) {
    for (size_type i1 = 0; i1 <= my_order; i1++) {
      size_type i = std::min(i1, my_order-j);
      size_type iloc = reference_element_t::ilat2loc_inod (my_order, ilat(i, j));
      xdof [iloc].put (gdat, dim); gdat << " " << udof[iloc] << endl;
    }
    gdat << endl;
  }
  gdat << endl << endl;
}
template<class T>
static
void
put_quadrangle (std::ostream& gdat, const geo_element& F, const geo_basic<T,sequential>& omega,
	const field_basic<T,sequential>& uh, const basis_on_pointset<T>& b)
{
  using namespace std;
  typedef typename geo_basic<T,sequential>::size_type size_type;
  typedef point_basic<size_type>                      ilat;
  size_type dim = omega.dimension();
  size_type degree = uh.get_space().get_numbering().get_basis().degree();
  std::vector<size_type> dis_idof1;
  uh.get_space().dis_idof (F, dis_idof1);
  std::vector<size_type> dis_inod;
  omega.dis_inod (F, dis_inod);
  std::vector<point_basic<T> > xdof (dis_idof1.size());
  for (size_type loc_idof = 0, loc_ndof = xdof.size(); loc_idof < loc_ndof; loc_idof++) {
    xdof[loc_idof] = piola_transformation (omega, b, F.variant(), dis_inod, loc_idof);
  }
  for (size_type j = 0; j < degree+1; j++) {
    for (size_type i = 0; i < degree+1; i++) {
      size_type loc_idof00 = reference_element_q::ilat2loc_inod (degree, ilat(i, j));
      xdof [loc_idof00].put (gdat, dim); gdat << " " << uh.dof(dis_idof1[loc_idof00]) << endl;
    }
    gdat << endl;
  }
  gdat << endl << endl;
}
template<class T>
void
put (std::ostream& gdat, const geo_element& K, const geo_basic<T,sequential>& omega,
	const field_basic<T,sequential>& uh,
	const basis_on_pointset<T>& geo_on_pointset,
	const basis_on_pointset<T>& field_on_pointset,
        size_t my_order,
	bound_type<T>& bbox)
{
  switch (K.variant()) {
   case reference_element::e: put_edge       (gdat, K, omega, uh, geo_on_pointset, field_on_pointset, my_order, bbox); break;
   case reference_element::t: put_triangle   (gdat, K, omega, uh, geo_on_pointset, field_on_pointset, my_order, bbox); break;
   case reference_element::q: put_quadrangle (gdat, K, omega, uh, geo_on_pointset); break;
   default: error_macro ("unsupported element variant `"<<K.variant()<<"'");
  }
}
// ----------------------------------------------------------------------------
// scalar field puts
// ----------------------------------------------------------------------------
template <class T>
odiststream&
visu_gnuplot_scalar (odiststream& ods, const field_basic<T,sequential>& uh)
{
  using namespace std;
  typedef typename field_basic<T,sequential>::float_type float_type;
  typedef typename geo_basic<float_type,sequential>::size_type size_type;
  typedef point_basic<size_type>                      ilat;
  ostream& os = ods.os();
  bool verbose  = iorheo::getverbose(os);
  bool clean    = iorheo::getclean(os);
  bool execute  = iorheo::getexecute(os);
  bool fill     = iorheo::getfill(os);    // show grid or fill elements
  bool elevation = iorheo::getelevation(os);
  bool color   = iorheo::getcolor(os);
  bool gray    = iorheo::getgray(os);
  bool black_and_white = iorheo::getblack_and_white(os);
  bool reader_on_stdin = iorheo::getreader_on_stdin(os);
  string format   = iorheo::getimage_format(os);
  string basename = iorheo::getbasename(os);
  size_type subdivide = iorheo::getsubdivide(os);
  size_type n_isovalue = iorheo::getn_isovalue(os);
  size_type n_isovalue_negative = iorheo::getn_isovalue_negative(os);
  string outfile_fmt = "";
  string tmp = get_tmpdir() + "/";
  if (!clean) tmp = "";

  const geo_basic<float_type,sequential>& omega = uh.get_geo();
  size_type dim     = omega.dimension();
  size_type map_dim = omega.map_dimension();
  size_type nv      = omega.sizes().ownership_by_dimension[0].size();
  size_type nedg    = omega.sizes().ownership_by_dimension[1].size();
  size_type nfac    = omega.sizes().ownership_by_dimension[2].size();
  size_type nvol    = omega.sizes().ownership_by_dimension[3].size();
  size_type ne      = omega.sizes().ownership_by_dimension[map_dim].size();

  const numbering<float_type,sequential>& fem = uh.get_space().get_numbering();
  if (subdivide == 0) { // subdivide is unset: use default
    subdivide = std::max(omega.order(), subdivide);
    subdivide = std::max(fem.degree (), subdivide);
  }
  basis_basic<T> subdivide_pointset ("P"+itos(subdivide));
  basis_on_pointset<T>   geo_on_pointset (subdivide_pointset, omega.get_piola_basis());
  basis_on_pointset<T> field_on_pointset (subdivide_pointset, fem.get_basis());

  bound_type<T> bbox;
  bbox.xmin = omega.xmin();
  bbox.xmax = omega.xmax();
  bbox.umin = uh.min();
  bbox.umax = uh.max();

  std::vector<T> values;
  if (n_isovalue_negative != 0) {
    for (size_t i = 0; i <= n_isovalue_negative; i++) {
      values.push_back (bbox.umin*(n_isovalue_negative - i)/n_isovalue_negative);
    }
    for (size_t i = 1; i <= n_isovalue - n_isovalue_negative; i++) {
      values.push_back (bbox.umax*i/(n_isovalue - n_isovalue_negative));
    }
  }
  string filelist;
  //
  // output .gdat
  //
  string filename = tmp+basename + ".gdat";
  string gdatname = filename;
  filelist = filelist + " " + filename;
  ofstream gdat (filename.c_str());
  if (verbose) clog << "! file \"" << filename << "\" created.\n";
  gdat << setprecision(numeric_limits<float_type>::digits10);
  size_type used_dim = (fill ? map_dim : 1);
  for (size_type ie = 0, ne = omega.size(used_dim); ie < ne; ie++) {
    const geo_element& K = omega.get_geo_element(used_dim,ie);
    put (gdat, K, omega, uh, geo_on_pointset, field_on_pointset, subdivide, bbox);
  }
  gdat.close();
  //
  // rounding bounds
  //
  T eps = 1e-7;
  bbox.umin = floorer(eps) (bbox.umin);
  bbox.umax = ceiler(eps)  (bbox.umax);
  //
  // output .plot
  //
  filename = tmp+basename + ".plot";
  filelist = filelist + " " + filename;
  ofstream plot (filename.c_str());
  if (verbose) clog << "! file \"" << filename << "\" created.\n";

  plot << "#!gnuplot" << endl
       << setprecision(numeric_limits<float_type>::digits10);
  if (format != "") {
    outfile_fmt = basename + "." + format;
    string terminal = format;
    if (terminal == "ps")  {
      terminal = "postscript eps";
      if (color) terminal += " color";
    }
    if (terminal == "jpg") terminal = "jpeg";
    if (terminal == "jpeg" || terminal == "png" || terminal == "gif") {
      terminal += " crop";
    }
    plot << "set terminal " << terminal    << endl
         << "set output \"" << outfile_fmt << "\"" << endl;
  }
  if (!black_and_white && map_dim == dim) {
    plot << "set noborder" << endl;
  }
  if (dim == 2) {
    plot << "umin = " << bbox.umin << endl
         << "umax = " << bbox.umax << endl;
    if (bbox.xmin[0] >= bbox.xmax[0]) plot << "#";
    plot << "set xrange [" << bbox.xmin[0] << ":" << bbox.xmax[0] << "]" << endl;
    if (bbox.xmin[1] >= bbox.xmax[1]) plot << "#";
    plot << "set yrange [" << bbox.xmin[1] << ":" << bbox.xmax[1] << "]" << endl;
    if (bbox.umin >= bbox.umax) plot << "#";
    plot << "set zrange [umin:umax]" << endl;
    if (bbox.xmin[0] >= bbox.xmax[0] || bbox.xmin[1] >= bbox.xmax[1]) {
      plot << "set size square" << endl;
    } else {
      plot << "set size ratio -1 # equal scales" << endl
           << "set noxtics" << endl
           << "set noytics" << endl;
    }
    if (elevation) {
      plot << "set xyplane at umin-0.1*(umax-umin)" << endl;
    } else {
      plot << "set view map" << endl;
    }
    if (color || gray) {
      plot << "set pm3d interpolate 10,10 corners2color mean" << endl;
      if (gray) {
        plot << "set palette gray" << endl;
      } else {
        plot << "set palette rgbformulae 33,13,-4" << endl;
      }
    } else { // black-and-white
      if (values.size() != 0) {
        plot << "set cntrparam levels discrete ";
        for (size_t i = 0, n = values.size(); i < n; i++) {
          plot << values[i];
	  if (i != n-1) plot << ", ";
        }
        plot << endl;
      } else {
        plot << "set cntrparam levels incremental umin,(umax-umin)/10.0,umax"<< endl;
      }
      plot << "unset surface" << endl
           << "set contour base" << endl
           << "set nokey" << endl
           << "set palette rgbformulae 0,0,0" << endl
           << "unset colorbox" << endl;
    }
  } else if (dim == 3 && map_dim == 2) {
    // field defined on a 3D surface
    point_basic<T> dx = 0.1*(omega.xmax() - omega.xmin());
    T dx_max = max(dx[0],max(dx[1],dx[2]));
    if (dx_max == 0) dx_max = 0.1;
    dx[0] = max(dx[0],dx_max);
    if (omega.dimension() >= 2) dx[1] = max(dx[1],dx_max);
    if (omega.dimension() == 3) dx[2] = max(dx[2],dx_max);
    point_basic<T> xmin = omega.xmin() - dx;
    point_basic<T> xmax = omega.xmax() + dx;
    plot << "set xrange [" << xmin[0] << ":" << xmax[0] << "]" << endl
         << "set yrange [" << xmin[1] << ":" << xmax[1] << "]" << endl
         << "set zrange [" << xmin[2] << ":" << xmax[2] << "]" << endl
         << "set xyplane at " << xmin[2] << endl
         << "set view equal xyz # equal scales" << endl
         << "set view 70,120" << endl
         << "set pm3d interpolate 5,5 corners2color mean" << endl
         << "set palette rgbformulae 33,13,-4" << endl;
    if (format != "") {
      plot << "set noxlabel" << endl
           << "set noylabel" << endl
           << "set nozlabel" << endl;
    } else {
      plot << "set xlabel \"x\"" << endl
           << "set ylabel \"y\"" << endl
           << "set zlabel \"z\"" << endl;
    }
  }
  if (dim == 1) {
    plot << "plot \"" << gdatname << "\" notitle with lines";
    if (black_and_white) {
      plot << " lc 0" << endl;
    }
    plot << endl;
  } else {
    if (!fill && dim == 2) {
      plot << "plot";
    } else {
      plot << "splot";
    } 
    plot << " \"" << gdatname << "\" notitle";
    if (map_dim == 2) {
      if (!black_and_white && fill) {
        plot << " with pm3d" << endl;
      } else {
        plot << " with lines palette" << endl;
      }
    } else { // a 2d line
      plot << " with lines palette lw 2" << endl;
    }
  }
  //
  // end of plot
  //
  if (format == "" && !reader_on_stdin) {
    plot << "pause -1 \"<return>\"\n";
  }
  plot.close();
  //
  // run gnuplot
  //
  int status = 0;
  string command;
  if (execute) {
      command = "gnuplot ";
      if (reader_on_stdin) command += "-persist ";
      command += tmp + basename + ".plot";
      if (verbose) clog << "! " << command << endl;
      cin.sync();
      status = system (command.c_str());
      if (format != "") {
        check_macro (file_exists (outfile_fmt), "! file \"" << outfile_fmt << "\" creation failed");
        if (verbose) clog << "! file \"" << outfile_fmt << "\" created" << endl;
      }
  }
  //
  // clear gnuplot data
  //
  if (clean) {
      command = "/bin/rm -f " + filelist;
      if (verbose) clog << "! " << command << endl;
      status = system (command.c_str());
  }
  return ods;
}
// ----------------------------------------------------------------------------
// vector field puts
// ----------------------------------------------------------------------------
template <class T>
odiststream&
visu_gnuplot_vector (odiststream& ods, const field_basic<T,sequential>& uh)
{
  typedef typename geo_basic<T,sequential>::size_type size_type;
  const geo_basic<T,sequential>& omega = uh.get_geo();
  check_macro (omega.get_piola_basis().name() == uh.get_space().get_numbering().name(),
	"gnuplot vector: unsupported non-isoparametric approx " << uh.get_space().get_numbering().name());
  size_type n_elt = omega.size();
  size_type d     = omega.dimension();
  T diam_omega = norm (omega.xmax() - omega.xmin()); // characteristic length in omega
  T h_moy = diam_omega/pow(n_elt,1./d);
  space_basic<T,sequential> Xh (uh.get_geo(), uh.get_approx());
  field_basic<T,sequential> norm_uh = interpolate(Xh, norm(uh));
  T norm_max_uh = norm_uh.max_abs();      // get max vector length
  if (norm_max_uh + 1 == 1) norm_max_uh = 1;
  T scale = h_moy/norm_max_uh;
  size_type n_comp = uh.size();
  std::vector<field_component_const<T,sequential> > uh_comp (n_comp);
  for (size_type i_comp = 0; i_comp < n_comp; i_comp++) {
    uh_comp[i_comp] = uh[i_comp];
  }
  array<point_basic<T>, sequential> x = omega.get_nodes();
  for (size_type inod = 0, nnod = x.size(); inod < nnod; inod++) {
    point_basic<T> vi;
    for (size_type i_comp = 0; i_comp < n_comp; i_comp++) {
      vi[i_comp] = uh[i_comp].dof(inod); // TODO: non-isoparam => interpolate
    }
    x[inod] += vi;
  }
  geo_basic<T,sequential> deformed_omega = omega;
  deformed_omega.set_nodes(x);
  space_basic<T,sequential> deformed_Vh (deformed_omega, norm_uh.get_space().get_numbering().name());
  field_basic<T,sequential> deformed_norm_uh (deformed_Vh);
  std::copy (norm_uh.begin_dof(), norm_uh.end_dof(), deformed_norm_uh.begin_dof());
  visu_gnuplot (ods, deformed_norm_uh);
  return ods;
}
// ----------------------------------------------------------------------------
// switch
// ----------------------------------------------------------------------------
template <class T>
odiststream&
visu_gnuplot (odiststream& ods, const field_basic<T,sequential>& uh)
{
  switch (uh.get_space().valued_tag()) {
    case space_constant::scalar: visu_gnuplot_scalar (ods, uh); break;
    case space_constant::vector: visu_gnuplot_vector (ods, uh); break;
    default: error_macro ("do not known how to print " << uh.valued() << "-valued field");
  }
  return ods;
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
#define _RHEOLEF_instanciate(T) \
template odiststream& visu_gnuplot<T> (odiststream&, const field_basic<T,sequential>&);

_RHEOLEF_instanciate(Float)
#undef _RHEOLEF_instanciate
} // rheolef namespace
