# ifndef _SKIT_EYE_H
# define _SKIT_EYE_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
/*Class:eye
NAME: @code{eye} - identity matrix 
DESCRIPTION:       
 Following matlab, the name EYE is used in place of I to denote identity matrices
 because I is often used as a subscript or as sqrt(-1). The dimensions of EYE are
 determined by context.  For example,
 @example
      	    b = a + 3*EYE
 @end example
 adds 3 to the diagonal elements of A.
 Small caps 'eye' is the class name, and upper case 'EYE' is the const variable
 "eye<double>()", i.e. the identity matrix.

 The preconditioner interface is usefull when calling algorithms without any
 preconditioners, e.g.
 @example
            int status = cg (a, x, b, EYE, 100, 1e-7);
 @end example

AUTHOR: Pierre.Saramito@imag.fr
DATE: 28 january 1997
End:
*/
// ================ [ class implementation ] ====================================
//<eye:
# include "rheolef/skitbase.h"
namespace rheolef { 

template<class T>
class eye {
  public:
    eye (const T& mu = 1);
    eye operator * (const T& mu) const { return eye(lambda*mu); }
    friend eye operator * (const T& mu, const eye<T>& E) { return eye(mu*E.lambda); }
    vec<T> operator * (const vec<T>& x) const;
    vec<T> solve (const vec<T>& x) const;
    vec<T> trans_solve (const vec<T>& x) const { return solve(x); }
  private:
    T lambda;
};
#define EYE eye<Float>()
//>eye:

// Remark: the shared libraries are sometime buggy
// and cannot initialize class constructor. So, do not define
// extern const eye<Float> EYE;

template<class T>
inline
eye<T>::eye(const T& mu)
 : lambda(mu)
{
}
template<class T>
inline
vec<T> 
eye<T>::solve(const vec<T>& x) const
{
    if (lambda == T(1)) return x; else return Float(1./lambda)*x;
}
template<class T>
inline
vec<T>
eye<T>::operator * (const vec<T>& x) const
{
   if (lambda == 1) return x;
   return lambda*x;
}
}// namespace rheolef
# endif // _SKIT_EYE_H
