\name{binPeaks}
\alias{binPeaks}
\title{
Align Peaks into discrete bins.
}
\description{
This function looks for similar peaks (mass) across
\code{\linkS4class{MassPeaks}} objects and equalizes their mass.
}
\usage{
binPeaks(l, method=c("strict", "relaxed", "reference"), tolerance=0.002)
}
\arguments{
  \item{l}{
    \code{list}, list of \code{\linkS4class{MassPeaks}} objects.
  }
  \item{method}{
    bin creation rule. \code{"strict"} creates bins never containing two
    or more peaks of the same sample. \code{"relaxed"} allows multiple
    peaks of the same sample in one bin. \code{"reference"} generates bins
    around the mass values from the first \code{\linkS4class{MassPeaks}} object
    in \code{l}.}
  \item{tolerance}{
    \code{double}, maximal relative deviation of a peak position (mass) to be
    considered as identical. Must be multiplied by \code{10^-6} for ppm,
    e.g. use \code{tolerance=5e-6} for 5 ppm.
  }
}
\details{
The algorithm is based on the following workflow:
\enumerate{
  \item{Put all mass in a sorted vector.}
  \item{Calculate differences between each neighbor.}
  \item{Divide the mass vector at the largest gap (largest difference) and form
    a left and a right bin.}
  \item{Rerun step 3 for the left and/or the right bin if they don't fulfill
    the following criteria:
    \itemize{
      \item{All peaks in a bin are near to the mean (\code{method == "strict"}
        or \code{method == "relaxed"})
        (\code{abs(mass-meanMass)/meanMass < tolerance}) or the
        reference mass (\code{method == "reference"};
        \code{abs(mass-reference)/reference < tolerance}).}
      \item{\code{method == "strict"}: The bin doesn't contain two or more
        peaks of the same sample.}
    }
  }
}
\code{method == "strict"}: The new peak positions (mass value) are the
mean mass of a bin. \cr
\code{method == "relaxed"}: The new peak positions for the highest peaks of each
sample in a bin are generated by the mean mass of this peaks. The lower
peaks are not changed.
\code{method == "reference"}: The new peak positions for the highest peaks of
each sample in a bin are generated by the mass of peaks of the first
\code{\linkS4class{MassPeaks}} object. Lower peaks are not changed.
}
\value{
Returns a \code{\link{list}} of mass adjusted
\code{\linkS4class{MassPeaks}} objects.
}
\author{
Sebastian Gibb \email{mail@sebastiangibb.de}
}
\seealso{
\code{\link[MALDIquant]{intensityMatrix}},
\code{\linkS4class{MassPeaks}}

Website: \url{https://strimmerlab.github.io/software/maldiquant/}
}
\examples{
## load package
library("MALDIquant")

## create two MassPeaks objects
p <- list(createMassPeaks(mass=seq(100, 500, 100), intensity=1:5),
          createMassPeaks(mass=c(seq(100.2, 300.2, 100), 395), intensity=1:4))

binnedPeaks <- binPeaks(p, tolerance=0.002)

## compare result
iM1 <- intensityMatrix(p)
iM2 <- intensityMatrix(binnedPeaks)

all(dim(iM1) == c(2, 9)) # TRUE
all(dim(iM2) == c(2, 6)) # TRUE

show(iM2)

## increase tolerance
binnedPeaks <- binPeaks(p, tolerance=0.1)

iM3 <- intensityMatrix(binnedPeaks)

all(dim(iM3) == c(2, 5)) # TRUE

show(iM3)

## differences between "strict" and "relaxed"
p <- c(createMassPeaks(mass=c(1, 1.01, 3), intensity=c(2, 1, 1)),
       createMassPeaks(mass=c(0.99, 3), intensity=rep(1, 2)),
       createMassPeaks(mass=c(1.02, 3), intensity=rep(1, 2)))

intensityMatrix(binPeaks(p, method="strict", tolerance=0.05))
intensityMatrix(binPeaks(p, method="relaxed", tolerance=0.05))

## use a reference
ref <- createMassPeaks(mass=c(1, 3), intensity=rep(1, 2))
## include the reference
intensityMatrix(binPeaks(c(ref, p), method="reference", tolerance=0.05))
## drop the reference
intensityMatrix(binPeaks(c(ref, p), method="reference", tolerance=0.05)[-1])
}
\keyword{methods}

