% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/decompose_tidiers.R
\name{decompose_tidiers}
\alias{decompose_tidiers}
\alias{augment.decomposed.ts}
\alias{augment.stl}
\title{Tidying methods for seasonal decompositions}
\usage{
\method{augment}{decomposed.ts}(x, ...)

\method{augment}{stl}(x, weights = TRUE, ...)
}
\arguments{
\item{x}{An object of class \code{"stl"} or \code{"decomposed.ts"},
resulting from a call to \code{\link[stats]{decompose}} or
\code{\link[stats]{stl}}.}

\item{...}{Extra arguments. Unused.}

\item{weights}{Whether to include the robust weights in the output.}
}
\value{
The \code{augment} method returns a tidy data frame with the following
columns:

\describe{
  \item{\code{.seasonal}}{The seasonal component of the decomposition.}
  \item{\code{.trend}}{The trend component of the decomposition.}
  \item{\code{.remainder}}{The remainder, or "random" component of the
  decomposition.}
  \item{\code{.weight}}{The final robust weights (\code{stl} only).}
  \item{\code{.seasadj}}{The seasonally adjusted (or "deseasonalised")
  series.}
}
}
\description{
These tidiers provide an \code{augment} method for the results of a seasonal
decomposition with \code{\link[stats]{decompose}} or
\code{\link[stats]{stl}}.
}
\details{
The \code{augment} method returns the computed seasonal and trend components,
as well as the "remainder" term and the seasonally adjusted (or
"deseasonalised") series.
}
\examples{

# Time series of temperatures in Nottingham, 1920-1939:
nottem

# Perform seasonal decomposition on the data with both decompose
# and stl:
d1 <- stats::decompose(nottem)
d2 <- stats::stl(nottem, s.window = "periodic", robust = TRUE)

# Compare the original series to its decompositions.

cbind(broom::tidy(nottem), broom::augment(d1),
      broom::augment(d2))

# Visually compare seasonal decompositions in tidy data frames.

library(tibble)
library(dplyr)
library(tidyr)
library(ggplot2)

decomps <- tibble(
    # Turn the ts objects into data frames.
    series = list(broom::tidy(nottem), broom::tidy(nottem)),
    # Add the models in, one for each row.
    decomp = c("decompose", "stl"),
    model = list(d1, d2)
) \%>\%
    rowwise() \%>\%
    # Pull out the fitted data using broom::augment.
    mutate(augment = list(broom::augment(model))) \%>\%
    ungroup() \%>\%
    # Unnest the data frames into a tidy arrangement of
    # the series next to its seasonal decomposition, grouped
    # by the method (stl or decompose).
    group_by(decomp) \%>\%
    unnest(series, augment) \%>\%
    mutate(index = 1:n()) \%>\%
    ungroup() \%>\%
    select(decomp, index, x, adjusted = .seasadj)

ggplot(decomps) +
    geom_line(aes(x = index, y = x), colour = "black") +
    geom_line(aes(x = index, y = adjusted, colour = decomp,
                  group = decomp))

}
\seealso{
\code{\link[stats]{decompose}}, \code{\link[stats]{stl}}
}
\author{
Aaron Jacobs
}
