/*
 * Jalview - A Sequence Alignment Editor and Viewer (2.11.5.0)
 * Copyright (C) 2025 The Jalview Authors
 * 
 * This file is part of Jalview.
 * 
 * Jalview is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * as published by the Free Software Foundation, either version 3
 * of the License, or (at your option) any later version.
 *  
 * Jalview is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranty 
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 * PURPOSE.  See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with Jalview.  If not, see <http://www.gnu.org/licenses/>.
 * The Jalview Authors are detailed in the 'AUTHORS' file.
 */
/**
 * author: Lauren Michelle Lui
 */

package jalview.util;

import java.awt.Color;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Random;

import org.jcolorbrewer.ColorBrewer;

import jalview.bin.Console;

public class ColorUtils
{
  private static final int MAX_CACHE_SIZE = 1729;

  /*
   * a cache for colours generated from text strings
   */
  private static Map<String, Color> myColours = new HashMap<>();

  private static Map<String, Color> myHSBSpacedColours = new HashMap<>();

  private static final double GOLDEN_RATIO_CONJUGATE = 0.6180339887;

  private static final float DEFAULT_HSB_DISTANCE_THRESHOLD = 0.5f; // Minimum HSB
                                                               // distance
                                                               // required

  private static float HSB_DISTANCE_THRESHOLD = DEFAULT_HSB_DISTANCE_THRESHOLD;

  private static final float DECREMENT_VAL_FOR_HSB_DISTANCE_THRESHOLD = 0.01f; // Value
                                                                               // decremented
                                                                               // from
                                                                               // HSB
                                                                               // distance
                                                                               // threshold
                                                                               // after
                                                                               // reaching
                                                                               // max
                                                                               // iteration
                                                                               // count

  private static final int MAX_ITERATION_FOR_ADJUSTING_COLOR = 100; // Max
                                                                    // iteration
                                                                    // for
                                                                    // adjusting
                                                                    // the
                                                                    // colour
                                                                    // for a
                                                                    // particular
                                                                    // threshold

  private static final int POS_HUE = 0;

  private static final int POS_SATURATION = 1;

  private static final int POS_BRIGHTNESS = 2;
  
  private static final float HSB_HIGH_SATURATION = 0.75f;
  private static final float HSB_HIGH_BRIGHTNESS = 0.9f;


  /**
   * Generates a random color, will mix with input color. Code taken from
   * http://stackoverflow
   * .com/questions/43044/algorithm-to-randomly-generate-an-aesthetically
   * -pleasing-color-palette
   * 
   * @param mix
   * @return Random color in RGB
   */
  public static final Color generateRandomColor(Color mix)
  {
    Random random = new Random();
    int red = random.nextInt(256);
    int green = random.nextInt(256);
    int blue = random.nextInt(256);

    // mix the color
    if (mix != null)
    {
      red = (red + mix.getRed()) / 2;
      green = (green + mix.getGreen()) / 2;
      blue = (blue + mix.getBlue()) / 2;
    }

    Color color = new Color(red, green, blue);
    return color;
  }

  /**
   * 
   * @return random color
   */
  public static final Color getARandomColor()
  {
    Color col = new Color((int) (Math.random() * 255),
            (int) (Math.random() * 255), (int) (Math.random() * 255));
    return col;
  }
  
  /**
   * Generate a particular colour from the HSB colour space for a particular
   * index.
   * 
   * @param index
   * @return colour generated for the index
   */
  public static Color getColorForIndex(int index)
  {

    float hue = (index * (float) GOLDEN_RATIO_CONJUGATE) % 1.0f;
    return Color.getHSBColor(hue, HSB_HIGH_SATURATION, HSB_HIGH_BRIGHTNESS);

  }

  /**
   * Convert to Tk colour code format
   * 
   * @param colour
   * @return
   * @see http
   *      ://www.cgl.ucsf.edu/chimera/current/docs/UsersGuide/colortool.html#
   *      tkcode
   */
  public static final String toTkCode(Color colour)
  {
    String colstring = "#" + ((colour.getRed() < 16) ? "0" : "")
            + Integer.toHexString(colour.getRed())
            + ((colour.getGreen() < 16) ? "0" : "")
            + Integer.toHexString(colour.getGreen())
            + ((colour.getBlue() < 16) ? "0" : "")
            + Integer.toHexString(colour.getBlue());
    return colstring;
  }

  /**
   * Returns a colour three shades darker. Note you can't guarantee that
   * brighterThan reverses this, as darkerThan may result in black.
   * 
   * @param col
   * @return
   */
  public static Color darkerThan(Color col)
  {
    return col == null ? null : col.darker().darker().darker();
  }

  /**
   * Returns a colour three shades brighter. Note you can't guarantee that
   * darkerThan reverses this, as brighterThan may result in white.
   * 
   * @param col
   * @return
   */
  public static Color brighterThan(Color col)
  {
    return col == null ? null : col.brighter().brighter().brighter();
  }

  /**
   * Returns a color between minColour and maxColour; the RGB values are in
   * proportion to where 'value' lies between minValue and maxValue
   * 
   * @param value
   * @param minValue
   * @param minColour
   * @param maxValue
   * @param maxColour
   * @return
   */
  public static Color getGraduatedColour(float value, float minValue,
          Color minColour, float maxValue, Color maxColour)
  {
    if (minValue == maxValue)
    {
      return minColour;
    }
    if (value < minValue)
    {
      value = minValue;
    }
    if (value > maxValue)
    {
      value = maxValue;
    }

    /*
     * prop = proportion of the way value is from minValue to maxValue
     */
    float prop = (value - minValue) / (maxValue - minValue);
    float r = minColour.getRed()
            + prop * (maxColour.getRed() - minColour.getRed());
    float g = minColour.getGreen()
            + prop * (maxColour.getGreen() - minColour.getGreen());
    float b = minColour.getBlue()
            + prop * (maxColour.getBlue() - minColour.getBlue());
    return new Color(r / 255, g / 255, b / 255);
  }

  /**
   * 'Fades' the given colour towards white by the specified proportion. A
   * factor of 1 or more results in White, a factor of 0 leaves the colour
   * unchanged, and a factor between 0 and 1 results in a proportionate change
   * of RGB values towards (255, 255, 255).
   * <p>
   * A negative bleachFactor can be specified to darken the colour towards Black
   * (0, 0, 0).
   * 
   * @param colour
   * @param bleachFactor
   * @return
   */
  public static Color bleachColour(Color colour, float bleachFactor)
  {
    if (bleachFactor >= 1f)
    {
      return Color.WHITE;
    }
    if (bleachFactor <= -1f)
    {
      return Color.BLACK;
    }
    if (bleachFactor == 0f)
    {
      return colour;
    }

    int red = colour.getRed();
    int green = colour.getGreen();
    int blue = colour.getBlue();

    if (bleachFactor > 0)
    {
      red += (255 - red) * bleachFactor;
      green += (255 - green) * bleachFactor;
      blue += (255 - blue) * bleachFactor;
      return new Color(red, green, blue);
    }
    else
    {
      float factor = 1 + bleachFactor;
      red *= factor;
      green *= factor;
      blue *= factor;
      return new Color(red, green, blue);
    }
  }

  /**
   * Parses a string into a Color, where the accepted formats are
   * <ul>
   * <li>an AWT colour name e.g. white</li>
   * <li>a hex colour value (without prefix) e.g. ff0000</li>
   * <li>an rgb triple e.g. 100,50,150</li>
   * </ul>
   * 
   * @param colour
   * @return the parsed colour, or null if parsing fails
   */
  public static Color parseColourString(String colour)
  {
    if (colour == null)
    {
      return null;
    }
    colour = colour.trim();

    Color col = null;

    if ("random".equals(colour))
    {
      return generateRandomColor(null);
    }

    try
    {
      int value = Integer.parseInt(colour, 16);
      col = new Color(value);
    } catch (NumberFormatException ex)
    {
    }

    if (col == null)
    {
      col = ColorUtils.getAWTColorFromName(colour);
    }

    if (col == null)
    {
      try
      {
        String[] tokens = colour.split(",");
        if (tokens.length == 3)
        {
          int r = Integer.parseInt(tokens[0].trim());
          int g = Integer.parseInt(tokens[1].trim());
          int b = Integer.parseInt(tokens[2].trim());
          col = new Color(r, g, b);
        }
      } catch (IllegalArgumentException ex) // IllegalArgumentException includes
                                            // NumberFormatException
      {
        // non-numeric token or out of 0-255 range
      }
    }

    return col;
  }

  /**
   * Constructs a colour from a text string. The hashcode of the whole string is
   * scaled to the range 0-135. This is added to RGB values made from the
   * hashcode of each third of the string, and scaled to the range 20-229.
   * 
   * @param name
   * @return
   */
  public static Color createColourFromName(String name)
  {
    if (name == null)
    {
      return Color.white;
    }
    if (myColours.containsKey(name))
    {
      return myColours.get(name);
    }
    int lsize = name.length();
    int start = 0;
    int end = lsize / 3;

    int rgbOffset = Math.abs(name.hashCode() % 10) * 15; // 0-135

    /*
     * red: first third
     */
    int r = Math.abs(name.substring(start, end).hashCode() + rgbOffset)
            % 210 + 20;
    start = end;
    end += lsize / 3;
    if (end > lsize)
    {
      end = lsize;
    }

    /*
     * green: second third
     */
    int g = Math.abs(name.substring(start, end).hashCode() + rgbOffset)
            % 210 + 20;

    /*
     * blue: third third
     */
    int b = Math.abs(name.substring(end).hashCode() + rgbOffset) % 210 + 20;

    Color color = new Color(r, g, b);

    if (myColours.size() < MAX_CACHE_SIZE)
    {
      myColours.put(name, color);
    }

    return color;
  }

  /**
   * Returns the Color constant for a given colour name e.g. "pink", or null if
   * the name is not recognised
   * 
   * @param name
   * @return
   */
  public static Color getAWTColorFromName(String name)
  {
    if (name == null)
    {
      return null;
    }
    Color col = null;
    name = name.toLowerCase(Locale.ROOT);

    // or make a static map; or use reflection on the field name
    switch (name)
    {
    case "black":
      col = Color.black;
      break;
    case "blue":
      col = Color.blue;
      break;
    case "cyan":
      col = Color.cyan;
      break;
    case "darkgray":
      col = Color.darkGray;
      break;
    case "gray":
      col = Color.gray;
      break;
    case "green":
      col = Color.green;
      break;
    case "lightgray":
      col = Color.lightGray;
      break;
    case "magenta":
      col = Color.magenta;
      break;
    case "orange":
      col = Color.orange;
      break;
    case "pink":
      col = Color.pink;
      break;
    case "red":
      col = Color.red;
      break;
    case "white":
      col = Color.white;
      break;
    case "yellow":
      col = Color.yellow;
      break;
    }

    return col;
  }

  public enum ColourScheme
  {
    NONE, AVOID_RED, AVOID_GREEN, AVOID_BLUE, SATURATED, MEDIUM_SATURATION, DESATURATED,
    GREYISH, GREYSCALE, BRIGHT, MEDIUM, DARK
  }

  public static float[] getHSBRanges(String colourScheme)
  {
    float Hmin = 0.0f;
    float Hmax = 1.0f;
    float Smin = 0.6f;
    float Smax = 1.0f;
    float Bmin = 0.6f;
    float Bmax = 1.0f;
    if (!colourScheme.contains(ColourScheme.NONE.name()))
    {
      for (String scheme : colourScheme.split(","))
      {
        Console.debug("Applying colourScheme component " + scheme);
        ColourScheme cs;
        try
        {
          cs = ColourScheme.valueOf(scheme);
        } catch (IllegalArgumentException | NullPointerException e)
        {
          Console.warn("Did not recognise something in the colour scheme '"
                  + colourScheme + "'");
          return new float[] { Hmin, Hmax, Smin, Smax, Bmin, Bmax };
        }
        switch (cs)
        {
        case AVOID_RED:
          Hmin = 0.15f;
          Hmax = 0.85f;
          break;
        case AVOID_GREEN:
          Hmin = 0.48f;
          Hmax = 0.18f;
          break;
        case AVOID_BLUE:
          Hmin = 0.81f;
          Hmax = 0.51f;
          break;
        case SATURATED:
          Smin = 1.0f;
          Smax = 1.0f;
          break;
        case DESATURATED:
          Smin = 0.2f;
          Smax = 0.6f;
          break;
        case GREYISH:
          Smin = 0.0f;
          Smax = 0.2f;
        case GREYSCALE:
          Smin = 0.0f;
          Smax = 0.0f;
          Bmin = 0.1f;
          Bmax = 0.9f;
          break;
        case BRIGHT:
          Bmin = 1.0f;
          Bmax = 1.0f;
          break;
        case MEDIUM:
          Bmin = 0.6f;
          Bmax = 0.8f;
        case DARK:
          Bmin = 0.1f;
          Bmax = 0.4f;
          break;
        case NONE:
          break;
        }
      }
    }
    return new float[] { Hmin, Hmax, Smin, Smax, Bmin, Bmax };
  }


  private static String getHSBColourSpacedCacheKey(String name,
          String colourScheme)
  {
    return name.hashCode() + "::" + colourScheme;
  }
  
  /**
   * This method restores the colour map from the loaded file.
   * It replaces the current colour for the labels with colour
   * present in the loaded file.
   * @param colorMap
   */
  public static void restoreMyHSBSpacedColours(
          Map<String, Color> colorMap)
  {
    for(Entry<String, Color> entry : colorMap.entrySet()) {
      String cacheKey = getHSBColourSpacedCacheKey(entry.getKey(), "NONE");
      ColorUtils.myHSBSpacedColours.put(cacheKey, entry.getValue());
    }
  }


  public static Color getColourFromNameAndScheme(String name,
          String colourScheme)
  {
    String cacheKey = getHSBColourSpacedCacheKey(name, colourScheme);
    if (myHSBSpacedColours.containsKey(cacheKey))
    {
      return myHSBSpacedColours.get(cacheKey);
    }
    float[] vals = getHSBRanges(colourScheme);
    Color col = null;
    if (vals.length > 5)
    {
      col = getHSBColourspacedColourFromName(name, vals[0], vals[1],
              vals[2], vals[3], vals[4], vals[5]);

      // col = getHSBColourFromName(name);

      int iterationCounter = 0; // reset counter
      HSB_DISTANCE_THRESHOLD = DEFAULT_HSB_DISTANCE_THRESHOLD; // reset
                                                               // threshold

      // Adjust colour until it is distinct
      while (!isColorDistinct(col))
      {
        col = adjustColor(col);       

        iterationCounter++;

        // If too many iterations occur, reduce the threshold to allow closer
        // colours
        if (iterationCounter > MAX_ITERATION_FOR_ADJUSTING_COLOR)
        {
          HSB_DISTANCE_THRESHOLD = HSB_DISTANCE_THRESHOLD
                  - DECREMENT_VAL_FOR_HSB_DISTANCE_THRESHOLD;
          iterationCounter = 0; // reset counter
        }
      }

      myHSBSpacedColours.put(cacheKey, col);
    }
    return col;
  }

  public static Color getHSBColourspacedColourFromName(String name,
          float Hmin, float Hmax, float Smin, float Smax, float Bmin,
          float Bmax)
  {
    if (name == null)
    {
      return Color.white;
    }

    // use first three bytes from MD5 rather than String.hashCode() as short
    // strings all low number hashCodes
    byte[] hash = StringUtils.getHashedBytes(name);
    int b = hash.length > 0 ? Byte.toUnsignedInt(hash[0]) : 0;
    int g = hash.length > 1 ? Byte.toUnsignedInt(hash[1]) : 0;
    int r = hash.length > 2 ? Byte.toUnsignedInt(hash[2]) : 0;

    float[] hsbf = Color.RGBtoHSB(r, g, b, null);

    if (hsbf.length < 3)
    {
      // This really shouldn't happen
      Console.warn("Unexpected short length of HSB float array");
    }
    float h0 = hsbf.length > 0 ? hsbf[0] : 0f;
    float s0 = hsbf.length > 1 ? hsbf[1] : 0f;
    float b0 = hsbf.length > 2 ? hsbf[2] : 0f;

    // Now map these HSB values into the given ranges

    // hue wraps around 1.0->0.0 so deal with a Hmin-Hmax range that goes across
    // this
    float h1 = 0f;
    if (Hmin > Hmax)
    {
      Hmax += 1f;
    }
    h1 = Hmin + (Hmax - Hmin) * h0;
    if (h1 > 1f)
    {
      h1 -= 1f;
    }
    float s1 = Smin + (Smax - Smin) * s0;
    float b1 = Bmin + (Bmax - Bmin) * b0;

    Console.debug("Setting new colour for '" + name + "' with H=" + h1
            + ", S=" + s1 + ", B=" + b1);

    return Color.getHSBColor(h1, s1, b1);
  }
    

  /**
   * Checks if the colour is distinct from existing colours in the cache.
   *
   * @param newColor
   *          Colour to be checked.
   * @return {@code true} if the colour is distinct; {@code false} otherwise.
   */
  private static boolean isColorDistinct(Color newColor)
  {
    float[] newHSB = convertRGBColorToHSB(newColor); // Convert new color to HSB

    for (Color existingColor : myHSBSpacedColours.values())
    {
      float[] existingHSB = convertRGBColorToHSB(existingColor);
      float distanceFromExistingColor = hsbDistance(newHSB, existingHSB);
      //float distanceFromExistingColor = CIEDE2000ColorDistance(newColor, existingColor);
      if (distanceFromExistingColor < HSB_DISTANCE_THRESHOLD)
      {
        return false; // Colours are not distinct
      }
    }
    return true; // Colours are distinct
  }
  

  /**
   * Adjusts the colour using golden ratio conjugate.
   *
   * @param col
   *          Colour to be adjusted.
   * @return adjusted color
   */
  private static Color adjustColor(Color col) {
    float[] hsbComponents = convertRGBColorToHSB(col); // Convert RGB to HSB
    hsbComponents[POS_HUE] = (float) ((hsbComponents[POS_HUE]
            + GOLDEN_RATIO_CONJUGATE) % 1.0); // Adjust the hue using the
                                              // golden ratio

    if (hsbComponents[POS_HUE] > 1f)
    {
      hsbComponents[POS_HUE] -= 1f;
    }
    // slightly tweak saturation and brightness

    // range: 0.4 - 0.9
    hsbComponents[POS_SATURATION] = 0.4f
            + (hsbComponents[POS_HUE] * 0.5f);

    // range: 0.5 - 0.9
    hsbComponents[POS_BRIGHTNESS] = 0.5f
            + (hsbComponents[POS_HUE] * 0.4f);

    Color adjustedColor = Color.getHSBColor((float) hsbComponents[POS_HUE],
            hsbComponents[POS_SATURATION],
            hsbComponents[POS_BRIGHTNESS]);
    
    return adjustedColor;
  }
  

  /**
   * Converts RGB colour into HSB.
   *
   * @param color
   *          RGB colour to be converted.
   * @return Array containing HSB values.
   */
  private static float[] convertRGBColorToHSB(Color color)
  {
    float[] hsb = new float[3];
    Color.RGBtoHSB(color.getRed(), color.getGreen(), color.getBlue(), hsb);
    return hsb;
  }
  

  /**
   * Calculates the Euclidean distance between two colours in the HSB space.
   *
   * @param hsbX
   *          HSB values of the first colour.
   * @param hsbY
   *          HSB values of the second colour.
   * @return Euclidean distance between the two colours in HSB space.
   */
  private static float hsbDistance(float[] hsbX, float[] hsbY)
  {
    // Compute differences in HSB components
    float diffH = Math.abs(hsbX[POS_HUE] - hsbY[POS_HUE]);
    if (diffH > 0.5)
    {
      diffH = 1.0f - diffH; // Adjust as hue has circular nature
    }
    float diffS = hsbX[POS_SATURATION] - hsbY[POS_SATURATION];
    float diffB = hsbX[POS_BRIGHTNESS] - hsbY[POS_BRIGHTNESS];

    // Calculate and return Euclidean distance
    return (float) Math.sqrt(diffH * diffH + diffS * diffS + diffB * diffB);
  } 
  
}