import textwrap
from typing import List, TYPE_CHECKING
from collections.abc import Sequence

import pytest

from debputy.lsp.languages.lsp_debian_control import _reformat_debian_control
from debputy.lsp.languages.lsp_debian_watch import _reformat_debian_watch
from debputy.lsp.maint_prefs import MaintainerPreferenceTable
from debputy.lsp.text_edit import apply_text_edits
from debputy.packages import DctrlParser
from debputy.plugin.api.feature_set import PluginProvidedFeatureSet
from lint_tests.lint_tutil import (
    ReformatWrapper,
    apply_formatting_edits,
)

if TYPE_CHECKING:
    import lsprotocol.types as types
else:
    import debputy.lsprotocol.types as types


@pytest.fixture
def reformater(
    debputy_plugin_feature_set: PluginProvidedFeatureSet,
    lint_dctrl_parser: DctrlParser,
    maintainer_preference_table: MaintainerPreferenceTable,
) -> ReformatWrapper:

    return ReformatWrapper(
        "/nowhere/debian/watch",
        _reformat_debian_watch,
        debputy_plugin_feature_set,
        lint_dctrl_parser,
        maintainer_preference_table,
    )


def test_dwatch_reformat(reformater: ReformatWrapper) -> None:
    lines = textwrap.dedent(
        """\
        Version: 5

        Source: http://example.com/release/@PACKAGE@.html
        # This comment should be preserved
        MatchingPattern: files/@PACKAGE@@ANY_VERSION@@ARCHIVE_EXT@
        # This comment should be preserved
        Pgp-SigUrlMangle: s%@ARCHIVE_EXT@$%.asc%
        Decompression: yes
    """
    ).splitlines(keepends=True)

    edits = reformater.reformat(lines)
    # By default, we do nothing
    assert not edits

    edits_black = reformater.reformat_with_named_style("black", lines)
    assert edits_black

    actual_reformatted_black = apply_formatting_edits(lines, edits_black)

    expected_reformatted_black = textwrap.dedent(
        """\
        Version: 5

        Source: http://example.com/release/@PACKAGE@.html
        # This comment should be preserved
        Matching-Pattern: files/@PACKAGE@@ANY_VERSION@@ARCHIVE_EXT@
        # This comment should be preserved
        Pgp-Sig-Url-Mangle: s%@ARCHIVE_EXT@$%.asc%
        Decompression: yes
    """
    )

    assert actual_reformatted_black == expected_reformatted_black
