; Java Library
;
; Copyright (C) 2020 Kestrel Institute (http://www.kestrel.edu)
;
; License: A 3-clause BSD license. See the LICENSE file distributed with ACL2.
;
; Author: Alessandro Coglio (coglio@kestrel.edu)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package "JAVA")

(include-book "java-primitive-array-model")
(include-book "java-types")

(include-book "kestrel/std/system/arity-plus" :dir :system)
(include-book "kestrel/std/system/function-namep" :dir :system)
(include-book "kestrel/std/system/number-of-results-plus" :dir :system)
(include-book "kestrel/std/system/table-alist-plus" :dir :system)
(include-book "std/lists/index-of" :dir :system)
(include-book "std/typed-lists/cons-listp" :dir :system)
(include-book "std/util/defval" :dir :system)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defxdoc+ atj-types
  :parents (atj-implementation)
  :short "Types used by ATJ for code generation."
  :long
  (xdoc::topstring
   (xdoc::p
    "In order to make the generated Java code more efficient and idiomatic,
     ATJ uses types that correspond to both ACL2 predicates and Java types.
     These ATJ types are used only when
     @(':deep') is @('nil') and @(':guards') is @('t').")
   (xdoc::p
    "For example, consider a unary ACL2 function
     whose guard is or implies @(tsee stringp),
     and the corresponding Java method generated by ATJ.
     Since under the assumption of guard satisfaction
     this method will always be called
     with an @('Acl2Value') that is an @('Acl2String'),
     the method can use @('Acl2String') instead of @('Acl2Value')
     as the type of the argument.
     Furthermore, suppose that, under the guard,
     the ACL2 function always returns @(tsee integerp).
     Then the Java method can use @('Acl2Integer') instead of @('Acl2Value')
     as the return type.
     In other words,
     narrower types than the one for all ACL2 values (i.e. @('Acl2Value'))
     can be used for the argument and result of this Java method.
     This narrowing is also used to generate methods
     that operate on Java primitive values and primitive arrays.")
   (xdoc::p
    "In general, establishing the narrower input and output types
     for a Java method generated from an ACL2 function
     may involve arbitrarily hard theorem proving:
     (i) proving that the guard implies that the inputs of the ACL2 function
     satisfy the ACL2 predicates corresponding to the input types, and
     (ii) proving that the guard implies that the outputs of the ACL2 function
     satisfy the ACL2 predicates corresponding to the output types;
     the number of outputs of an ACL2 function is greater than 1
     if the function returns an @(tsee mv) value;
     otherwise the number of outputs is 1.
     Since we do not want ATJ to attempt any theorem proving,
     we provide a macro @(tsee atj-main-function-type)
     to perform those theorem proving tasks under user control
     and to record the input and output types of ACL2 functions in a table,
     and we have ATJ make use of this table.
     Note that these types are different from
     ACL2's built-in types used for typeset reasoning,
     ACL2's tau system types,
     and our ACL2 model of Java types.")
   (xdoc::p
    "With a table of the types of the involved ACL2 functions at hand
     (the table being constructed
     via calls of @(tsee atj-main-function-type)),
     ATJ performs a type analysis of the ACL2 terms in function bodies
     before translating them to Java.
     Generally speaking,
     ATJ compares the type inferred for an actual argument of a function
     (this type is inferred by analyzing terms recursively)
     with the type of the corresponding formal argument of the function
     (this type is retrieved from the table of function types):
     if they differ, ATJ inserts code to convert from the former to the latter,
     unless the former is a subtype of the latter in Java.
     The conversion may be a type cast,
     e.g. to convert from @('Acl2Value') to @('Acl2String');
     the cast is guaranteed to succeed,
     assuming that the ACL2 guards are verified.
     The conversion may also be a change in representation in the future.")
   (xdoc::p
    "The ATJ type information stored in the table
     determines/specifies the input and output types of the Java methods
     generated for the corresponding ACL2 functions.
     In general, there may be different choices of types possible
     for certain ACL2 functions:
     different choices will lead to different Java code.
     For instance,
     if a function's guard implies that an argument satisfies @(tsee integerp),
     that function's argument can be assigned
     a type corresponding to @('Acl2Integer'),
     or a type corresponding to @('Acl2Rational').
     The types of these Java methods are part of the ``API''
     that the generated Java code provides to external Java code.")
   (xdoc::p
    "In some cases, ACL2 functions return outputs of narrower types
     when given inputs of narrower types.
     Prime examples are the arithmetic operations
     @(tsee binary-+), @(tsee binary-*), and @(tsee unary--).
     All of their input and output types are
     the type corresponding to @(tsee acl2-numberp),
     based on their guards:
     this can be recorded via @(tsee atj-main-function-type).
     Based on these types, the corresponding Java methods
     will take and return @('Acl2Number') values.
     Now, consider a unary function @('f') that takes integers
     (i.e. it has a recorded input type corresponding to @(tsee integerp)),
     and a term @('(f (binary-+ <i> <j>))'),
     where @('<i>') and @('<j>') are integer-valued terms.
     When this term is translated to Java,
     a cast (from @('Acl2Number')) to @('Acl2Integer') will be inserted
     around the call of the method corresponding to @(tsee binary-+),
     in order to fit the @('Acl2Integer') type of
     the argument of the method corresponding to @('f').")
   (xdoc::p
    "However, due to well-known closure properties,
     @(tsee binary-+), like @(tsee binary-+) and @(tsee unary--),
     maps @(tsee rationalp) inputs to @(tsee rationalp) outputs,
     and @(tsee integerp) inputs to @(tsee integerp) outputs.
     This means that we could generate three overloaded methods
     for each such ACL2 function:
     one with @('Acl2Number') argument and result types (as above),
     one with @('Acl2Rational') argument and result types, and
     one with @('Acl2Integer') argument and result types.
     This will make the cast in the example above unnecessary,
     because if the Java expressions that translate @('<i>') and @('<j>')
     statically have type @('Acl2Integer'),
     the Java compiler will pick the most specific overloaded method,
     which returns @('Acl2Integer').")
   (xdoc::p
    "This is not limited to primitive arithmetic operations.
     Any ACL2 function may have the property of
     returning outputs of narrower types when given inputs of narrower types.
     Even if the output types are not narrower,
     the internal computations may be more efficient on narrower inputs,
     e.g. the cast in the example above can be avoided
     when that call of @('f') is part of some function @('g')
     that may not even return numbers (e.g. it may return booleans).")
   (xdoc::p
    "Thus, we provide another macro, @(tsee atj-other-function-type),
     to record additional input and output types for ACL2 functions.
     ATJ makes use of these additional types
     to generate multiple overloaded methods for single ACL2 functions.
     In general, via these two macros, each ACL2 function may have
     more than one input/output type associated with it
     (where an input/output type is a full function type,
     consisting of zero or more input types and one or more output types):
     (i) a primary (`main') input/output type,
     provable from the guards as described above; and
     (ii) zero or more secondary (`other') input/output types.
     The secondary input types are narrower than the primary ones,
     but do not have to be provable just from the guard;
     what must be proved,
     via a theorem generated by @(tsee atj-other-function-type),
     is that the guard and the input types imply the output type.")
   (xdoc::p
    "The above is just an overview of the use of types by ATJ.
     More details are in the documentation of their implementation
     and of the code generation functions that use them."))
  :order-subtopics t
  :default-parent t)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(fty::deftagsum atj-atype
  :short "Fixtype of the ATJ types that denote built-in ACL2 types."
  :long
  (xdoc::topstring-p
   "See @(tsee atj-type).")
  (:integer ())
  (:rational ())
  (:number ())
  (:character ())
  (:string ())
  (:symbol ())
  (:cons ())
  (:value ())
  :pred atj-atypep)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(fty::deftagsum atj-type
  :short "Fixtype of all the ATJ types."
  :long
  (xdoc::topstring
   (xdoc::p
    "These are used for code generation, as explained "
    (xdoc::seetopic "atj-types" "here")
    ".")
   (xdoc::p
    "Currently ATJ uses types for
     all the AIJ public class types for ACL2 values
     (integers, rationals, numbers,
     characters, strings, symbols,
     @(tsee cons) pairs, and all values),
     whose fixtype is @(tsee atj-atype),
     as well as types for the Java primitive types,
     and types for Java primitive array types.
     More types may be added in the future.")
   (xdoc::p
    "Each ATJ type denotes
     (i) an ACL2 predicate (see @(tsee atj-type-to-pred)) and
     (ii) a Java type (see @(tsee atj-type-to-jitype)).
     It is not the case that
     just the @(':acl2') types denote ACL2 types
     and just the @(':jprim') and @(':jprimarr') types denote Java types:
     each type denotes both an ACL2 and a Java type.
     The distinction is just that
     the @(':acl2') types denote built-in ACL2 types,
     which are therefore independent from Java
     (even though they have a Java representation, in AIJ),
     while the @(':jprim') and @(':jprimarr') types are Java-specific."))
  (:acl2 ((get atj-atype)))
  (:jprim ((get primitive-type)))
  (:jprimarr ((comp primitive-type)))
  :pred atj-typep
  ///

  (defruled atj-type-equiv-alt-def
    (equal (atj-type-equiv x y)
           (atj-type-case
            x
            :acl2 (atj-type-case
                   y
                   :acl2 (atj-atype-equiv x.get y.get)
                   :jprim nil
                   :jprimarr nil)
            :jprim (atj-type-case
                    y
                    :acl2 nil
                    :jprim (primitive-type-equiv x.get y.get)
                    :jprimarr nil)
            :jprimarr (atj-type-case
                       y
                       :acl2 nil
                       :jprim nil
                       :jprimarr (primitive-type-equiv x.comp y.comp))))
    :enable (atj-type-fix
             atj-type-acl2->get
             atj-type-jprim->get
             atj-type-jprimarr->comp
             atj-type-kind-possibilities)
    :rule-classes :definition))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-irrelevant ()
  :returns (type atj-typep)
  :short "An irrelevant ATJ type,
          usable as dummy return value with hard errors."
  (with-guard-checking :none (ec-call (atj-type-fix :irrelevant)))
  ///
  (in-theory (disable (:e atj-type-irrelevant))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-to-keyword ((type atj-typep))
  :returns (kwd keywordp)
  :short "Map each ATJ type to a distinct keyword."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is useful, for instance, to print ATJ types in a more readable form
     that hides the internal representation of their fixtype.
     We also these keywords to refer to the types
     in the developer documentation.")
   (xdoc::p
    "Also see @(tsee atj-type-from-keyword)."))
  (atj-type-case type
                 :acl2 (atj-atype-case type.get
                                       :integer :ainteger
                                       :rational :arational
                                       :number :anumber
                                       :character :acharacter
                                       :string :astring
                                       :symbol :asymbol
                                       :cons :acons
                                       :value :avalue)
                 :jprim (primitive-type-case type.get
                                             :boolean :jboolean
                                             :char :jchar
                                             :byte :jbyte
                                             :short :jshort
                                             :int :jint
                                             :long :jlong
                                             :float :jfloat
                                             :double :jdouble)
                 :jprimarr (primitive-type-case type.comp
                                                :boolean :jboolean[]
                                                :char :jchar[]
                                                :byte :jbyte[]
                                                :short :jshort[]
                                                :int :jint[]
                                                :long :jlong[]
                                                :float :jfloat[]
                                                :double :jdouble[]))
  :hooks (:fix)
  ///

  (defrule atj-type-to-keyword-injective
    (equal (equal (atj-type-to-keyword type1)
                  (atj-type-to-keyword type2))
           (atj-type-equiv type1 type2))
    :enable (atj-type-fix
             primitive-type-fix
             atj-type-acl2->get
             atj-type-jprim->get
             atj-type-jprimarr->comp)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-from-keyword ((kwd keywordp))
  :returns (type atj-typep)
  :short "Map keywords back to ATJ types."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is the inverse of @(tsee atj-type-to-keyword).")
   (xdoc::p
    "An error occurs if the keyword does not correspond to any ATJ type."))
  (case kwd
    (:ainteger (atj-type-acl2 (atj-atype-integer)))
    (:arational (atj-type-acl2 (atj-atype-rational)))
    (:anumber (atj-type-acl2 (atj-atype-number)))
    (:acharacter (atj-type-acl2 (atj-atype-character)))
    (:astring (atj-type-acl2 (atj-atype-string)))
    (:asymbol (atj-type-acl2 (atj-atype-symbol)))
    (:acons (atj-type-acl2 (atj-atype-cons)))
    (:avalue (atj-type-acl2 (atj-atype-value)))
    (:jboolean (atj-type-jprim (primitive-type-boolean)))
    (:jchar (atj-type-jprim (primitive-type-char)))
    (:jbyte (atj-type-jprim (primitive-type-byte)))
    (:jshort (atj-type-jprim (primitive-type-short)))
    (:jint (atj-type-jprim (primitive-type-int)))
    (:jlong (atj-type-jprim (primitive-type-long)))
    (:jfloat (atj-type-jprim (primitive-type-float)))
    (:jdouble (atj-type-jprim (primitive-type-double)))
    (:jboolean[] (atj-type-jprimarr (primitive-type-boolean)))
    (:jchar[] (atj-type-jprimarr (primitive-type-char)))
    (:jbyte[] (atj-type-jprimarr (primitive-type-byte)))
    (:jshort[] (atj-type-jprimarr (primitive-type-short)))
    (:jint[] (atj-type-jprimarr (primitive-type-int)))
    (:jlong[] (atj-type-jprimarr (primitive-type-long)))
    (:jfloat[] (atj-type-jprimarr (primitive-type-float)))
    (:jdouble[] (atj-type-jprimarr (primitive-type-double)))
    (otherwise (prog2$ (raise
                        "The keyword ~x0 does not correspond to any ATJ type."
                        kwd)
                       (atj-type-irrelevant))))
  ///

  (defrule atj-type-from-keyword-of-atj-type-to-keyword
    (equal (atj-type-from-keyword (atj-type-to-keyword type))
           (atj-type-fix type))
    :enable (atj-type-to-keyword
             atj-type-fix
             atj-type-acl2->get
             atj-type-jprim->get
             atj-type-jprimarr->comp)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(fty::deflist atj-type-list
  :short "Fixtype of true lists of ATJ types."
  :elt-type atj-type
  :true-listp t
  :elementp-of-nil nil
  :pred atj-type-listp
  ///

  (defruled atj-type-list-equiv-alt-def
    (equal (atj-type-list-equiv x y)
           (cond ((endp x) (endp y))
                 ((endp y) (endp x))
                 (t (and (atj-type-equiv (car x) (car y))
                         (atj-type-list-equiv (cdr x) (cdr y))))))
    :enable atj-type-list-fix
    :rule-classes :definition))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-list-to-keyword-list ((types atj-type-listp))
  :returns (kwds keyword-listp)
  :short "Lift @(tsee atj-type-to-keyword) to lists."
  (cond ((endp types) nil)
        (t (cons (atj-type-to-keyword (car types))
                 (atj-type-list-to-keyword-list (cdr types)))))
  :hooks (:fix))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-list-from-keyword-list ((kwds keyword-listp))
  :returns (types atj-type-listp)
  :short "Lift @(tsee atj-type-from-keyword) to lists."
  (cond ((endp kwds) nil)
        (t (cons (atj-type-from-keyword (car kwds))
                 (atj-type-list-from-keyword-list (cdr kwds))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(fty::deflist atj-type-list-list
  :short "Fixtype of true lists of true lists of ATJ types."
  :elt-type atj-type-list
  :true-listp t
  :elementp-of-nil t
  :pred atj-type-list-listp
  ///

  (defrule atj-type-list-listp-of-remove-duplicates-equal
    (implies (atj-type-list-listp x)
             (atj-type-list-listp (remove-duplicates-equal x)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(fty::defalist atj-symbol-type-alist
  :short "Fixtype of alists from symbols to ATJ types."
  :key-type symbol
  :val-type atj-type
  :true-listp t
  :keyp-of-nil t
  :valp-of-nil nil
  :pred atj-symbol-type-alistp
  ///

  (defrule atj-typep-of-cdr-of-assoc-equal-when-atj-symbol-type-alistp
    (implies (atj-symbol-type-alistp alist)
             (iff (atj-typep (cdr (assoc-equal key alist)))
                  (assoc-equal key alist))))

  (defrule atj-symbol-type-alistp-of-pairlis$
    (implies (and (symbol-listp keys)
                  (atj-type-listp vals)
                  (equal (len keys) (len vals)))
             (atj-symbol-type-alistp (pairlis$ keys vals)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(fty::defoption atj-maybe-type
  atj-type
  :short "Fixtype of ATJ types and @('nil')."
  :pred atj-maybe-typep)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(fty::deflist atj-maybe-type-list
  :short "Fixtype of lists of ATJ types and @('nil')."
  :elt-type atj-maybe-type
  :true-listp t
  :elementp-of-nil t
  :pred atj-maybe-type-listp
  ///

  (defruled atj-maybe-type-list-equiv-alt-def
    (equal (atj-maybe-type-list-equiv x y)
           (cond ((endp x) (endp y))
                 ((endp y) (endp x))
                 (t (and (atj-maybe-type-equiv (car x) (car y))
                         (atj-maybe-type-list-equiv (cdr x) (cdr y))))))
    :enable atj-maybe-type-list-fix
    :rule-classes :definition)

  (defrule atj-maybe-type-listp-when-atj-type-listp
    (implies (atj-type-listp x)
             (atj-maybe-type-listp x))
    :enable atj-maybe-type-listp))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-to-pred ((type atj-typep))
  :returns (pred pseudo-termfnp)
  :short "ACL2 predicate denoted by an ATJ type."
  :long
  (xdoc::topstring
   (xdoc::p
    "The predicate recognizes the values of the type.")
   (xdoc::p
    "The predicates for the @(':acl2') types are straightforward.
     The predicates for the @(':jprim') types
     are the recognizers of the corresponding
     Java primitive values in our Java language formalization.
     The predicates for the @(':jprimarr') types
     are the recognizers of the corresponding
     Java primitive arrays in our ATJ's model of primitive arrays.
     Also see "
    (xdoc::seetopic "atj-java-primitives" "here")
    " and "
    (xdoc::seetopic "atj-java-primitive-arrays" "here")
    "."))
  (atj-type-case type
                 :acl2 (atj-atype-case type.get
                                       :integer 'integerp
                                       :rational 'rationalp
                                       :number 'acl2-numberp
                                       :character 'characterp
                                       :string 'stringp
                                       :symbol 'symbolp
                                       :cons 'consp
                                       :value '(lambda (_) 't))
                 :jprim (primitive-type-case type.get
                                             :boolean 'boolean-value-p
                                             :char 'char-value-p
                                             :byte 'byte-value-p
                                             :short 'short-value-p
                                             :int 'int-value-p
                                             :long 'long-value-p
                                             :float 'float-value-p
                                             :double 'double-value-p)
                 :jprimarr (primitive-type-case type.comp
                                                :boolean 'boolean-array-p
                                                :char 'char-array-p
                                                :byte 'byte-array-p
                                                :short 'short-array-p
                                                :int 'int-array-p
                                                :long 'long-array-p
                                                :float 'float-array-p
                                                :double 'double-array-p))
  :hooks (:fix))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-atype-<= ((sub atj-atypep) (sup atj-atypep))
  :returns (yes/no booleanp)
  :short "Partial order over the ATJ types that denote built-in ACL2 types."
  :long
  (xdoc::topstring-p
   "See @(tsee atj-type-<=).")
  (and
   (atj-atype-case
    sub
    :integer (member-eq (atj-atype-kind sup) '(:integer :rational :number
                                               :value))
    :rational (member-eq (atj-atype-kind sup) '(:rational :number :value))
    :number (member-eq (atj-atype-kind sup) '(:number :value))
    :character (member-eq (atj-atype-kind sup) '(:character :value))
    :string (member-eq (atj-atype-kind sup) '(:string :value))
    :symbol (member-eq (atj-atype-kind sup) '(:symbol :value))
    :cons (member-eq (atj-atype-kind sup) '(:cons :value))
    :value (atj-atype-case sup :value))
   t)
  :hooks (:fix)
  ///

  (defrule atj-atype-<=-reflexive
    (implies (atj-atype-equiv x y)
             (atj-atype-<= x y)))

  (defrule atj-atype-<=-antisymmetric
    (implies (and (atj-atype-<= x y)
                  (atj-atype-<= y x))
             (atj-atype-equiv x y)))

  (defrule atj-atype-<=-transitive
    (implies (and (atj-atype-<= x y)
                  (atj-atype-<= y z))
             (atj-atype-<= x z)))

  (defrule atj-atype-<=-of-value-left
    (implies (atj-atype-case x :value)
             (equal (atj-atype-<= x y)
                    (atj-atype-case y :value))))

  (defrule atj-atype-<=-of-cons-left
    (implies (atj-atype-case x :cons)
             (equal (atj-atype-<= x y)
                    (or (atj-atype-case y :cons)
                        (atj-atype-case y :value))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-<= ((sub atj-typep) (sup atj-typep))
  :returns (yes/no booleanp)
  :short "Partial order over all the ATJ types."
  :long
  (xdoc::topstring
   (xdoc::p
    "The ATJ types form a partial order.
     The ordering on the @(':acl2') types
     is based on the inclusion of the ACL2 predicates they denote;
     this denotation is defined by @(tsee atj-type-to-pred).
     Each of the @(':jprim') and @(':jprimarr') types
     is incomparable with all the types except itself.")
   (xdoc::p
    "This definition of partial order is motivated by
     the conversions that we want to allow, in the generated Java code,
     between (the Java representations of) the ACL2 values denoted by the types.
     While we want to allow conversions between the @(':acl2') types,
     we keep the @(':jprim') and @(':jprimarr') types
     all separate from each other and from the @(':acl2') types.
     This is not the only possible choice and definition of partial order.")
   (xdoc::p
    "To validate this definition of partial order,
     we prove that the relation is indeed a partial order,
     i.e. reflexive, anti-symmetric, and transitive.
     We also prove that @(tsee atj-type-to-pred) is monotonic,
     i.e. that for each subtype/supertype pair
     each value satisfying the subtype's predicate
     also satisfies the supertype's predicate;
     we generate a theorem for each such pair,
     because the predicate inclusion relation is at the meta level.
     The motonocity theorem validates that the partial order
     is consistent with the inclusion of the denoted ACL2 types.")
   (xdoc::p
    "While @(tsee atj-type-to-pred) is order-presering (i.e. monotonic),
     it is not order-reflecting (and thus not an order embedding):
     if @('(atj-type-to-pred x)') is included in @('(atj-type-to-pred y)'),
     @('(atj-type-<= x y)') does not necessarily hold.
     The counterexample to being order-reflective consists of
     @('x') being a @(':jprim') or @(':jprimarr') type and
     @('y') being the @(':acl2') type of all ACL2 values."))
  (atj-type-case
   sub
   :acl2 (atj-type-case sup
                        :acl2 (atj-atype-<= sub.get sup.get)
                        :jprim nil
                        :jprimarr nil)
   :jprim (atj-type-case sup
                         :acl2 nil
                         :jprim (primitive-type-equiv sub.get sup.get)
                         :jprimarr nil)
   :jprimarr (atj-type-case sup
                            :acl2 nil
                            :jprim nil
                            :jprimarr (primitive-type-equiv sub.comp sup.comp)))
  :hooks (:fix)
  ///

  (defrule atj-type-<=-reflexive
    (implies (atj-type-equiv x y)
             (atj-type-<= x y)))

  (defrule atj-type-<=-antisymmetric
    (implies (and (atj-type-<= x y)
                  (atj-type-<= y x))
             (atj-type-equiv x y))
    :disable (atj-type-equiv atj-atype-equiv primitive-type-equiv)
    :enable (atj-type-equiv-alt-def
             atj-atype-<=-antisymmetric))

  (defrule atj-type-<=-transitive
    (implies (and (atj-type-<= x y)
                  (atj-type-<= y z))
             (atj-type-<= x z))
    :enable atj-atype-<=-transitive)
  ;; rewrite rule ATJ-ATYPE-<=-OF-VALUE-LEFT applies
  ;; rewrite rule ATJ-ATYPE-<=-OF-CONS-LEFT applies

  ;; monotonicity theorem for (SUB, SUP) if SUB <= SUP, otherwise NIL:
  (define atj-type-to-pred-gen-mono-thm ((sub atj-typep) (sup atj-typep))
    (if (atj-type-<= sub sup)
        `((defthm ,(packn (list 'atj-type-to-pred-thm-
                                (atj-type-to-keyword sub)
                                '-
                                (atj-type-to-keyword sup)))
            (implies (,(atj-type-to-pred sub) val)
                     (,(atj-type-to-pred sup) val))
            :rule-classes nil))
      nil)
    :hooks (:fix))

  ;; monotonicity theorems for all (SUB, SUP) with SUP' in SUPS:
  (define atj-type-to-pred-gen-mono-thms-1 ((sub atj-typep)
                                            (sups atj-type-listp))
    (cond ((endp sups) nil)
          (t (append (atj-type-to-pred-gen-mono-thm sub (car sups))
                     (atj-type-to-pred-gen-mono-thms-1 sub (cdr sups)))))
    :hooks (:fix))

  ;; monotonicity theorems for all (SUB, SUP) with SUB in SUBS and SUP in SUPS:
  (define atj-type-to-pred-gen-mono-thms-2 ((subs atj-type-listp)
                                            (sups atj-type-listp))
    (cond ((endp subs) nil)
          (t (append (atj-type-to-pred-gen-mono-thms-1 (car subs) sups)
                     (atj-type-to-pred-gen-mono-thms-2 (cdr subs) sups))))
    :hooks (:fix))

  ;; monotonicity theorems for all pairs of types:
  (define atj-type-to-pred-gen-mono-thms ()
    (b* ((types (list (atj-type-acl2 (atj-atype-integer))
                      (atj-type-acl2 (atj-atype-rational))
                      (atj-type-acl2 (atj-atype-number))
                      (atj-type-acl2 (atj-atype-character))
                      (atj-type-acl2 (atj-atype-string))
                      (atj-type-acl2 (atj-atype-symbol))
                      (atj-type-acl2 (atj-atype-cons))
                      (atj-type-acl2 (atj-atype-value))
                      (atj-type-jprim (primitive-type-boolean))
                      (atj-type-jprim (primitive-type-char))
                      (atj-type-jprim (primitive-type-byte))
                      (atj-type-jprim (primitive-type-short))
                      (atj-type-jprim (primitive-type-int))
                      (atj-type-jprim (primitive-type-long))
                      (atj-type-jprim (primitive-type-float))
                      (atj-type-jprim (primitive-type-double))
                      (atj-type-jprimarr (primitive-type-boolean))
                      (atj-type-jprimarr (primitive-type-char))
                      (atj-type-jprimarr (primitive-type-byte))
                      (atj-type-jprimarr (primitive-type-short))
                      (atj-type-jprimarr (primitive-type-int))
                      (atj-type-jprimarr (primitive-type-long))
                      (atj-type-jprimarr (primitive-type-float))
                      (atj-type-jprimarr (primitive-type-double)))))
      `(encapsulate
         ()
         (set-ignore-ok t)
         ,@(atj-type-to-pred-gen-mono-thms-2 types types))))

  ;; macro to generate the monotonicity theorems:
  (defmacro atj-type-to-pred-mono ()
    `(make-event (atj-type-to-pred-gen-mono-thms)))

  ;; generate the monotonicity theorems:
  (atj-type-to-pred-mono))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-< ((sub atj-typep) (sup atj-typep))
  :returns (yes/no booleanp)
  :short "Irreflexive kernel (i.e. strict version) of @(tsee atj-type-<=)."
  :long
  (xdoc::topstring
   "Since @(tsee atj-type-<=) fixes its arguments,
    we use (in)equality modulo fixing here,
    so that this function fixes its arguments too.")
  (and (atj-type-<= sub sup)
       (not (atj-type-equiv sub sup)))
  :hooks (:fix))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-maybe-type-<= ((sub atj-maybe-typep) (sup atj-maybe-typep))
  :returns (yes/no booleanp)
  :short "Extension of @(tsee atj-type-<=) to include @('nil') as top."
  :long
  (xdoc::topstring
   (xdoc::p
    "For certain purposes, we want to calculate
     the least upper bound of two ATJ types w.r.t. @(tsee atj-type-<=).
     However, the ATJ types with this partial order
     do not quite form a join semilattice,
     because there are no upper bounds, for instance,
     for two different @(':jprim') types.")
   (xdoc::p
    "Thus, we extend the partial order
     to the set of ATJ types plus @('nil'),
     where @('nil') is above every type.")
   (xdoc::p
    "We show that this extended relation is a partial order,
     i.e. reflexive, anti-symmetric, and transitive."))
  (or (null sup)
      (and (not (null sub))
           (atj-type-<= sub sup)))
  :hooks (:fix)
  ///

  (defrule atj-maybe-type-<=-reflexive
    (implies (atj-maybe-type-equiv x y)
             (atj-maybe-type-<= x y)))

  (defrule atj-maybe-type-<=-antisymmetric
    (implies (and (atj-maybe-type-<= x y)
                  (atj-maybe-type-<= y x))
             (atj-maybe-type-equiv x y))
    :enable (atj-maybe-type-fix
             atj-type-fix
             atj-type-acl2->get
             atj-type-jprim->get
             atj-type-jprimarr->comp
             atj-type-<=
             atj-atype-<=))

  (defrule atj-maybe-type-<=-transitive
    (implies (and (atj-maybe-type-<= x y)
                  (atj-maybe-type-<= y z))
             (atj-maybe-type-<= x z))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-maybe-type-join ((x atj-maybe-typep) (y atj-maybe-typep))
  :returns (lub atj-maybe-typep)
  :short "Least upper bound of two ATJ types or @('nil')s."
  :long
  (xdoc::topstring
   (xdoc::p
    "As discussed in @(tsee atj-maybe-type-<=),
     the addition of @('nil') as top element of the partial order
     results in a join semilattice.")
   (xdoc::p
    "We define this operation by five cases:
     the first two are obvious,
     while the remaining three are motivated by the fact that
     @(':acons'), @(':avalue'), and @('nil') are the only elements
     each of which has more than one elements strictly smaller.")
   (xdoc::p
    "To validate this definition of least upper bound,
     we prove that the this operation indeed returns an upper bound
     that is less than or equal to any other upper bound,
     i.e. that it returns the least upper bound.")
   (xdoc::p
    "The commutativity, idempotence, and associativity of the join operation
     follows from these and the partial order properties,
     according to lattice theory.
     So we do not prove these properties explicitly here."))
  (cond ((atj-maybe-type-<= x y) (atj-maybe-type-fix y))
        ((atj-maybe-type-<= y x) (atj-maybe-type-fix x))
        ((and (atj-maybe-type-<= x (atj-type-acl2 (atj-atype-cons)))
              (atj-maybe-type-<= y (atj-type-acl2 (atj-atype-cons))))
         (atj-type-acl2 (atj-atype-cons)))
        ((and (atj-maybe-type-<= x (atj-type-acl2 (atj-atype-value)))
              (atj-maybe-type-<= y (atj-type-acl2 (atj-atype-value))))
         (atj-type-acl2 (atj-atype-value)))
        (t nil))
  :hooks (:fix)
  ///

  (defrule atj-maybe-type-join-upper-bound-left
    (atj-maybe-type-<= x (atj-maybe-type-join x y))
    :enable atj-maybe-type-<=)

  (defrule atj-maybe-type-join-upper-bound-right
    (atj-maybe-type-<= y (atj-maybe-type-join x y))
    :enable atj-maybe-type-<=)

  (defrule atj-type-join-least
    (implies (and (atj-maybe-type-<= x z)
                  (atj-maybe-type-<= y z))
             (atj-maybe-type-<= (atj-maybe-type-join x y) z))
    :enable (atj-maybe-type-<= atj-type-<= atj-atype-<=)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-join ((x atj-typep) (y atj-typep))
  :returns (lub atj-maybe-typep)
  :short "Least upper bound of two ATJ types."
  :long
  (xdoc::topstring
   (xdoc::p
    "We have defined @(tsee atj-maybe-type-join)
     in order to exhibit and prove the semilattice structure,
     but we always want to use ATJ types as arguments, never @('nil').
     So we introduce this function,
     which operates on types but may return @('nil'),
     which can be also interpreted as saying that
     the two ATJ types have no (least) upper bound w.r.t @(tsee atj-type-<=)."))
  (atj-maybe-type-join x y))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-list-<= ((sub atj-type-listp) (sup atj-type-listp))
  :returns (yes/no booleanp)
  :short "Lift @(tsee atj-type-<=) to lists."
  :long
  (xdoc::topstring
   (xdoc::p
    "Lists are ordered element-wise.
     Given two lists of different lengths
     such that the shorter one is a prefix of the longer one
     (i.e. the two lists cannot be ordered based on their initial elements),
     the shorter one is smaller than the longer one.")
   (xdoc::p
    "We show that the resulting relation is a partial order,
     i.e. reflexive, anti-symmetric, and transitive."))
  (cond ((endp sub) t)
        ((endp sup) nil)
        (t (and (atj-type-<= (car sub) (car sup))
                (atj-type-list-<= (cdr sub) (cdr sup)))))
  :hooks (:fix)
  ///

  (defrule atj-type-list-<=-reflexive
    (implies (atj-type-list-equiv x y)
             (atj-type-list-<= x y))
    :enable atj-type-list-fix)

  (defrule atj-type-list-<=-antisymmetric
    (implies (and (atj-type-list-<= x y)
                  (atj-type-list-<= y x))
             (atj-type-list-equiv x y))
    :disable atj-type-list-equiv
    :enable atj-type-list-equiv-alt-def)

  (defrule atj-type-list-<=-transitive
    (implies (and (atj-type-list-<= x y)
                  (atj-type-list-<= y z))
             (atj-type-list-<= x z))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-list-< ((sub atj-type-listp) (sup atj-type-listp))
  :returns (yes/no booleanp)
  :short "Irreflexive kernel (i.e. strict version)
          of @(tsee atj-type-list-<=)."
  :long
  (xdoc::topstring
   "Since @(tsee atj-type-list-<=) fixes its arguments,
    we use (in)equality modulo fixing here,
    so that this function fixes its arguments too.")
  (and (atj-type-list-<= sub sup)
       (not (atj-type-list-equiv sub sup)))
  :hooks (:fix))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-maybe-type-list-<= ((sub atj-maybe-type-listp)
                                (sup atj-maybe-type-listp))
  :returns (yes/no booleanp)
  :short "Lift @(tsee atj-maybe-type-<=) to lists."
  :long
  (xdoc::topstring
   (xdoc::p
    "Lists are ordered element-wise.
     Given two lists of different lengths
     such that the shorter one is a prefix of the longer one
     (i.e. the two lists cannot be ordered based on their initial elements),
     the shorter one is smaller than the longer one.")
   (xdoc::p
    "We show that the resulting relation is a partial order,
     i.e. reflexive, anti-symmetric, and transitive."))
  (cond ((endp sub) t)
        ((endp sup) nil)
        (t (and (atj-maybe-type-<= (car sub) (car sup))
                (atj-maybe-type-list-<= (cdr sub) (cdr sup)))))
  :hooks (:fix)
  ///

  (defrule atj-maybe-type-list-<=-reflexive
    (implies (atj-maybe-type-list-equiv x y)
             (atj-maybe-type-list-<= x y))
    :enable atj-maybe-type-list-fix)

  (defrule atj-maybe-type-list-<=-antisymmetric
    (implies (and (atj-maybe-type-list-<= x y)
                  (atj-maybe-type-list-<= y x))
             (atj-maybe-type-list-equiv x y))
    :disable atj-maybe-type-list-equiv
    :enable atj-maybe-type-list-equiv-alt-def)

  (defrule atj-maybe-type-list-<=-transitive
    (implies (and (atj-maybe-type-list-<= x y)
                  (atj-maybe-type-list-<= y z))
             (atj-maybe-type-list-<= x z))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-maybe-type-list-join ((x atj-maybe-type-listp)
                                  (y atj-maybe-type-listp))
  :returns (lub atj-maybe-type-listp)
  :short "Lift @(tsee atj-maybe-type-join) to lists."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is done element-wise.
     When the shorter list is exhausted,
     we return (what remains of) the longer list.")
   (xdoc::p
    "We show that this indeed returns the least upper bound
     of the order relation lifted to lists."))
  (cond ((endp x) (atj-maybe-type-list-fix y))
        ((endp y) (atj-maybe-type-list-fix x))
        (t (cons (atj-maybe-type-join (car x) (car y))
                 (atj-maybe-type-list-join (cdr x) (cdr y)))))
  :hooks (:fix)
  ///

  (defrule atj-maybe-type-list-join-upper-bound-left
    (atj-maybe-type-list-<= x (atj-maybe-type-list-join x y))
    :enable atj-maybe-type-list-<=)

  (defrule atj-maybe-type-list-join-upper-bound-right
    (atj-maybe-type-list-<= y (atj-maybe-type-list-join x y))
    :enable atj-maybe-type-list-<=)

  (defrule atj-maybe-type-list-join-least
    (implies (and (atj-maybe-type-list-<= x z)
                  (atj-maybe-type-list-<= y z))
             (atj-maybe-type-list-<= (atj-maybe-type-list-join x y) z))
    :enable atj-maybe-type-list-<=)

  (defret consp-of-atj-maybe-type-list-join
    (equal (consp lub)
           (or (consp x)
               (consp y)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-list-join ((x atj-type-listp) (y atj-type-listp))
  :guard (= (len x) (len y))
  :returns (lub atj-maybe-type-listp)
  :short "Lift @(tsee atj-type-join) to lists."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is done element-wise.
     When the shorter list is exhausted,
     we return (what remains of) the longer list.")
   (xdoc::p
    "We show that this agrees with @(tsee atj-maybe-type-list-join)
     over lists of ATJ types.
     Note that @(tsee atj-maybe-type-list-join) has been defined
     just to show the semilattice properties,
     but we always want to use, as arguments,
     lists of ATJ types without @('nil')s of the same length
     (so we add a length equality requirement to the guard)."))
  (cond ((endp x) (atj-type-list-fix y))
        ((endp y) (atj-type-list-fix x))
        (t (cons (atj-type-join (car x) (car y))
                 (atj-type-list-join (cdr x) (cdr y)))))
  ///

  (defruled atj-type-list-join-alt-def
    (implies (and (atj-type-listp x)
                  (atj-type-listp y))
             (equal (atj-type-list-join x y)
                    (atj-maybe-type-list-join x y)))
    :enable (atj-maybe-type-list-join
             atj-type-join))

  (defret consp-of-atj-type-list-join
    (equal (consp lub)
           (or (consp x)
               (consp y)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-to-jitype ((type atj-typep))
  :returns (jtype atj-jitypep)
  :short "Java input type denoted by an ATJ type."
  :long
  (xdoc::topstring
   (xdoc::p
    "The @(':acl2') types denote
     the corresponding AIJ class types.
     The @(':jprim') types denote
     the corresponding Java primitive types.
     The @(':jprimarr') types denote
     the corresponding Java primitive array types.")
   (xdoc::p
    "Currently this is an injective mapping,
     each ATJ type denotes a distinct Java type.
     This might change in the future,
     as new ATJ types are added."))
  (atj-type-case
   type
   :acl2 (atj-atype-case type.get
                         :integer *aij-type-int*
                         :rational *aij-type-rational*
                         :number *aij-type-number*
                         :character *aij-type-char*
                         :string *aij-type-string*
                         :symbol *aij-type-symbol*
                         :cons *aij-type-cons*
                         :value *aij-type-value*)
   :jprim (primitive-type-case type.get
                               :boolean (jtype-boolean)
                               :char (jtype-char)
                               :byte (jtype-byte)
                               :short (jtype-short)
                               :int (jtype-int)
                               :long (jtype-long)
                               :float (jtype-float)
                               :double (jtype-double))
   :jprimarr (primitive-type-case type.comp
                                  :boolean (jtype-array (jtype-boolean))
                                  :char (jtype-array (jtype-char))
                                  :byte (jtype-array (jtype-byte))
                                  :short (jtype-array (jtype-short))
                                  :int (jtype-array (jtype-int))
                                  :long (jtype-array (jtype-long))
                                  :float (jtype-array (jtype-float))
                                  :double (jtype-array (jtype-double))))
  :hooks (:fix))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-list-to-jitype-list ((types atj-type-listp))
  :returns (jtypes atj-jitype-listp)
  :short "Lift @(tsee atj-type-to-jitype) to lists."
  (cond ((endp types) nil)
        (t (cons (atj-type-to-jitype (car types))
                 (atj-type-list-to-jitype-list (cdr types)))))
  :hooks (:fix)
  ///

  (defret len-of-atj-type-list-to-jitype-list
    (equal (len jtypes)
           (len types))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-list-list-to-jitype-list-list ((typess atj-type-list-listp))
  :returns (jtypess atj-jitype-list-listp)
  :short "Lift @(tsee atj-type-list-to-jitype-list) to lists."
  (cond ((endp typess) nil)
        (t (cons (atj-type-list-to-jitype-list (car typess))
                 (atj-type-list-list-to-jitype-list-list (cdr typess)))))
  :hooks (:fix)
  ///

  (defret len-of-atj-type-list-to-jitype-list-list
    (equal (len jtypess)
           (len typess))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-of-value (val)
  :returns (type atj-typep)
  :short "ATJ type of an ACL2 value."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is the type that ATJ assigns to a quoted constant
     with the given value.
     Note that a constant like @('2') does not get type @(':jint').
     Instead, ATJ assigns @(':jint') to a term like @('(int-value 2)')
     (when @(':deep') is @('nil') and @(':guards') is @('t'));
     see the code generation functions."))
  (cond ((integerp val) (atj-type-acl2 (atj-atype-integer)))
        ((rationalp val) (atj-type-acl2 (atj-atype-rational)))
        ((acl2-numberp val) (atj-type-acl2 (atj-atype-number)))
        ((characterp val) (atj-type-acl2 (atj-atype-character)))
        ((stringp val) (atj-type-acl2 (atj-atype-string)))
        ((symbolp val) (atj-type-acl2 (atj-atype-symbol)))
        ((consp val) (atj-type-acl2 (atj-atype-cons)))
        (t (prog2$ (raise "Internal errror: ~
                           the value ~x0 is not a number, ~
                           a character, a string, a symbol, or a CONS."
                          val)
                   (atj-type-irrelevant)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(fty::defprod atj-function-type
  :short "Fixtype of ATJ function types."
  :long
  (xdoc::topstring
   (xdoc::p
    "An ATJ function type consists of
     (zero or more) types for the arguments (i.e. inputs)
     and (one or more) types for the results (i.e. outputs).
     This is like an arrow type in higher-order languages.")
   (xdoc::p
    "We also augment the output types with array names.
     These are represented via a list of symbols,
     whose length must match the length of the output type list
     (this length constraint is not explicitly captured in this fixtype,
     but it is an expected invariant).
     The @('nil') symbol may be used in any position of the list,
     meaning that there is no array name for the corresponding output type.
     A non-@('nil') symbol may be used only in a position
     whose corresponding output type is a @(':jprimarr') type.
     In this case the symbol must match a formal parameter of the function
     that has the same array type as input type.
     The non-@('nil') symbols must be all distinct.")
   (xdoc::p
    "The purpose of these array names is to support
     the analysis of single-threaded use of Java primitive arrays
     (to justify destructive array updates).
     The idea is that if a function takes an array as input
     (i.e. that input type is a @(':jprimarr') type)
     and if the function may modify that array (directly or indirectly),
     then the possibly modified array must be returned as a result:
     the explicit non-@('nil') array name assigned to a result
     specifies which result that is, and simplifies the analysis.
     If instead a function does not modify an input array,
     no result with the same name as the input needs to exist.
     Results of non-array types use @('nil') as array (non-)name.
     If a function creates an array (directly or indirectly) returns it,
     then @('nil') is used for that result,
     i.e. the array has no name because it does not modify any input array
     (and thus there is no input name to match);
     it represents a newly created array."))
  ((inputs atj-type-list)
   (outputs atj-type-list)
   (arrays symbol-list))
  :layout :list)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(fty::defoption atj-maybe-function-type
  atj-function-type-p
  :short "Fixtype of ATJ function types and @('nil')."
  :pred atj-maybe-function-type-p)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(fty::deflist atj-function-type-list
  :short "Fixtype of true lists of ATJ function types."
  :elt-type atj-function-type
  :true-listp t
  :elementp-of-nil nil
  :pred atj-function-type-listp)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-function-type-list->inputs ((fn-types atj-function-type-listp))
  :returns (in-typess atj-type-list-listp)
  :short "Lift @(tsee atj-function-type->inputs) to lists."
  (cond ((endp fn-types) nil)
        (t (cons (atj-function-type->inputs (car fn-types))
                 (atj-function-type-list->inputs (cdr fn-types))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-function-type-list->outputs ((fn-types atj-function-type-listp))
  :returns (out-typess atj-type-list-listp)
  :short "Lift @(tsee atj-function-type->outputs) to lists."
  (cond ((endp fn-types) nil)
        (t (cons (atj-function-type->outputs (car fn-types))
                 (atj-function-type-list->outputs (cdr fn-types))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(fty::defprod atj-function-type-info
  :short "Fixtype of ATJ function type information."
  :long
  (xdoc::topstring
   (xdoc::p
    "In general, each ACL2 function has, associated with it,
     a primary (`main') function type
     and zero or more secondary (`other') function types,
     as mentioned in " (xdoc::seetopic "atj-types" "here") "."))
  ((main atj-function-type)
   (others atj-function-type-list))
  :layout :list)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-function-type-info->outputs ((info atj-function-type-info-p))
  :returns (out-typess atj-type-list-listp)
  :short "Return the list of all the output type lists
          in a function's type information."
  (cons
   (atj-function-type->outputs (atj-function-type-info->main info))
   (atj-function-type-list->outputs (atj-function-type-info->others info))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(fty::defoption atj-maybe-function-type-info
  atj-function-type-info
  :short "Fixtype of ATJ function type information and @('nil').")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-number-of-results ((fn symbolp) (wrld plist-worldp))
  :returns (numres posp)
  :short "Number of results returned by a function."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is similar to @(tsee number-of-results+),
     but that function has a guard disallowing the function symbol
     to be a member of the built-in constant @('*stobjs-out-invalid*'),
     i.e. to be @(tsee if) or @(tsee return-last).
     For ATJ's purpose, we totalize @(tsee number-of-results+)
     by having it return 1 on these two functions.
     The actual result is irrelevant,
     because @(tsee return-last) is removed
     by one of ATJ's pre-translation steps,
     and @(tsee if) is treated specially by ATJ."))
  (cond ((member-eq fn *stobjs-out-invalid*) 1)
        (t (number-of-results+ fn wrld))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defval *atj-function-type-info-table-name*
  :short "Name of the table that associates ATJ types to ACL2 functions."
  'atj-function-type-info-table)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defsection atj-function-type-info-table
  :short "Table that associates ATJ types to ACL2 functions."
  :long
  (xdoc::topstring
   (xdoc::p
    "This table is populated by successful calls of
     the @(tsee atj-main-function-type)
     and @(tsee atj-other-function-type) macros."))
  (make-event
   `(table ,*atj-function-type-info-table-name* nil nil
      :guard (and (symbolp acl2::key)
                  (atj-function-type-info-p acl2::val)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-get-function-type-info-from-table ((fn symbolp) (wrld plist-worldp))
  :returns (fn-info? atj-maybe-function-type-info-p)
  :short "Retrieve the ATJ function type information
          of the specified function from the table."
  :long
  (xdoc::topstring-p
   "If the table has no entry for the function, @('nil') is returned.")
  (b* ((table (table-alist+ *atj-function-type-info-table-name* wrld))
       (pair (assoc-eq fn table))
       ((when pair)
        (b* ((fn-info (cdr pair)))
          (if (atj-function-type-info-p fn-info)
              fn-info
            (raise "Internal error: ~
                    malformed function information ~x0 for function ~x1."
                   fn-info fn)))))
    nil)
  :prepwork ((local (include-book "std/alists/assoc" :dir :system)))
  ///

  (defrule atj-function-type-info-p-of-atj-get-function-type-info-from-table
    (iff (atj-function-type-info-p
          (atj-get-function-type-info-from-table fn wrld))
         (atj-get-function-type-info-from-table fn wrld))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-function-type-info-default ((fn symbolp) (wrld plist-worldp))
  :returns (fn-info atj-function-type-info-p)
  :short "Default ATJ function type information for a function."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is used when a function has no entry in the table.
     It consists of a primary function type of all @(':avalue') types,
     no secondary function types,
     and all @('nil') array names (since there are no array types).")
   (xdoc::p
    "To calculate the output types,
     we need the number of results returned by @('fn')."))
  (b* ((number-of-inputs (arity+ fn wrld))
       (number-of-outputs (atj-number-of-results fn wrld)))
    (make-atj-function-type-info
     :main (make-atj-function-type
            :inputs (repeat number-of-inputs
                            (atj-type-acl2 (atj-atype-value)))
            :outputs (repeat number-of-outputs
                             (atj-type-acl2 (atj-atype-value)))
            :arrays (repeat number-of-outputs nil))
     :others nil))
  :prepwork ((local
              (include-book "std/typed-lists/symbol-listp" :dir :system))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-get-function-type-info ((fn symbolp)
                                    (guards$ booleanp)
                                    (wrld plist-worldp))
  :returns (fn-info atj-function-type-info-p)
  :short "Obtain the ATJ function type information of the specified function."
  :long
  (xdoc::topstring
   (xdoc::p
    "If the @(':guards') input is @('t'),
     we retrieve the type information from the table
     via @(tsee atj-get-function-type-info-from-table).
     If the @(':guards') input is @('nil'),
     we return the defult function type information,
     because in this case types are effectively ignored."))
  (if guards$
      (b* ((fn-info? (atj-get-function-type-info-from-table fn wrld)))
        (or fn-info? (atj-function-type-info-default fn wrld)))
    (atj-function-type-info-default fn wrld)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-process-output-type-spec (out-type-spec
                                      (formals symbol-listp)
                                      (in-types atj-type-listp))
  :guard (= (len formals) (len in-types))
  :returns (mv (type atj-typep)
               (array symbolp))
  :short "Process an output type specification in
          @(tsee atj-main-function-type) or @(tsee atj-other-function-type)."
  :long
  (xdoc::topstring
   (xdoc::p
    "An output type must be either a keyword that denotes an ATJ type,
     or a doublet consisting of a non-@('nil') symbol
     followed by a keyword that denotes an ATJ type.
     The second case is allowed only if the type is an array one,
     and in that case the non-@('nil') symbol must be
     some formal argument of the function that has the same array type:
     the non-@('nil') is the name of the array."))
  (if (tuplep 2 out-type-spec)
      (b* (((list sym kwd) out-type-spec)
           ((unless (symbolp sym))
            (raise "Invalid array name ~x0." sym)
            (mv (atj-type-irrelevant) nil))
           (array sym)
           ((unless (keywordp kwd))
            (raise "Invalid type keyword ~x0." kwd)
            (mv (atj-type-irrelevant) nil))
           (type (atj-type-from-keyword kwd))
           ((when (and sym (not (atj-type-case type :jprimarr))))
            (raise "Invalid array name ~x0 for non-array type ~x1." array kwd)
            (mv (atj-type-irrelevant) nil))
           (pos (index-of array formals))
           ((when (not pos))
            (raise "Array name ~x0 not among formals ~x1." array formals)
            (mv (atj-type-irrelevant) nil))
           (in-type (nth pos in-types))
           ((unless (equal type in-type))
            (raise "The type ~x0 of the ~x1 input does not match ~
                    the type ~x2 of the ~x1 output."
                   (atj-type-to-keyword in-type) array kwd)
            (mv (atj-type-irrelevant) nil)))
        (mv type array))
    (if (keywordp out-type-spec)
        (mv (atj-type-from-keyword out-type-spec) nil)
      (prog2$ (raise "Invalid output type specification ~x0." out-type-spec)
              (mv (atj-type-irrelevant) nil)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-process-output-type-specs ((out-type-specs true-listp)
                                       (formals symbol-listp)
                                       (in-types atj-type-listp))
  :guard (= (len formals) (len in-types))
  :returns (mv (types atj-type-listp)
               (arrays symbol-listp))
  :short "Lift @(tsee atj-process-output-type-spec) to lists."
  (b* (((when (endp out-type-specs)) (mv nil nil))
       ((mv type array) (atj-process-output-type-spec (car out-type-specs)
                                                      formals
                                                      in-types))
       ((mv types arrays) (atj-process-output-type-specs (cdr out-type-specs)
                                                         formals
                                                         in-types)))
    (mv (cons type types) (cons array arrays))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-main-function-type-input-theorem ((fn symbolp)
                                              (guard pseudo-termp)
                                              (formal symbolp)
                                              (in-type atj-typep)
                                              (wrld plist-worldp))
  :returns (event "A @(tsee acl2::pseudo-event-formp).")
  :mode :program ; because of UNTRANSLATE
  :short "Theorem generated by @(tsee atj-main-function-type)
          for an input of an ACL2 function."
  :long
  (xdoc::topstring
   (xdoc::p
    "The theorem states that, under the guard,
     the specified formal argument satisfies the predicate
     that corresponds to the specified type.")
   (xdoc::p
    "The theorem has no rule classes because its only purpose is
     to make sure that its formula holds.
     The theorem is local (to the @(tsee encapsulate) generated by the macro)
     for the same reason."))
  (b* ((thm-name (packn-pos (list 'atj-
                                  fn
                                  '-input-
                                  formal
                                  '-
                                  (atj-type-to-keyword in-type))
                            (pkg-witness (symbol-package-name fn))))
       (thm-formula (implicate guard
                               `(,(atj-type-to-pred in-type) ,formal))))
    `(local
      (defthm ,thm-name
        ,(untranslate thm-formula t wrld)
        :rule-classes nil))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-main-function-type-input-theorems ((fn symbolp)
                                               (guard pseudo-termp)
                                               (formals symbol-listp)
                                               (in-types atj-type-listp)
                                               (wrld plist-worldp))
  :guard (= (len formals) (len in-types))
  :returns (events "A @(tsee acl2::pseudo-event-form-listp).")
  :mode :program ; because of ATJ-MAIN-FUNCTION-TYPE-INPUT-THEOREM
  :short "Theorems generated by @(tsee atj-main-function-type)
          for all the inputs of an ACL2 function."
  :long
  (xdoc::topstring
   (xdoc::p
    "This lifts @(tsee atj-main-function-type-input-theorem) to lists."))
  (if (endp formals)
      nil
    (cons (atj-main-function-type-input-theorem
           fn guard (car formals) (car in-types) wrld)
          (atj-main-function-type-input-theorems
           fn guard (cdr formals) (cdr in-types) wrld))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-main-function-type-output-theorem ((fn symbolp)
                                               (guard pseudo-termp)
                                               (formals symbol-listp)
                                               (result maybe-natp)
                                               (out-type atj-typep)
                                               (wrld plist-worldp))
  :returns (event "A @(tsee acl2::pseudo-event-formp).")
  :mode :program ; because of UNTRANSLATE
  :short "Theorem generated by @(tsee atj-main-function-type)
          for the/an output of an ACL2 function."
  :long
  (xdoc::topstring
   (xdoc::p
    "The @('result') parameter is @('nil') if @('fn') returns a single result;
     otherwise, it is the (0-based) index of one of the @('fn')'s results.")
   (xdoc::p
    "The theorem states that, under the guard,
     the result of the function (applied to its formals)
     satisfies the predicate that corresponds to the specified type.")
   (xdoc::p
    "The theorem has no rule classes because its only purpose is
     to make sure that its formula holds.
     The theorem is local (to the @(tsee encapsulate) generated by the macro)
     for the same reason."))
  (b* ((thm-name (if result
                     (packn-pos (list 'atj-
                                      fn
                                      '-output-
                                      result
                                      '-
                                      (atj-type-to-keyword out-type))
                                (pkg-witness (symbol-package-name fn)))
                   (packn-pos (list 'atj-
                                    fn
                                    '-output-
                                    (atj-type-to-keyword out-type))
                              (pkg-witness (symbol-package-name fn)))))
       (fn-result (if result
                      `(mv-nth ,result (,fn ,@formals))
                    `(,fn ,@formals)))
       (thm-formula (implicate guard
                               `(,(atj-type-to-pred out-type) ,fn-result))))
    `(local
      (defthm ,thm-name
        ,(untranslate thm-formula t wrld)
        :rule-classes nil))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-main-function-type-output-theorems ((fn symbolp)
                                                (guard pseudo-termp)
                                                (formals symbol-listp)
                                                (nresults natp)
                                                (rev-out-types atj-type-listp)
                                                (wrld plist-worldp))
  :guard (= nresults (len rev-out-types))
  :returns (events "A @(tsee acl2::pseudo-event-form-listp).")
  :mode :program ; because of ATJ-MAIN-FUNCTION-TYPE-OUTPUT-THEOREM
  :short "Theorems generated by @(tsee atj-main-function-type)
          for all the outputs of a multi-valued ACL2 function."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is only used when @('fn') returns multiple values.
     We generate an output type theorem for each,
     counting down from @('nresults - 1') to 0.")
   (xdoc::p
    "The output types are reversed (as suggested by the @('rev-') prefix,
     so that they ``match'' the counting down mentioned just above."))
  (if (zp nresults)
      nil
    (cons (atj-main-function-type-output-theorem
           fn guard formals (1- nresults) (car rev-out-types) wrld)
          (atj-main-function-type-output-theorems
           fn guard formals (1- nresults) (cdr rev-out-types) wrld))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-main-function-type-fn (fn
                                   in-type-specs
                                   out-type-spec/specs
                                   (wrld plist-worldp))
  :returns (event "A @(tsee acl2::maybe-pseudo-event-formp).")
  :mode :program ; because of ATJ-MAIN-FUNCTION-TYPE-INPUT/OUTPUT-THEOREM(S)
  :short "Top-level event generated by @(tsee atj-main-function-type)."
  :long
  (xdoc::topstring
   (xdoc::p
    "This includes the theorems for the function inputs and outputs,
     as well as an event to record the function type in the table.")
   (xdoc::p
    "If the table already includes an entry for the function,
     the proposed function type is compared with the existing one.
     If they are the same, the call is considered redundant
     and no further action is taken.
     If they differ, it is an error."))
  (b* (((unless (symbolp fn))
        (raise "The first input, ~x0, must be a symbol." fn))
       (formals (formals fn wrld)) ; error if not FUNCTION-SYMBOLP
       ((unless (keyword-listp in-type-specs))
        (raise "The second input, ~x0, ~
                must be a true list of ATJ type keywords."
               in-type-specs))
       (in-types ; error id not valid ATJ type keywords:
        (atj-type-list-from-keyword-list in-type-specs))
       ((unless (= (len in-types) (len formals)))
        (raise "The number of input types ~x0 must match ~
                the arity ~x1 of the function ~x2."
               in-type-specs (len formals) fn))
       (nresults (atj-number-of-results fn wrld))
       ((mv out-types arrays)
        (if (= nresults 1)
            (b* (((mv out-type array) (atj-process-output-type-spec
                                       out-type-spec/specs formals in-types)))
              (mv (list out-type) (list array)))
          (if (tuplep nresults out-type-spec/specs)
              (atj-process-output-type-specs
               out-type-spec/specs formals in-types)
            (prog2$ (raise "The third input, ~x0, ~
                            must be a true list of length ~x1 ~
                            of output type specifications."
                           out-type-spec/specs nresults)
                    (mv nil nil)))))
       ((unless (no-duplicatesp-eq (remove-eq nil arrays)))
        (raise "The list of output array names, ~x0, ~
                contains non-NIL duplicates."
               arrays))
       (fn-info? (atj-get-function-type-info-from-table fn wrld))
       ((when fn-info?)
        (b* ((main (atj-function-type-info->main fn-info?)))
          (if (and (equal (atj-function-type->inputs main) in-types)
                   (equal (atj-function-type->outputs main) out-types))
              `(value-triple :redundant)
            (raise "The proposed ATJ main function type [~x0 -> ~x1] for ~x2 ~
                    differs from the already recorded [~x3 -> ~x4]."
                   in-type-specs
                   out-type-spec/specs
                   fn
                   (atj-type-list-to-keyword-list
                    (atj-function-type->inputs main))
                   (atj-type-list-to-keyword-list
                    (atj-function-type->outputs main))))))
       (guard (guard fn nil wrld))
       (input-thms
        (atj-main-function-type-input-theorems fn guard formals in-types wrld))
       (output-thms
        (if (= nresults 1)
            (list (atj-main-function-type-output-theorem
                   fn guard formals nil (car out-types) wrld))
          (atj-main-function-type-output-theorems
           fn guard formals nresults (rev out-types) wrld)))
       (fn-ty (make-atj-function-type :inputs in-types
                                      :outputs out-types
                                      :arrays arrays))
       (fn-info (make-atj-function-type-info :main fn-ty :others nil)))
    `(encapsulate
       ()
       (set-ignore-ok t)
       ,@input-thms
       ,@output-thms
       (table ,*atj-function-type-info-table-name* ',fn ',fn-info))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defsection atj-main-function-type
  :short (xdoc::topstring
          "Macro to prove and record the primary input and output "
          (xdoc::seetopic "atj-types" "types")
          " of an ACL2 function.")
  :long
  (xdoc::topstring
   (xdoc::p
    "This has to be used on the functions of interest
     (i.e. functions for which we want to generate Java code)
     prior to calling ATJ,
     so that ATJ can take advantage of the type information
     recorded for the functions.
     This is only relevant
     when @(':deep') is @('nil') and @(':guards') is @('t');
     in all other cases, the type information is ignored.")
   (xdoc::p
    "For instance, the file @('types-for-natives.lisp') uses this macro
     on the ACL2 functions that are implemented natively in AIJ.")
   (xdoc::p
    "If ATJ encounters a function that is not in the table,
     it assumes the widest possible type (i.e. the one for all ACL2 values)
     for all the inputs and outputs of the function.
     See the code generation functions for details.")
   (xdoc::@def "atj-main-function-type"))
  (defmacro atj-main-function-type (fn in-type-specs out-type-spec/specs)
    `(make-event
      (atj-main-function-type-fn
       ',fn ',in-type-specs ',out-type-spec/specs (w state)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-other-function-type-theorem ((fn symbolp)
                                         (guard pseudo-termp)
                                         (formals symbol-listp)
                                         (in-types atj-type-listp)
                                         (result maybe-natp)
                                         (out-type atj-typep)
                                         (wrld plist-worldp))
  :guard (= (len formals) (len in-types))
  :returns (event "A @(tsee acl2::pseudo-event-formp).")
  :mode :program ; because of UNTRANSLATE
  :short "Theorem generated by @(tsee atj-other-function-type)
          for the/an output of an ACL2 function."
  :long
  (xdoc::topstring
   (xdoc::p
    "The @('result') parameter is @('nil') if @('fn') returns a single result;
     otherwise, it is the (0-based) index of one of the @('fn')'s results.")
   (xdoc::p
    "The theorem states that if the formal parameters
     satisfy both the function's guard
     and the predicates that correspond to the input types,
     then the result of the function (applied to its formal parameters)
     satisfies the predicate that corresponds to the output type.")
   (xdoc::p
    "The theorem has no rule classes because its only purpose is
     to make sure that its formula holds.
     The theorem is local (to the @(tsee encapsulate) generated by the macro)
     for the same reason."))
  (b* ((thm-name (if result
                     (packn-pos (list 'atj-
                                      fn
                                      '-other-type-
                                      result
                                      '-
                                      (atj-type-to-keyword out-type))
                                (pkg-witness (symbol-package-name fn)))
                   (packn-pos (list 'atj-
                                    fn
                                    '-other-type)
                              (pkg-witness (symbol-package-name fn)))))
       (type-hyps (atj-other-function-type-theorem-aux formals in-types))
       (fn-result (if result
                      `(mv-nth ,result (,fn ,@formals))
                    `(,fn ,@formals)))
       (concl `(,(atj-type-to-pred out-type) ,fn-result))
       (thm-formula (implicate (conjoin (cons guard type-hyps)) concl)))
    `(local
      (defthm ,thm-name
        ,(untranslate thm-formula t wrld)
        :rule-classes nil)))

  :prepwork
  ((define atj-other-function-type-theorem-aux ((formals symbol-listp)
                                                (in-types atj-type-listp))
     :guard (= (len formals) (len in-types))
     :returns (terms pseudo-term-listp
                     :hyp :guard
                     :hints (("Goal" :in-theory (enable atj-type-to-pred))))
     :parents nil
     (cond ((endp formals) nil)
           (t (cons `(,(atj-type-to-pred (car in-types)) ,(car formals))
                    (atj-other-function-type-theorem-aux (cdr formals)
                                                         (cdr in-types))))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-other-function-type-theorems ((fn symbolp)
                                          (guard pseudo-termp)
                                          (formals symbol-listp)
                                          (in-types atj-type-listp)
                                          (nresults natp)
                                          (out-types atj-type-listp)
                                          (wrld plist-worldp))
  :guard (= nresults (len out-types))
  :returns (events "A @(tsee acl2::pseudo-event-form-listp).")
  :mode :program ; because of ATJ-OTHER-FUNCTION-TYPE-THEOREM
  :short "Theorems generated by @(tsee atj-other-function-type)
          for all the outputs of a multi-valued ACL2 function."
  :long
  (xdoc::topstring
   (xdoc::p
    "This is only used when @('fn') returns multiple values.
     We generate an output type theorem for each,
     counting down from @('nresults - 1') to 0."))
  (if (zp nresults)
      nil
    (cons (atj-other-function-type-theorem
           fn guard formals in-types (1- nresults) (car out-types) wrld)
          (atj-other-function-type-theorems
           fn guard formals in-types (1- nresults) (cdr out-types) wrld))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-check-other-function-type ((new-in-types atj-type-listp)
                                       (old-fn-types atj-function-type-listp)
                                       (all-in-jtypess atj-jitype-list-listp))
  :returns (yes/no booleanp)
  :short "Check the new input types
          passed to @(tsee atj-other-function-type)
          against the existing secondary function types."
  :long
  (xdoc::topstring
   (xdoc::p
    "The primary and secondary input types attached to a function
     are used to generate overloaded methods for the function.
     It must be possible, at compile time, to always resolve the method,
     based on the most specific argument types.
     This should be guaranteed if the set of all the function's input types
     (more precisely, the set of all the input type tuples of the function)
     is closed under greatest lower bounds that do not include @('nil')s,
     as explained below.")
   (xdoc::p
    "For example, consider a binary function @('f')
     with two function types whose inputs are
     @('(:arational :ainteger)') and @('(:ainteger :arational)').
     These will give rise to two overloaded methods for @('f'),
     one with argument types @('Acl2Rational') and @('Acl2Integer'),
     and one with argument types @('Acl2Integer') and @('Acl2Rational').
     Consider a method call whose actual argument types
     are @('Acl2Integer') and @('Acl2Integer'):
     if only those two overloaded methods are available,
     then there is no most specific method that is based just on the types.
     However, if a third overloaded method were available
     with @('Acl2Integer') and @('Acl2Integer') as argument types,
     that would be the most specific method to call.
     This is the case if the table includes a function type for @('f')
     with input types @('(:ainteger :ainteger)').")
   (xdoc::p
    "Generalizing from this example,
     we want the set of all function input types
     to be closed under greatest lower bounds
     (w.r.t. the Java type partial order).
     This way, if some tuple of actual arguments
     fits two different overloaded methods,
     it will also fit the method corresponding to the greatest lower bound;
     therefore, there will be always a ``minimum'' method
     that will be selected at compile time and called at run time.
     However, recall that @(tsee atj-jitype-meet) may produce @('nil'):
     if the greatest lower bound contains a @('nil') component,
     the closure requirement does not apply,
     because it means that some types are incompatible
     and therefore there is no confusion about most specific types.")
   (xdoc::p
    "Besides the closure property just explained,
     we also ensure that the proposed new input types
     differ from all the existing secondary input types.
     We maintain the uniqueness, for each function,
     of the (primary and secondary) input types in the table,
     so that there is exactly one overloaded method for each input type tuple.
     Redundant calls of @(tsee atj-other-function-type)
     are handled before calling this function.")
   (xdoc::p
    "The @('new-in-types') parameter of this function
     consist of the new proposed input types.
     The @('old-fn-types') parameter
     consists of all the existing secondary input types already in the table,
     which we @(tsee cdr) through and compare against @('new-in-types').
     The @('all-in-jtypess') parameter of this function
     consists of the Java type counterparts of
     all the tuples of existing secondary input types in the table,
     plus @('new-in-types');
     this stays constant, we do not @(tsee cdr) through it
     because we need the whole collection to check the closure property.
     We include (the Java type counterpart of) @('new-in-types')
     because the new proposed input types contribute to the closure properties:
     they will be in the table if all the checks succeed;
     for instance, if only @(':arational') is in the table
     and we are trying to add @(':ainteger'),
     their greatest lower bound is @(':ainteger'),
     which will be in the table.
     The primary function type is not included
     in @('old-fn-types') or @('all-in-jtypess'),
     because the primary input types are always checked
     to be strictly wider than @('new-in-types'),
     in @(tsee atj-other-function-type-fn)."))
  (b* (((when (endp old-fn-types)) t)
       (old-fn-type (car old-fn-types))
       (old-in-types (atj-function-type->inputs old-fn-type))
       ((when (equal new-in-types old-in-types))
        (raise "The proposed input types ~x0 must differ from ~
                the existing secondary input types ~
                for the function, ~
                but they are equal to some of these existing types."
               new-in-types))
       (old-in-jtypes (atj-type-list-to-jitype-list old-in-types))
       (new-in-jtypes (atj-type-list-to-jitype-list new-in-types))
       ((unless (= (len old-in-jtypes) (len new-in-jtypes)))
        (raise "Internal error: ~
                the number of proposed input types ~x0 differs from ~
                the number of existing input types ~x1."
               new-in-types old-in-types))
       (glb (atj-jitype-list-meet old-in-jtypes new-in-jtypes))
       ((unless (or (member-eq nil glb)
                    (member-equal glb all-in-jtypess)))
        (raise "The Java counterparts ~x0 of the proposed input types ~x1 ~
                and the Java counterparts ~x2 of ~
                some existing secondary input types ~x3, ~
                have a greatest lower bound ~x4 ~
                that is not the Java counterpart ~
                of any types currently in the table ~
                or of the proposed input types. ~
                This may cause ambiguities in resolving method overloading. ~
                Consider adding types whose Java counterparts are ~x4 first, ~
                and then the proposed input types ~x1."
               new-in-jtypes new-in-types
               old-in-jtypes old-in-types
               glb)))
    (atj-check-other-function-type new-in-types
                                   (cdr old-fn-types)
                                   all-in-jtypess)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-other-function-type-fn (fn
                                    in-type-specs
                                    out-type-spec/specs
                                    (wrld plist-worldp))
  :returns (event "A @(tsee acl2::maybe-pseudo-event-formp).")
  :mode :program ; because of ATJ-OTHER-FUNCTION-TYPE-THEOREM
  :short "Top-level event generated by @(tsee atj-other-function-type)."
  :long
  (xdoc::topstring
   (xdoc::p
    "This includes the theorem(s) stating that
     the guard and input types imply the output type(s),
     as well as an event to record the function type in the table.")
   (xdoc::p
    "It is an error if the function does not have a primary type already;
     that is, @(tsee atj-main-function-type) must be called
     before calling @(tsee atj-other-function-type).")
   (xdoc::p
    "The proposed input types must be narrower than the primary input types;
     otherwise, there would be no advantage
     in adding these secondary input types,
     and in generating overloaded method corresponding to these types.
     The proposed output types must be narrower than or the same as
     the primary output types:
     since the primary output types are proved under the guard assumption only,
     while the secondary output types are proved
     with additional type hypotheses,
     it does not make sense that the secondary output types
     is wider than, or unrelated to, the primary output types;
     this situation probably signals the misstatement of some types
     to either @(tsee atj-main-function-type)
     or @(tsee atj-other-function-type).")
   (xdoc::p
    "If the proposed function type is already in the table,
     the call of @(tsee atj-other-function-type) is considered redundant
     and no further action is taken.")
   (xdoc::p
    "We also ensure that the addition of the proposed types
     does not cause method overloading ambiguities:
     see @(tsee atj-check-other-function-type) for details.")
   (xdoc::p
    "We may add additional sanity checks in the future,
     e.g. that if the new input types are narrower than or equal to
     some already existing secondary types,
     then the output types must satisfy that relation too.
     The reason is analogous to the one discussed above
     to motivate the check against the primary output types;
     but here we are talking about the secondary output types."))
  (b* (((unless (symbolp fn))
        (raise "The first input, ~x0, must be a symbol." fn))
       (formals (formals fn wrld)) ; error if not FUNCTION-SYMBOLP
       ((unless (keyword-listp in-type-specs))
        (raise "The second input, ~x0, ~
                must be a true list of ATJ type keywords."
               in-type-specs))
       (in-types ; error if not valid ATJ type keywords:
        (atj-type-list-from-keyword-list in-type-specs))
       ((unless (= (len in-types) (len formals)))
        (raise "The number of input types ~x0 must match ~
                the arity ~x1 of the function ~x2."
               in-type-specs (len formals) fn))
       (nresults (atj-number-of-results fn wrld))
       ((mv out-types arrays)
        (if (= nresults 1)
            (b* (((mv out-type array) (atj-process-output-type-spec
                                       out-type-spec/specs formals in-types)))
              (mv (list out-type) (list array)))
          (if (tuplep nresults out-type-spec/specs)
              (atj-process-output-type-specs
               out-type-spec/specs formals in-types)
            (prog2$ (raise "The third input, ~x0, ~
                            must be a true list of length ~x1 ~
                            of output type specifications."
                           out-type-spec/specs nresults)
                    (mv nil nil)))))
       ((unless (no-duplicatesp-eq (remove-eq nil arrays)))
        (raise "The list of output array names, ~x0, ~
                contains non-NIL duplicates."
               arrays))
       (fn-info? (atj-get-function-type-info-from-table fn wrld))
       ((unless fn-info?)
        (raise "The function ~x0 does not have a primary function type yet. ~
                Use ATJ-MAIN-FUNCTION-TYPE to define it, ~
                and then try again this ATJ-OTHER-FUNCTION-TYPE."
               fn))
       (main-fn-type (atj-function-type-info->main fn-info?))
       (main-in-types (atj-function-type->inputs main-fn-type))
       (main-out-types (atj-function-type->outputs main-fn-type))
       ((unless (atj-type-list-< in-types main-in-types))
        (raise "The proposed inputs types ~x0 must be strictly narrower ~
                than the primary input types ~x1."
               in-type-specs
               (atj-type-list-to-keyword-list main-in-types)))
       ((unless (atj-type-list-<= out-types main-out-types))
        (raise "The proposed output types ~x0 must be ~
                narrower than or equal to, ~
                the primary output types ~x1."
               out-type-spec/specs
               (atj-type-list-to-keyword-list main-out-types)))
       (other-fn-types (atj-function-type-info->others fn-info?))
       (new-fn-type (make-atj-function-type :inputs in-types
                                            :outputs out-types
                                            :arrays (repeat nresults nil)))
       ((when (member-equal new-fn-type other-fn-types))
        `(value-triple :redundant))
       (other-in-types (atj-function-type-list->inputs other-fn-types))
       (all-in-jtypess (atj-type-list-list-to-jitype-list-list
                        (cons in-types other-in-types)))
       (- (atj-check-other-function-type in-types
                                         other-fn-types
                                         all-in-jtypess))
       (guard (guard fn nil wrld))
       (thms
        (if (= nresults 1)
            (list (atj-other-function-type-theorem
                   fn guard formals in-types nil (car out-types) wrld))
          (atj-other-function-type-theorems
           fn guard formals in-types nresults out-types wrld)))
       (new-fn-info (change-atj-function-type-info
                     fn-info? :others (cons new-fn-type other-fn-types))))
    `(encapsulate
       ()
       (set-ignore-ok t)
       ,@thms
       (table ,*atj-function-type-info-table-name* ',fn ',new-fn-info))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defsection atj-other-function-type
  :short (xdoc::topstring
          "Macro to prove and record secondary input and output "
          (xdoc::seetopic "atj-types" "types")
          " of an ACL2 function.")
  :long
  (xdoc::topstring
   (xdoc::p
    "This has to be used on the functions of interest
     (i.e. functions for which we want to generate Java code)
     prior to calling ATJ,
     so that ATJ can take advantage of the type information
     recorded for the functions.
     This is only relevant
     when @(':deep') is @('nil') and @(':guards') is @('t');
     in all other cases, the type information is ignored.")
   (xdoc::p
    "Each of the successful calls of this macro
     will result in an overloaded method with the specified types."))
  (defmacro atj-other-function-type (fn in-types out-type/types)
    `(make-event
      (atj-other-function-type-fn
       ',fn ',in-types ',out-type/types (w state)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-function-type-of-min-input-types
  ((in-types atj-type-listp)
   (fn-types atj-function-type-listp))
  :returns (fn-type? atj-maybe-function-type-p)
  :short "Function type with the minimum input types."
  :long
  (xdoc::topstring
   (xdoc::p
    "When this function is called,
     @('in-types') are the types inferred for
     the actual arguments of a function call,
     and @('fn-types') are all the function types (primary and secondary)
     of the called function.
     The goal here is to see if the argument types match any function type,
     in the sense that the Java counterparts
     of the input types of the function type
     are greater than or equal to the Java counterparts
     of the types of the actual arguments.
     If no such function type is found, we return @('nil').
     If instead some exist, we select the one with the  minimum input types,
     which always exists because of the closure property
     enforced by @(tsee atj-other-function-type),
     and we return that function type.
     In other words, given the types of the actual arguments,
     the output types of the returned function type (if any)
     tell us the result type of the overloaded method
     that will be resolved at compile time."))
  (atj-function-type-of-min-input-types-aux
   (atj-type-list-to-jitype-list in-types)
   (atj-function-type-list-fix fn-types)
   nil
   nil)

  :prepwork
  ((define atj-function-type-of-min-input-types-aux
     ((in-jtypes atj-jitype-listp)
      (fn-types atj-function-type-listp)
      (current-min-in-jtypes atj-jitype-listp)
      (current-fn-type? atj-maybe-function-type-p))
     :returns (fn-type? atj-maybe-function-type-p
                        :hyp (and (atj-function-type-listp fn-types)
                                  (atj-maybe-function-type-p current-fn-type?)))
     (b* (((when (endp fn-types)) current-fn-type?)
          (fn-type (car fn-types))
          (fn-in-types (atj-function-type->inputs fn-type))
          (fn-in-jtypes (atj-type-list-to-jitype-list fn-in-types))
          ((mv current-min-in-jtypes current-fn-type?)
           (if (and (atj-jitype-list-<= in-jtypes fn-in-jtypes)
                    (or (null current-fn-type?) ; i.e. none found yet
                        (atj-jitype-list-< fn-in-jtypes
                                           current-min-in-jtypes)))
               (mv fn-in-jtypes fn-type)
             (mv current-min-in-jtypes current-fn-type?))))
       (atj-function-type-of-min-input-types-aux in-jtypes
                                                 (cdr fn-types)
                                                 current-min-in-jtypes
                                                 current-fn-type?)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-to-type-list ((type atj-typep))
  :returns (types atj-type-listp)
  :short "Turn a single ATJ type into a singleton list of it."
  :long
  (xdoc::topstring-p
   "This is just @(tsee list),
    but we introduce an explicit function for greater clarity.")
  (list (atj-type-fix type))
  :hooks (:fix)
  ///

  (more-returns
   (types consp :rule-classes :type-prescription)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-list-to-type-list-list ((types atj-type-listp))
  :returns (typess atj-type-list-listp)
  :short "Lift @(tsee atj-type-to-type-list) to lists."
  (cond ((endp types) nil)
        (t (cons (atj-type-to-type-list (car types))
                 (atj-type-list-to-type-list-list (cdr types)))))
  :hooks (:fix)
  ///

  (more-returns
   (typess cons-listp))

  (defret len-of-atj-type-list-to-type-list-list
    (equal (len typess)
           (len types))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define atj-type-list-to-type ((types atj-type-listp))
  :guard (consp types)
  :returns (type atj-typep)
  :short "Ensure that a non-empty list of types is a singleton,
          and return its only element."
  :long
  (xdoc::topstring
   (xdoc::p
    "In some cases, a non-empty list of types is expected to be a singleton.
     For instance, the type list may be the output types of a function
     that is known to return a single result.
     This utility can be used in these cases,
     to check the expectation for robustness,
     and to retrieve the single type from the singleton list."))
  (if (= (len types) 1)
      (atj-type-fix (car types))
    (prog2$
     (raise "Internal error: ~x0 is not a singleton list of types." types)
     (atj-type-irrelevant)))
  :hooks (:fix))
